# 🎯 Baden GET Endpoint Guide

## 📋 Overview

Endpoint Baden mendukung metode GET untuk menampilkan data dari tabel `orders`. Endpoint ini menggunakan Basic Authentication dan dapat menampilkan data dengan berbagai parameter query.

## 🔗 Endpoint Information

- **URL:** `https://copilotproduction.site/api/in/baden`
- **Method:** GET
- **Authentication:** Basic Auth
- **Target Table:** `orders`

## 🔐 Authentication

```bash
Username: baden
Password: baden
Authorization Header: Basic YmFkZW46YmFkZW4=
```

## 📊 Response Format

### Success Response (200 OK)

```json
{
  "success": true,
  "message": "Data retrieved successfully",
  "received_at": "2025-08-08T02:57:15.314077Z",
  "data": {
    "receiver_info": {
      "id": 1,
      "name": "Baden Basic Auth Receiver",
      "endpoint": "baden",
      "description": "Receiver for Baden System with Basic Authentication"
    },
    "request_info": {
      "method": "GET",
      "received_at": "2025-08-08T02:57:15.314077Z",
      "ip_address": "127.0.0.1",
      "user_agent": "Mozilla/5.0..."
    },
    "parameters": {},
    "pagination": {
      "limit": 10,
      "offset": 0,
      "sort": "id",
      "order": "desc"
    },
    "data": {
      "total_records": 7,
      "returned_records": 7,
      "records": [
        {
          "id": 1,
          "order_id": "ORD-001",
          "customer_name": "Ahmad Dhani",
          "total_amount": "150000.00",
          "items": "[{\"name\":\"Laptop Asus\",\"qty\":1,\"price\":150000.00}]",
          "status": "completed",
          "processed": 1,
          "processed_at": "2025-08-06T02:57:15.000000Z",
          "created_at": "2025-08-05T02:57:15.000000Z",
          "updated_at": "2025-08-06T02:57:15.000000Z"
        },
        {
          "id": 2,
          "order_id": "ORD-002",
          "customer_name": "Siti Nurhaliza",
          "total_amount": "250000.00",
          "items": "[{\"name\":\"Smartphone Samsung\",\"qty\":1,\"price\":250000.00}]",
          "status": "pending",
          "processed": 0,
          "processed_at": null,
          "created_at": "2025-08-07T02:57:15.000000Z",
          "updated_at": "2025-08-07T02:57:15.000000Z"
        }
      ]
    }
  }
}
```

### Error Response (401 Unauthorized)

```json
{
  "success": false,
  "error": "Authentication failed",
  "message": "Invalid credentials",
  "code": 401,
  "received_at": "2025-08-08T02:57:15.314077Z"
}
```

## 🔧 Query Parameters

| Parameter | Type | Default | Description |
|-----------|------|---------|-------------|
| `limit` | integer | 10 | Jumlah record yang dikembalikan (max: 1000) |
| `offset` | integer | 0 | Offset untuk pagination |
| `sort` | string | "id" | Field untuk sorting |
| `order` | string | "desc" | Urutan sorting: "asc" atau "desc" |
| `status` | string | - | Filter berdasarkan status order |
| `customer_name` | string | - | Filter berdasarkan nama customer |
| `order_id` | string | - | Filter berdasarkan ID order |
| `processed` | boolean | - | Filter berdasarkan status processed |

## 📝 Usage Examples

### 1. Basic GET Request

```bash
curl -X GET "https://copilotproduction.site/api/in/baden" \
  -H "Authorization: Basic YmFkZW46YmFkZW4=" \
  -H "Content-Type: application/json"
```

### 2. GET Request with Pagination

```bash
curl -X GET "https://copilotproduction.site/api/in/baden?limit=5&offset=0" \
  -H "Authorization: Basic YmFkZW46YmFkZW4=" \
  -H "Content-Type: application/json"
```

### 3. GET Request with Sorting

```bash
curl -X GET "https://copilotproduction.site/api/in/baden?sort=created_at&order=desc" \
  -H "Authorization: Basic YmFkZW46YmFkZW4=" \
  -H "Content-Type: application/json"
```

### 4. GET Request with Filters

```bash
# Filter by status
curl -X GET "https://copilotproduction.site/api/in/baden?status=completed" \
  -H "Authorization: Basic YmFkZW46YmFkZW4=" \
  -H "Content-Type: application/json"

# Filter by customer name
curl -X GET "https://copilotproduction.site/api/in/baden?customer_name=Ahmad" \
  -H "Authorization: Basic YmFkZW46YmFkZW4=" \
  -H "Content-Type: application/json"

# Filter by order ID
curl -X GET "https://copilotproduction.site/api/in/baden?order_id=ORD-001" \
  -H "Authorization: Basic YmFkZW46YmFkZW4=" \
  -H "Content-Type: application/json"
```

### 5. GET Request with Multiple Parameters

```bash
curl -X GET "https://copilotproduction.site/api/in/baden?limit=3&offset=0&sort=total_amount&order=desc&status=completed" \
  -H "Authorization: Basic YmFkZW46YmFkZW4=" \
  -H "Content-Type: application/json"
```

## 🌐 JavaScript/Fetch Example

```javascript
async function getBadenData() {
    try {
        const response = await fetch('https://copilotproduction.site/api/in/baden', {
            method: 'GET',
            headers: {
                'Authorization': 'Basic YmFkZW46YmFkZW4=',
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            }
        });

        const data = await response.json();
        
        if (response.ok) {
            console.log('Data retrieved successfully:', data);
            return data;
        } else {
            console.error('Error:', data);
            throw new Error(data.message || 'Request failed');
        }
    } catch (error) {
        console.error('Fetch error:', error);
        throw error;
    }
}

// Usage with parameters
async function getBadenDataWithParams(params = {}) {
    const queryString = new URLSearchParams(params).toString();
    const url = `https://copilotproduction.site/api/in/baden?${queryString}`;
    
    try {
        const response = await fetch(url, {
            method: 'GET',
            headers: {
                'Authorization': 'Basic YmFkZW46YmFkZW4=',
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            }
        });

        const data = await response.json();
        
        if (response.ok) {
            console.log('Data retrieved successfully:', data);
            return data;
        } else {
            console.error('Error:', data);
            throw new Error(data.message || 'Request failed');
        }
    } catch (error) {
        console.error('Fetch error:', error);
        throw error;
    }
}

// Example usage
getBadenDataWithParams({
    limit: 5,
    offset: 0,
    sort: 'created_at',
    order: 'desc',
    status: 'completed'
});
```

## 📊 Sample Data Structure

Tabel `orders` memiliki struktur sebagai berikut:

| Column | Type | Description |
|--------|------|-------------|
| `id` | bigint | Primary key |
| `order_id` | varchar | Unique order identifier |
| `customer_name` | varchar | Customer name |
| `total_amount` | decimal(10,2) | Total order amount |
| `items` | json | Order items (JSON array) |
| `status` | varchar | Order status (pending, processing, completed, cancelled) |
| `processed` | boolean | Whether order is processed |
| `processed_at` | timestamp | Processing timestamp |
| `created_at` | timestamp | Creation timestamp |
| `updated_at` | timestamp | Last update timestamp |

## 🧪 Testing

### 1. Test Page
Visit: `https://copilotproduction.site/test-baden-get.html`

### 2. Create Sample Data
Visit: `https://copilotproduction.site/create-baden-sample-data.php`

### 3. Direct API Test
Visit: `https://copilotproduction.site/api/in/baden` (with Basic Auth)

## ⚠️ Error Handling

### Common Error Codes

- **401 Unauthorized**: Invalid credentials
- **404 Not Found**: Endpoint not found
- **429 Too Many Requests**: Rate limit exceeded
- **500 Internal Server Error**: Server error

### Error Response Format

```json
{
  "success": false,
  "error": "Error type",
  "message": "Detailed error message",
  "code": 401,
  "received_at": "2025-08-08T02:57:15.314077Z"
}
```

## 🔒 Security Notes

1. **Basic Authentication**: Username dan password dikirim dalam header Authorization
2. **Rate Limiting**: Endpoint memiliki rate limiting untuk mencegah abuse
3. **Input Validation**: Semua parameter query divalidasi
4. **SQL Injection Protection**: Menggunakan prepared statements

## 📈 Performance Tips

1. **Use Pagination**: Gunakan `limit` dan `offset` untuk data besar
2. **Filter Data**: Gunakan filter untuk mengurangi jumlah data
3. **Optimize Sorting**: Pilih field yang sudah di-index untuk sorting
4. **Cache Results**: Implement caching di client side jika diperlukan

## 🔗 Related Links

- [Test Baden GET Endpoint](./public/test-baden-get.html)
- [Create Sample Data](./public/create-baden-sample-data.php)
- [API Documentation](./API_TARGET_DOCUMENTATION.md)
- [Basic Auth Fix Guide](./BASIC_AUTH_FIX_GUIDE.md)
