# GET Request dengan Body Data - Penyimpanan Otomatis

## ✅ **Fitur Baru: GET dengan Body Data**

Sekarang GET request juga dapat menerima body data (raw JSON) dan menyimpannya ke database seperti POST request.

## 🔄 **Cara Kerja**

### **1. GET Request dengan Body Data**
```bash
curl -X GET "https://your-domain.com/api/in/ujicoba" \
  -H "Content-Type: application/json" \
  -d '{
    "id": 123,
    "nama": "test user",
    "alamat": "test address",
    "gender": "male",
    "data": {
      "key": "value",
      "nested": {
        "data": "example"
      }
    }
  }'
```

### **2. Response (Sama dengan POST)**
```json
{
    "success": true,
    "message": "Data received successfully",
    "received_at": "2024-01-20T10:30:00.000000Z"
}
```

### **3. GET Request tanpa Body Data (Normal)**
```bash
curl -X GET "https://your-domain.com/api/in/ujicoba"
```

**Response:**
```json
{
    "success": true,
    "message": "Data retrieved successfully",
    "received_at": "2024-01-20T10:30:00.000000Z"
}
```

## 🔧 **Logika yang Diterapkan**

### **Deteksi Body Data**
```php
$bodyData = $request->getContent();
$hasBodyData = !empty($bodyData);

if ($hasBodyData) {
    // Parse JSON dan simpan ke database
    $data = json_decode($bodyData, true);
    // ... proses penyimpanan seperti POST
} else {
    // GET request normal tanpa body
    // ... proses GET biasa
}
```

### **Penyimpanan Data**
- ✅ **Auto-create table** jika belum ada
- ✅ **Field mapping** (`id` → `external_id`)
- ✅ **JSON handling** untuk nested data
- ✅ **Validation** JSON format
- ✅ **Error handling** lengkap

## 📊 **Perbedaan Behavior**

| Request Type | Body Data | Behavior |
|-------------|-----------|----------|
| GET | ❌ Tidak ada | Normal GET (retrieve data) |
| GET | ✅ Ada | Simpan data ke database |
| POST | ✅ Ada | Simpan data ke database |

## 🚀 **Test Cases**

### **Test 1: GET dengan Body Data**
```bash
curl -X GET "https://your-domain.com/api/in/ujicoba" \
  -H "Content-Type: application/json" \
  -d '{"id": 999, "nama": "GET Test", "alamat": "Test Address"}'
```

**Expected Response:**
```json
{
    "success": true,
    "message": "Data received successfully",
    "received_at": "2024-01-20T10:30:00.000000Z"
}
```

### **Test 2: GET tanpa Body Data**
```bash
curl -X GET "https://your-domain.com/api/in/ujicoba"
```

**Expected Response:**
```json
{
    "success": true,
    "message": "Data retrieved successfully",
    "received_at": "2024-01-20T10:30:00.000000Z"
}
```

### **Test 3: GET dengan Invalid JSON**
```bash
curl -X GET "https://your-domain.com/api/in/ujicoba" \
  -H "Content-Type: application/json" \
  -d '{invalid json}'
```

**Expected Response:**
```json
{
    "success": false,
    "error": "Invalid JSON in request body",
    "code": 400,
    "received_at": "2024-01-20T10:30:00.000000Z"
}
```

## 📋 **Keuntungan Fitur Ini**

1. **✅ Fleksibilitas**: GET request bisa menyimpan data
2. **✅ Konsistensi**: Response format sama dengan POST
3. **✅ Backward Compatibility**: GET tanpa body tetap berfungsi normal
4. **✅ Auto-save**: Data langsung tersimpan ke database
5. **✅ Error Handling**: Validasi JSON dan error handling lengkap

## 🔍 **Logging**

Semua aktivitas GET dengan body data akan dicatat:

```php
Log::info('GET request with body data detected', [
    'receiver_id' => $receiver->id,
    'body_size' => strlen($bodyData),
    'content_type' => $request->header('Content-Type')
]);

Log::info('GET body data saved to table', [
    'table' => $receiver->target_table,
    'records_inserted' => count($insertData)
]);
```

## 📊 **Monitoring**

Setelah implementasi:

- ✅ **GET dengan body**: Data tersimpan ke database
- ✅ **GET tanpa body**: Normal GET behavior
- ✅ **Response konsisten**: Format sama dengan POST
- ✅ **Error handling**: Validasi dan error response
- ✅ **Logging**: Tracking lengkap untuk debugging

Sekarang GET request dapat menyimpan data seperti POST! 🎉 