# GET Request dengan Body Data Raw

## 📋 Overview

Sistem API8-Fix mendukung **GET request dengan body data raw JSON**, seperti contoh yang Anda minta: `{"data": "customer"}`.

## 🎯 Contoh Request

### **1. GET dengan Body Data Sederhana**

```bash
curl -X GET 'http://localhost/api8-fix/public/api/in/test12' \
  -H 'Content-Type: application/json' \
  -d '{"data": "customer"}'
```

**Response yang diharapkan:**
```json
{
    "success": true,
    "message": "Data received successfully",
    "received_at": "2024-01-15T10:30:45.123456Z",
    "data": {
        "data": "customer"
    }
}
```

### **2. GET dengan Body Data Kompleks**

```bash
curl -X GET 'http://localhost/api8-fix/public/api/in/test12' \
  -H 'Content-Type: application/json' \
  -d '{
    "data": "customer",
    "id": 123,
    "name": "John Doe",
    "email": "john@example.com"
  }'
```

**Response yang diharapkan:**
```json
{
    "success": true,
    "message": "Data received successfully",
    "received_at": "2024-01-15T10:30:45.123456Z",
    "data": {
        "data": "customer",
        "id": 123,
        "name": "John Doe",
        "email": "john@example.com"
    }
}
```

### **3. GET dengan Body Data Array**

```bash
curl -X GET 'http://localhost/api8-fix/public/api/in/test12' \
  -H 'Content-Type: application/json' \
  -d '{
    "data": ["customer1", "customer2"],
    "type": "array_test"
  }'
```

### **4. GET dengan Body Data + Query Parameters**

```bash
curl -X GET 'http://localhost/api8-fix/public/api/in/test12?debug_storage=true&format=json' \
  -H 'Content-Type: application/json' \
  -d '{"data": "customer"}'
```

**Response dengan Debug Info:**
```json
{
    "success": true,
    "message": "Data received successfully",
    "received_at": "2024-01-15T10:30:45.123456Z",
    "data": {
        "data": "customer"
    },
    "saved_to_table": true
}
```

## 🔧 Cara Kerja Sistem

### **1. Deteksi Body Data**
Sistem akan mendeteksi apakah GET request memiliki body data:
```php
$bodyData = $request->getContent();
$hasBodyData = !empty($bodyData);
```

### **2. Parsing JSON**
Body data akan di-parse sebagai JSON:
```php
$data = json_decode($bodyData, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    return $this->errorResponse('Invalid JSON in request body', 400, $request, $receiver);
}
```

### **3. Proses seperti POST**
Jika ada body data, sistem akan memproses seperti POST request:
- ✅ Validasi JSON
- ✅ Simpan ke database (jika target_table dikonfigurasi)
- ✅ Log request
- ✅ Return response dengan data yang dikirim

### **4. Fallback ke GET Normal**
Jika tidak ada body data, sistem akan menjalankan GET request normal:
- ✅ Query parameters validation
- ✅ Fetch data dari database
- ✅ Return formatted response

## 📊 Response Format

### **Success Response:**
```json
{
    "success": true,
    "message": "Data received successfully",
    "received_at": "2024-01-15T10:30:45.123456Z",
    "data": {
        // Data yang dikirim dalam request body
    }
}
```

### **Error Response:**
```json
{
    "success": false,
    "error": "Invalid JSON in request body",
    "code": 400,
    "received_at": "2024-01-15T10:30:45.123456Z"
}
```

## 🎯 Use Cases

### **1. Data Submission via GET**
```bash
# Submit customer data via GET
curl -X GET 'http://localhost/api8-fix/public/api/in/customers' \
  -H 'Content-Type: application/json' \
  -d '{"data": "customer", "name": "John", "email": "john@example.com"}'
```

### **2. Configuration Update**
```bash
# Update configuration via GET
curl -X GET 'http://localhost/api8-fix/public/api/in/config' \
  -H 'Content-Type: application/json' \
  -d '{"data": "config", "setting": "enabled", "value": true}'
```

### **3. Batch Data Processing**
```bash
# Process multiple records via GET
curl -X GET 'http://localhost/api8-fix/public/api/in/batch' \
  -H 'Content-Type: application/json' \
  -d '{
    "data": "batch",
    "records": [
      {"id": 1, "name": "User 1"},
      {"id": 2, "name": "User 2"}
    ]
  }'
```

## ⚠️ Important Notes

### **1. Content-Type Header**
Pastikan selalu menggunakan header:
```
Content-Type: application/json
```

### **2. JSON Format**
Body data harus dalam format JSON yang valid:
```json
{"data": "customer"}
```

### **3. Database Storage**
Jika receiver memiliki `target_table` yang dikonfigurasi, data akan otomatis disimpan ke database.

### **4. Auto-Create Table**
Jika `auto_create_table` diaktifkan, tabel akan dibuat otomatis jika belum ada.

### **5. Debug Mode**
Gunakan query parameter `debug_storage=true` untuk melihat status penyimpanan database.

## 🚀 Testing

Jalankan file test yang telah dibuat:
```bash
php test_get_with_body.php
```

Atau gunakan Postman/Insomnia dengan konfigurasi:
- **Method**: GET
- **URL**: `http://localhost/api8-fix/public/api/in/test12`
- **Headers**: `Content-Type: application/json`
- **Body**: Raw JSON `{"data": "customer"}`
