# GET Request dengan Response Format Sederhana

## 📋 Overview

Fitur ini memungkinkan Anda melakukan **GET request dengan body data** untuk mengambil data dari tabel database dengan response format yang sederhana dan mudah dipahami.

## 🎯 Format Request

### **Method:** GET
### **URL:** `/api/in/{endpoint}`
### **Headers:**
```
Content-Type: application/json
Accept: application/json
```

### **Body JSON:**
```json
{
    "data": "nama_tabel"
}
```

## 📊 Format Response

### **Response Sukses:**
```json
{
    "sukses": true,
    "datas": [
        {
            "nama": "sulaiman",
            "Alamat": "Bogor",
            "Gender": "Male"
        },
        {
            "nama": "Daud",
            "Alamat": "Jakarta",
            "Gender": "Male"
        }
    ]
}
```

### **Response Error:**
```json
{
    "sukses": false,
    "error": "Table 'table_not_exists' does not exist"
}
```

## 📝 Contoh Penggunaan

### **1. Ambil Data Customer**

**Request:**
```bash
curl -X GET "https://your-domain.com/api/in/test-receiver" \
  -H "Content-Type: application/json" \
  -d '{
    "data": "customer"
  }'
```

**Response:**
```json
{
    "sukses": true,
    "datas": [
        {
            "nama": "sulaiman",
            "Alamat": "Bogor",
            "Gender": "Male"
        },
        {
            "nama": "Daud",
            "Alamat": "Jakarta",
            "Gender": "Male"
        },
        {
            "nama": "Sarah",
            "Alamat": "Bandung",
            "Gender": "Female"
        }
    ]
}
```

### **2. Ambil Data dengan Filter**

**Request:**
```bash
curl -X GET "https://your-domain.com/api/in/test-receiver" \
  -H "Content-Type: application/json" \
  -d '{
    "data": "customer",
    "Gender": "Male"
  }'
```

**Response:**
```json
{
    "sukses": true,
    "datas": [
        {
            "nama": "sulaiman",
            "Alamat": "Bogor",
            "Gender": "Male"
        },
        {
            "nama": "Daud",
            "Alamat": "Jakarta",
            "Gender": "Male"
        }
    ]
}
```

### **3. Ambil Data dengan Multiple Filter**

**Request:**
```bash
curl -X GET "https://your-domain.com/api/in/test-receiver" \
  -H "Content-Type: application/json" \
  -d '{
    "data": "customer",
    "Gender": "Male",
    "Alamat": "Jakarta"
  }'
```

**Response:**
```json
{
    "sukses": true,
    "datas": [
        {
            "nama": "Daud",
            "Alamat": "Jakarta",
            "Gender": "Male"
        }
    ]
}
```

### **4. Ambil Data dengan Limit**

**Request:**
```bash
curl -X GET "https://your-domain.com/api/in/test-receiver" \
  -H "Content-Type: application/json" \
  -d '{
    "data": "customer",
    "limit": 2
  }'
```

**Response:**
```json
{
    "sukses": true,
    "datas": [
        {
            "nama": "sulaiman",
            "Alamat": "Bogor",
            "Gender": "Male"
        },
        {
            "nama": "Daud",
            "Alamat": "Jakarta",
            "Gender": "Male"
        }
    ]
}
```

## 🔧 Parameter yang Didukung

| Parameter | Type | Default | Deskripsi |
|-----------|------|---------|-----------|
| `data` | string | - | **Wajib** - Nama tabel yang akan diambil datanya |
| `limit` | integer | 10 | Jumlah record yang diambil |
| `offset` | integer | 0 | Jumlah record yang dilewati |
| `sort` | string | "id" | Kolom untuk sorting |
| `order` | string | "desc" | Urutan sorting ("asc" atau "desc") |
| `*` | any | - | Filter berdasarkan kolom tabel |

## 🚨 Error Handling

### **1. Tabel Tidak Ditemukan**

**Request:**
```json
{
    "data": "table_not_exists"
}
```

**Response:**
```json
{
    "sukses": false,
    "error": "Table 'table_not_exists' does not exist"
}
```

### **2. Parameter Wajib Kosong**

**Request:**
```json
{
    "limit": 10
}
```

**Response:**
```json
{
    "sukses": false,
    "error": "Table name is required in request body"
}
```

### **3. JSON Invalid**

**Request:**
```bash
curl -X GET "https://your-domain.com/api/in/test-receiver" \
  -H "Content-Type: application/json" \
  -d 'invalid json'
```

**Response:**
```json
{
    "sukses": false,
    "error": "Invalid JSON in request body"
}
```

## 💻 Contoh Penggunaan dengan JavaScript

```javascript
// Ambil data customer
fetch('https://your-domain.com/api/in/test-receiver', {
    method: 'GET',
    headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json'
    },
    body: JSON.stringify({
        data: 'customer'
    })
})
.then(response => response.json())
.then(data => {
    if (data.sukses) {
        console.log('Data customer:', data.datas);
        data.datas.forEach(customer => {
            console.log(customer.nama + ' - ' + customer.Alamat);
        });
    } else {
        console.error('Error:', data.error);
    }
});
```

## 🔍 Contoh Penggunaan dengan PHP

```php
<?php
$url = 'https://your-domain.com/api/in/test-receiver';
$body = [
    'data' => 'customer'
];

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'GET');
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body));
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Content-Type: application/json',
    'Accept: application/json'
]);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

$data = json_decode($response, true);

if ($data['sukses']) {
    echo "Data customer:\n";
    foreach ($data['datas'] as $customer) {
        echo "- {$customer['nama']} ({$customer['Alamat']})\n";
    }
} else {
    echo "Error: " . $data['error'] . "\n";
}
?>
```

## 🚀 Testing

Gunakan file `test_simple_get_response.php` untuk testing:

```bash
php test_simple_get_response.php
```

## 📋 Keunggulan Format Response

### **1. Sederhana dan Jelas**
- ✅ Response minimal dengan hanya 2 field: `sukses` dan `datas`
- ✅ Mudah dipahami dan diproses
- ✅ Tidak ada metadata yang membingungkan

### **2. Konsisten**
- ✅ Format response selalu sama untuk semua tabel
- ✅ Error handling yang konsisten
- ✅ Field names yang konsisten

### **3. Fleksibel**
- ✅ Mendukung semua jenis tabel
- ✅ Filter berdasarkan kolom apapun
- ✅ Pagination dan sorting

### **4. Performant**
- ✅ Response size minimal
- ✅ Query dioptimasi
- ✅ Caching friendly

## 🎯 Use Cases

### **1. Frontend Applications**
```javascript
// Ambil data untuk ditampilkan di tabel
fetch('/api/in/customers', {
    method: 'GET',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ data: 'customer' })
})
.then(r => r.json())
.then(data => {
    if (data.sukses) {
        renderTable(data.datas);
    }
});
```

### **2. Mobile Applications**
```javascript
// Ambil data untuk mobile app
const response = await fetch('/api/in/products', {
    method: 'GET',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ 
        data: 'products',
        limit: 20,
        category: 'electronics'
    })
});

const data = await response.json();
if (data.sukses) {
    displayProducts(data.datas);
}
```

### **3. Data Export**
```php
// Export data ke CSV
$response = file_get_contents('https://your-domain.com/api/in/customers', false, stream_context_create([
    'http' => [
        'method' => 'GET',
        'header' => 'Content-Type: application/json',
        'content' => json_encode(['data' => 'customer'])
    ]
]));

$data = json_decode($response, true);
if ($data['sukses']) {
    exportToCSV($data['datas']);
}
```

## ✅ Kesimpulan

Format response sederhana ini memberikan:

- ✅ **Kemudahan penggunaan** - Format yang intuitif
- ✅ **Fleksibilitas** - Mendukung berbagai jenis data
- ✅ **Performance** - Response size minimal
- ✅ **Konsistensi** - Format yang seragam
- ✅ **Error handling** - Pesan error yang jelas

Format ini ideal untuk aplikasi yang membutuhkan response JSON yang sederhana dan mudah diproses.
 Copyright 2025 Copilot Production
 
 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at
 
     https://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
-->

