<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\ApiSource;
use App\Models\ApiLog;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Log;

class AutoImportData extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'import:auto {--source= : Specific API source ID to import} {--all : Import all active sources} {--force : Force import even if recently fetched}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Automatically import data from API sources to database tables';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting automatic data import...');

        if ($this->option('all')) {
            $this->importAllSources();
        } elseif ($sourceId = $this->option('source')) {
            $this->importSpecificSource($sourceId);
        } else {
            $this->importScheduledSources();
        }

        $this->info('Automatic data import completed!');
    }

    /**
     * Import all active API sources
     */
    private function importAllSources()
    {
        $sources = ApiSource::where('status', 'active')->get();
        
        $this->info("Found {$sources->count()} active API sources");
        
        foreach ($sources as $source) {
            $this->importSource($source);
        }
    }

    /**
     * Import specific API source by ID
     */
    private function importSpecificSource($sourceId)
    {
        $source = ApiSource::find($sourceId);
        
        if (!$source) {
            $this->error("API source with ID {$sourceId} not found");
            return;
        }

        $this->importSource($source);
    }

    /**
     * Import sources based on their schedule
     */
    private function importScheduledSources()
    {
        $sources = ApiSource::where('status', 'active')
            ->where('schedule_type', '!=', 'manual')
            ->get();

        $this->info("Found {$sources->count()} scheduled API sources");

        foreach ($sources as $source) {
            if ($this->shouldImport($source)) {
                $this->importSource($source);
            } else {
                $this->line("Skipping {$source->name} - not due for import yet");
            }
        }
    }

    /**
     * Check if source should be imported based on schedule
     */
    private function shouldImport(ApiSource $source)
    {
        if ($this->option('force')) {
            return true;
        }

        if (!$source->last_fetched_at) {
            return true;
        }

        $lastFetched = $source->last_fetched_at;
        $now = now();

        switch ($source->schedule_type) {
            case 'hourly':
                return $lastFetched->diffInHours($now) >= ($source->schedule_frequency ?? 1);
            
            case 'daily':
                return $lastFetched->diffInDays($now) >= ($source->schedule_frequency ?? 1);
            
            case 'weekly':
                return $lastFetched->diffInWeeks($now) >= ($source->schedule_frequency ?? 1);
            
            default:
                return false;
        }
    }

    /**
     * Import data from a specific API source
     */
    private function importSource(ApiSource $source)
    {
        $this->info("Importing data from: {$source->name}");

        try {
            // Prepare HTTP request
            $request = Http::timeout(30);
            
            // Add authentication if needed
            if ($source->auth_type !== 'none') {
                $request = $this->addAuthentication($request, $source);
            }

            // Add custom headers if any
            if ($source->headers) {
                $request = $request->withHeaders($source->headers);
            }

            // Make the request
            $response = $request->get($source->url);

            if ($response->successful()) {
                $data = $response->json();
                $importedCount = $this->processAndSaveData($source, $data);

                // Log successful import
                $this->logImport($source, $response, null, $importedCount);
                $source->update(['last_fetched_at' => now()]);

                $this->info("✓ Successfully imported {$importedCount} records from {$source->name}");
            } else {
                throw new \Exception("API request failed with status: {$response->status()}");
            }

        } catch (\Exception $e) {
            $this->error("✗ Failed to import from {$source->name}: {$e->getMessage()}");
            
            // Log error
            $this->logImport($source, null, $e->getMessage(), 0);
            
            Log::error("Auto import failed for source {$source->id}: {$e->getMessage()}");
        }
    }

    /**
     * Add authentication to HTTP request
     */
    private function addAuthentication($request, ApiSource $source)
    {
        switch ($source->auth_type) {
            case 'api_key':
                return $request->withHeaders(['X-API-Key' => $source->auth_token]);
            
            case 'bearer_token':
                return $request->withHeaders(['Authorization' => 'Bearer ' . $source->auth_token]);
            
            case 'basic_auth':
                return $request->withBasicAuth($source->auth_username, $source->auth_password);
            
            default:
                return $request;
        }
    }

    /**
     * Process and save data to database
     */
    private function processAndSaveData(ApiSource $source, $data)
    {
        $tableName = $source->table_name;
        
        if (!Schema::hasTable($tableName)) {
            throw new \Exception("Table '{$tableName}' does not exist");
        }

        $tableColumns = Schema::getColumnListing($tableName);
        $insertData = [];

        // Handle different data structures
        if (is_array($data)) {
            if (isset($data['data']) && is_array($data['data'])) {
                // Structured response with 'data' key
                $records = $data['data'];
            } elseif (isset($data[0])) {
                // Array of records
                $records = $data;
            } else {
                // Single record
                $records = [$data];
            }

            foreach ($records as $record) {
                if (is_array($record)) {
                    $processedRecord = $this->mapDataToColumns($record, $tableColumns, $source);
                    if ($processedRecord) {
                        $insertData[] = $processedRecord;
                    }
                }
            }
        }

        // Insert data in batches
        if (!empty($insertData)) {
            $chunks = array_chunk($insertData, 1000); // Process in chunks of 1000
            foreach ($chunks as $chunk) {
                DB::table($tableName)->insert($chunk);
            }
        }

        return count($insertData);
    }

    /**
     * Map API data to database columns
     */
    private function mapDataToColumns($record, $tableColumns, ApiSource $source)
    {
        $mappedRecord = [];
        
        // Add timestamp fields
        $mappedRecord['created_at'] = now();
        $mappedRecord['updated_at'] = now();

        // Use field mapping if available
        if ($source->field_mapping && is_array($source->field_mapping)) {
            foreach ($source->field_mapping as $apiField => $dbField) {
                if (in_array($dbField, $tableColumns) && isset($record[$apiField])) {
                    $mappedRecord[$dbField] = $record[$apiField];
                }
            }
        } else {
            // Auto-map fields based on column names
            foreach ($record as $key => $value) {
                $dbField = $this->normalizeFieldName($key);
                if (in_array($dbField, $tableColumns)) {
                    $mappedRecord[$dbField] = $value;
                }
            }
        }

        // Always add raw_data if column exists
        if (in_array('raw_data', $tableColumns)) {
            $mappedRecord['raw_data'] = json_encode($record);
        }

        // Add source field if column exists
        if (in_array('source', $tableColumns)) {
            $mappedRecord['source'] = $source->name;
        }

        return $mappedRecord;
    }

    /**
     * Normalize field names for database columns
     */
    private function normalizeFieldName($fieldName)
    {
        // Convert camelCase to snake_case
        $normalized = preg_replace('/(?<!^)[A-Z]/', '_$0', $fieldName);
        return strtolower($normalized);
    }

    /**
     * Log import activity
     */
    private function logImport(ApiSource $source, $response = null, $error = null, $recordsCount = 0)
    {
        ApiLog::create([
            'api_source_id' => $source->id,
            'endpoint' => $source->url,
            'method' => 'GET',
            'status_code' => $response ? $response->status() : 0,
            'response_time' => $response ? ($response->handlerStats()['total_time'] ?? 0) * 1000 : 0,
            'response_body' => $response ? json_encode($response->json()) : null,
            'response_data' => $response ? json_encode($response->json()) : null,
            'error_message' => $error,
            'requested_at' => now()
        ]);
    }
} 