<?php

namespace App\Console;

use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Foundation\Console\Kernel as ConsoleKernel;

class Kernel extends ConsoleKernel
{
    /**
     * Define the application's command schedule.
     */
    protected function schedule(Schedule $schedule): void
    {
        // Auto import data from API sources based on their schedule
        $schedule->command('import:auto')
                 ->everyMinute()
                 ->withoutOverlapping()
                 ->onOneServer()
                 ->appendOutputTo(storage_path('logs/auto-import.log'));

        // Import data from hourly scheduled APIs every hour
        $schedule->command('api:import --type=hourly')
                 ->hourly()
                 ->withoutOverlapping()
                 ->onOneServer()
                 ->appendOutputTo(storage_path('logs/api-import.log'));

        // Import data from daily scheduled APIs every day at 2 AM
        $schedule->command('api:import --type=daily')
                 ->dailyAt('02:00')
                 ->withoutOverlapping()
                 ->onOneServer()
                 ->appendOutputTo(storage_path('logs/api-import.log'));

        // Send data to hourly scheduled API targets every hour
        $schedule->command('api:send --type=hourly')
                 ->hourly()
                 ->withoutOverlapping()
                 ->onOneServer()
                 ->appendOutputTo(storage_path('logs/api-send.log'));

        // Send data to daily scheduled API targets every day at 3 AM
        $schedule->command('api:send --type=daily')
                 ->dailyAt('03:00')
                 ->withoutOverlapping()
                 ->onOneServer()
                 ->appendOutputTo(storage_path('logs/api-send.log'));

        // Monitor API health every 5 minutes
        $schedule->command('api:monitor-health')
                 ->everyFiveMinutes()
                 ->withoutOverlapping()
                 ->onOneServer()
                 ->appendOutputTo(storage_path('logs/api-monitoring.log'));

        // Send detailed health report every hour
        $schedule->command('api:monitor-health --verbose')
                 ->hourly()
                 ->withoutOverlapping()
                 ->onOneServer()
                 ->appendOutputTo(storage_path('logs/api-monitoring-detailed.log'));

        // Clean up old API logs (keep only last 30 days)
        $schedule->call(function () {
            \App\Models\ApiLog::where('created_at', '<', now()->subDays(30))->delete();
            \App\Models\ApiSendLog::where('created_at', '<', now()->subDays(30))->delete();
            \App\Models\ApiReceiveLog::where('created_at', '<', now()->subDays(30))->delete();
        })->weekly()->name('cleanup-api-logs');

        // Clean up old notifications (keep only last 90 days)
        $schedule->call(function () {
            \Illuminate\Support\Facades\DB::table('notifications')
                ->where('created_at', '<', now()->subDays(90))
                ->delete();
        })->weekly()->name('cleanup-notifications');
    }

    /**
     * Register the commands for the application.
     */
    protected function commands(): void
    {
        $this->load(__DIR__.'/Commands');

        require base_path('routes/console.php');
    }
}
