<?php

namespace App\Http\Controllers;

use App\Models\ApiClient;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;

class ApiClientController extends Controller
{
    public function __construct()
    {
        // Remove auth middleware
    }

    public function index()
    {
        $clients = ApiClient::latest()->paginate(10);
        return view('api-clients.index', compact('clients'));
    }

    public function create()
    {
        return view('api-clients.create');
    }

    public function store(Request $request)
    {
        // Add debug logging
        $sessionToken = session()->token() ?? '';
        $requestToken = $request->input('_token', '');
        
        Log::info('ApiClient store method called', [
            'request_data' => $request->all(),
            'method' => $request->method(),
            'url' => $request->url(),
            'session_exists' => session()->isStarted(),
            'csrf_token' => $requestToken,
            'session_token' => $sessionToken
        ]);
        
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'description' => 'nullable|string',
                'is_active' => 'boolean',
                'rate_limit' => 'nullable|integer|min:0',
                'allowed_ips' => 'nullable|string'
            ]);

            $data = [
                'name' => $validated['name'],
                'description' => $validated['description'] ?? null,
                'is_active' => $request->has('is_active'),
            ];
            if (array_key_exists('rate_limit', $validated) && $validated['rate_limit'] !== null) {
                $data['rate_limit'] = (int) $validated['rate_limit'];
            }

            if (!empty($validated['allowed_ips'])) {
                $ips = preg_split('/[,\n\r\t ]+/', $validated['allowed_ips']);
                $ips = array_values(array_filter(array_map('trim', $ips), fn($v) => $v !== ''));
                $data['allowed_ips'] = $ips;
            }

            // Generate client credentials
            $data['client_id'] = (string) Str::uuid();
            $data['client_secret'] = Str::random(48);
            // Generate legacy tokens required by schema
            $data['usertoken'] = Str::random(40);
            $data['passtoken'] = Str::random(64);

            $client = ApiClient::create($data);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'API Client created successfully.',
                    'data' => [
                        'id' => $client->id,
                        'client_id' => $client->client_id,
                        'client_secret' => $client->client_secret,
                    ]
                ]);
            }

            // Handle different button actions
            $action = $request->input('action', 'save_and_return');

            if ($action === 'save_and_continue') {
                return redirect()->route('api-clients.show', $client->id)
                    ->with('success', 'API Client created successfully! Here are your credentials.');
            }

            return redirect()->route('api-clients.index')
                ->with('success', 'API Client created successfully.');
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed.',
                    'errors' => $e->errors()
                ], 422);
            }
            
            return back()->withErrors($e->errors())->withInput();
            
        } catch (\Exception $e) {
            Log::error('ApiClient creation error: ' . $e->getMessage(), [
                'request_data' => $request->all(),
                'stack_trace' => $e->getTraceAsString()
            ]);
            
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred while creating the API Client. Please try again.'
                ], 500);
            }
            
            return back()->with('error', 'An error occurred while creating the API Client. Please try again.')
                        ->withInput();
        }
    }

    public function show(ApiClient $apiClient)
    {
        return view('api-clients.show', compact('apiClient'));
    }

    public function edit(ApiClient $apiClient)
    {
        return view('api-clients.edit', compact('apiClient'));
    }

    public function update(Request $request, ApiClient $apiClient)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'description' => 'nullable|string',
                'is_active' => 'boolean',
                'rate_limit' => 'nullable|integer|min:0',
                'allowed_ips' => 'nullable|string'
            ]);

            $data = [
                'name' => $validated['name'],
                'description' => $validated['description'] ?? null,
                'is_active' => $request->has('is_active'),
            ];
            if (array_key_exists('rate_limit', $validated) && $validated['rate_limit'] !== null) {
                $data['rate_limit'] = (int) $validated['rate_limit'];
            }

            if (!empty($validated['allowed_ips'])) {
                $ips = preg_split('/[,\n\r\t ]+/', $validated['allowed_ips']);
                $ips = array_values(array_filter(array_map('trim', $ips), fn($v) => $v !== ''));
                $data['allowed_ips'] = $ips;
            } else {
                $data['allowed_ips'] = null;
            }

            $apiClient->update($data);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'API Client updated successfully.',
                    'data' => $validated
                ]);
            }

            return redirect()->route('api-clients.index')
                ->with('success', 'API Client updated successfully.');
                
        } catch (\Exception $e) {
            Log::error('ApiClient update error: ' . $e->getMessage(), [
                'request_data' => $request->all(),
                'stack_trace' => $e->getTraceAsString()
            ]);
            
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred while updating the API Client. Please try again.'
                ], 500);
            }
            
            return back()->with('error', 'An error occurred while updating the API Client. Please try again.')
                        ->withInput();
        }
    }

    public function destroy(ApiClient $apiClient)
    {
        // Check if client has any tokens
        if ($apiClient->tokens()->exists()) {
            return back()->with('error', 'Cannot delete client that has associated tokens.');
        }

        $apiClient->delete();

        if (request()->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'API Client deleted successfully.'
            ]);
        }

        return redirect()->route('api-clients.index')
            ->with('success', 'API Client deleted successfully.');
    }
    
    /**
     * Get client credentials for auto-sync
     */
    public function getCredentials(ApiClient $apiClient)
    {
        return response()->json([
            'success' => true,
            'client' => [
                'id' => $apiClient->id,
                'name' => $apiClient->name,
                'client_id' => $apiClient->client_id,
                'client_secret' => $apiClient->client_secret,
                'usertoken' => $apiClient->usertoken,
                'passtoken' => $apiClient->passtoken,
            ]
        ]);
    }
} 