<?php

namespace App\Http\Controllers;

use App\Models\ApiReceiver;
use App\Models\ApiReceiveLog;
use App\Models\ApiClient;
use App\Models\ApiCallbackLog;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;
use Barryvdh\DomPDF\Facade\Pdf;

class ApiReceiverController extends Controller
{
    // Removed dependency injection temporarily to fix blank page issue
    public function index()
    {
        $api_receivers = ApiReceiver::latest()->paginate(10);
        
        // Calculate proper statistics for dashboard
        $stats = [
            'total_receivers' => ApiReceiver::count(),
            'active_receivers' => ApiReceiver::where('is_active', true)->count(),
            'get_endpoints' => ApiReceiver::where('allowed_methods', 'like', '%GET%')->count(),
            'post_endpoints' => ApiReceiver::where('allowed_methods', 'like', '%POST%')->count(),
            'total_requests' => ApiReceiveLog::count(),
            'recent_requests' => ApiReceiveLog::latest()->limit(5)->count(),
        ];
        
        return view('api-receivers.index', compact('api_receivers', 'stats'));
    }

    public function create()
    {
        // Debug: Log that create method is being called
        Log::info('ApiReceiverController::create called');
        
        try {
            // Get API clients for dropdown
            $apiClients = ApiClient::where('is_active', true)->orderBy('name')->get();
            // Ambil daftar tabel dari database
            $tables = Schema::getConnection()->getDoctrineSchemaManager()->listTableNames();
            
            // Return the view with data
            return view('api-receivers.create', compact('apiClients', 'tables'));
                
        } catch (\Exception $e) {
            Log::error('Error in ApiReceiverController::create', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return redirect()->back()
                ->with('error', 'Error loading create form: ' . $e->getMessage());
        }
    }

    public function store(Request $request)
    {
        // Debug: Log request data
        Log::info('ApiReceiver store attempt', [
            'request_data' => $request->all(),
            'user_agent' => $request->userAgent(),
            'ip' => $request->ip()
        ]);

        try {
            // Validation for API Receiver
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'description' => 'nullable|string',
                'endpoint' => 'required|string|max:255|unique:api_receivers,endpoint',
                'target_table' => 'required|string|max:255',
                'auth_type' => 'required|in:none,api_key,bearer_token,basic_auth,hmac_signature',
                'auth_token' => 'nullable|string',
                'auth_username' => 'nullable|string|max:255',
                'auth_password' => 'nullable|string|max:255',
                'allowed_methods' => 'required|string|max:100',
                'field_mapping' => 'nullable|string',
                'validation_rules' => 'nullable|string',
                'response_format' => 'nullable|string|in:json,xml,plain',
                'rate_limit' => 'nullable|integer|min:1',
                'auto_create_table' => 'boolean',
                'is_active' => 'boolean',
                'allowed_ips' => 'nullable|string',
                'sample_data' => 'nullable|string',
                'api_client_id' => 'nullable|integer|exists:api_clients,id',
                'duplicate_handling' => 'nullable|in:allow,skip,update,reject',
                'unique_fields' => 'nullable|string',
                'data_filter' => 'nullable|in:all,latest_only,today_only,last_24h,last_week,new,updated',
                'max_records_per_request' => 'nullable|integer|min:1|max:1000',
                'is_bidirectional' => 'boolean',
                'callback_url' => 'nullable|url',
                'callback_method' => 'nullable|in:POST,PUT,PATCH',
                'success_response_template' => 'nullable|string',
                'error_response_template' => 'nullable|string',
                'auto_acknowledge' => 'boolean',
                'retry_attempts' => 'nullable|integer|min:0|max:10',
                'retry_delay' => 'nullable|integer|min:1',
                'enable_scheduling' => 'boolean',
                'schedule_type' => 'nullable|in:manual,interval,cron',
                'schedule_interval' => 'nullable|integer|min:1',
                'schedule_time' => 'nullable|date_format:H:i',
                'active_hours_start' => 'nullable|date_format:H:i',
                'active_hours_end' => 'nullable|date_format:H:i',
                'timezone' => 'nullable|string',
            ]);

            // Konversi checkbox ke boolean
            $validated['is_active'] = isset($validated['is_active']) ? (bool) $validated['is_active'] : false;
            $validated['auto_create_table'] = isset($validated['auto_create_table']) ? (bool) $validated['auto_create_table'] : false;
            $validated['total_received'] = 0;
            $validated['total_callbacks_sent'] = 0;
            $validated['total_callbacks_failed'] = 0;
            
            // Handle api_client_id - convert empty string to null
            if (isset($validated['api_client_id']) && (empty($validated['api_client_id']) || $validated['api_client_id'] === '')) {
                $validated['api_client_id'] = null;
            }
            
            // Auto-sync authentication from API Client if selected
            if (!empty($validated['api_client_id'])) {
                $apiClient = \App\Models\ApiClient::find($validated['api_client_id']);
                if ($apiClient && $apiClient->is_active) {
                    // Auto-set basic auth credentials from client
                    if ($validated['auth_type'] === 'basic_auth') {
                        $validated['auth_username'] = $apiClient->usertoken ?? $validated['auth_username'];
                        $validated['auth_password'] = $apiClient->passtoken ?? $validated['auth_password'];
                    }
                    // Auto-set bearer token from client
                    elseif ($validated['auth_type'] === 'bearer_token') {
                        $validated['auth_token'] = $apiClient->client_secret ?? $validated['auth_token'];
                    }
                    // Auto-set API key from client
                    elseif ($validated['auth_type'] === 'api_key') {
                        $validated['auth_token'] = $apiClient->client_id ?? $validated['auth_token'];
                    }
                }
            }

            // Process validation rules JSON
            if (!empty($validated['validation_rules'])) {
                try {
                    $rules = json_decode($validated['validation_rules'], true);
                    if (json_last_error() !== JSON_ERROR_NONE) {
                        throw new \Exception('Invalid JSON format for validation rules');
                    }
                    $validated['validation_rules'] = $rules;
                } catch (\Exception $e) {
                    return redirect()->back()
                        ->withInput()
                        ->with('error', 'Invalid validation rules JSON: ' . $e->getMessage());
                }
            } else {
                $validated['validation_rules'] = null;
            }

            // Set default values for new fields
            $validated['duplicate_handling'] = $validated['duplicate_handling'] ?? 'skip';
            $validated['unique_fields'] = $validated['unique_fields'] ?? 'id';
            $validated['data_filter'] = $validated['data_filter'] ?? 'latest_only';
            $validated['max_records_per_request'] = $validated['max_records_per_request'] ?? 10;
            $validated['response_format'] = $validated['response_format'] ?? 'json';
            $validated['timezone'] = 'Asia/Jakarta';
            $validated['active_hours_start'] = '00:00';
            $validated['active_hours_end'] = '23:59';

            // Convert checkbox values to proper booleans
            $validated['is_bidirectional'] = isset($validated['is_bidirectional']) ? (bool) $validated['is_bidirectional'] : false;
            $validated['auto_acknowledge'] = isset($validated['auto_acknowledge']) ? (bool) $validated['auto_acknowledge'] : false;
            $validated['enable_scheduling'] = isset($validated['enable_scheduling']) ? (bool) $validated['enable_scheduling'] : false;

            // Generate unique endpoint jika tidak ada
            if (empty($validated['endpoint'])) {
                $validated['endpoint'] = Str::slug($validated['name']) . '-' . Str::random(6);
            }

            Log::info('ApiReceiver validated data', ['validated' => $validated]);

            $apiReceiver = ApiReceiver::create($validated);

            Log::info('ApiReceiver created successfully', ['id' => $apiReceiver->id]);

            return redirect()->route('api-receivers.index')
                ->with('success', 'API Receiver created successfully.');

        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('ApiReceiver validation failed', [
                'errors' => $e->errors(),
                'request_data' => $request->all()
            ]);
            throw $e;
        } catch (\Exception $e) {
            Log::error('ApiReceiver creation failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Failed to create API Receiver: ' . $e->getMessage());
        }
    }

    public function show(ApiReceiver $apiReceiver)
    {
        return view('api-receivers.show', compact('apiReceiver'));
    }

    public function edit(ApiReceiver $apiReceiver)
    {
        // Ambil daftar tabel dari database
        $tables = Schema::getConnection()->getDoctrineSchemaManager()->listTableNames();
        // Ambil API clients untuk dropdown (samakan dengan create)
        $apiClients = ApiClient::where('is_active', true)->orderBy('name')->get();
        return view('api-receivers.edit', compact('apiReceiver', 'tables', 'apiClients'));
    }

    public function update(Request $request, ApiReceiver $apiReceiver)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'endpoint' => 'required|string|max:255|unique:api_receivers,endpoint,' . $apiReceiver->id,
            'target_table' => 'required|string|max:255',
            'auth_type' => 'required|in:none,api_key,bearer_token,basic_auth,hmac_signature',
            'auth_token' => 'nullable|string',
            'auth_secret' => 'nullable|string',
            'auth_username' => 'nullable|string|max:255',
            'auth_password' => 'nullable|string|max:255',
            'allowed_methods' => 'required|string',
            'field_mapping' => 'nullable|string',
            'validation_rules' => 'nullable|string',
            'response_format' => 'nullable|string|in:json,xml,plain',
            'rate_limit' => 'nullable|integer|min:1',
            'auto_create_table' => 'boolean',
            'is_active' => 'boolean',
            'allowed_ips' => 'nullable|string',
            'sample_data' => 'nullable|string',
            'api_client_id' => 'nullable|exists:api_clients,id',
            'duplicate_handling' => 'nullable|in:allow,skip,update,reject',
            'unique_fields' => 'nullable|string',
            'data_filter' => 'nullable|in:all,latest_only,today_only,last_24h,last_week,new,updated',
            'max_records_per_request' => 'nullable|integer|min:1|max:1000',
            'enable_scheduling' => 'boolean',
            'schedule_type' => 'nullable|in:manual,interval,cron',
            'schedule_interval' => 'nullable|integer|min:1',
            'schedule_time' => 'nullable|date_format:H:i',
            'active_hours_start' => 'nullable|date_format:H:i',
            'active_hours_end' => 'nullable|date_format:H:i',
            'timezone' => 'nullable|string',
            'is_bidirectional' => 'boolean',
            'callback_url' => 'nullable|url',
            'callback_method' => 'nullable|in:POST,PUT,PATCH',
            'success_response_template' => 'nullable|string',
            'error_response_template' => 'nullable|string',
            'auto_acknowledge' => 'boolean',
            'retry_attempts' => 'nullable|integer|min:0|max:10',
            'retry_delay' => 'nullable|integer|min:1'
        ]);

        // Convert checkbox string values to proper booleans (null-safe)
        $validated['is_active'] = (bool) ($validated['is_active'] ?? false);
        $validated['auto_create_table'] = (bool) ($validated['auto_create_table'] ?? false);
        $validated['is_bidirectional'] = (bool) ($validated['is_bidirectional'] ?? false);
        $validated['auto_acknowledge'] = (bool) ($validated['auto_acknowledge'] ?? false);
        $validated['enable_scheduling'] = (bool) ($validated['enable_scheduling'] ?? false);
        
        // Process validation rules JSON
        if (!empty($validated['validation_rules'])) {
            try {
                $rules = json_decode($validated['validation_rules'], true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    throw new \Exception('Invalid JSON format for validation rules');
                }
                $validated['validation_rules'] = $rules;
            } catch (\Exception $e) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Invalid validation rules JSON: ' . $e->getMessage());
            }
        } else {
            $validated['validation_rules'] = null;
        }
        
        $apiReceiver->update($validated);

        return redirect()->route('api-receivers.index')
            ->with('success', 'API Receiver updated successfully.');
    }

    public function destroy(ApiReceiver $apiReceiver)
    {
        $apiReceiver->delete();

        return redirect()->route('api-receivers.index')
            ->with('success', 'API Receiver deleted successfully.');
    }

    public function receive(Request $request, $endpoint)
    {
        try {
            // Log the incoming request for debugging
            Log::info('API Receiver request received', [
                'endpoint' => $endpoint,
                'method' => $request->method(),
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'headers' => $request->headers->all(),
                'query_params' => $request->query(),
                'body' => $request->getContent()
            ]);

            // Find the receiver
            $receiver = ApiReceiver::where('endpoint', $endpoint)
                ->where('is_active', true)
                ->first();

            if (!$receiver) {
                // Log the error
                Log::warning('Receiver not found', [
                    'endpoint' => $endpoint,
                    'available_endpoints' => ApiReceiver::where('is_active', true)->pluck('endpoint')->toArray()
                ]);

                // Return simple error response for GET requests
                if ($request->method() === 'GET') {
                    return response()->json([
                        'sukses' => false,
                        'error' => 'Receiver not found'
                    ], 404);
                }

                // Return detailed error response for other methods
                return response()->json([
                    'success' => false,
                    'error' => 'Receiver not found',
                    'message' => "No active receiver found for endpoint: {$endpoint}",
                    'code' => 404,
                    'available_endpoints' => ApiReceiver::where('is_active', true)->pluck('endpoint')->toArray(),
                    'received_at' => now()->toISOString(),
                    'request_info' => [
                        'method' => $request->method(),
                        'ip' => $request->ip(),
                        'user_agent' => $request->userAgent()
                    ]
                ], 404);
            }

            // Log receiver found
            Log::info('Receiver found', [
                'receiver_id' => $receiver->id,
                'receiver_name' => $receiver->name,
                'allowed_methods' => $receiver->allowed_methods,
                'auth_type' => $receiver->auth_type
            ]);

            // Check allowed methods
            $allowedMethods = array_map('trim', explode(',', $receiver->allowed_methods));
            if (!in_array($request->method(), $allowedMethods)) {
                Log::warning('Method not allowed', [
                    'request_method' => $request->method(),
                    'allowed_methods' => $allowedMethods,
                    'receiver_id' => $receiver->id
                ]);

                // Return simple error response for GET requests
                if ($request->method() === 'GET') {
                    return response()->json([
                        'sukses' => false,
                        'error' => "Method '{$request->method()}' not allowed"
                    ], 405);
                }

                // Return detailed error response for other methods
                return $this->errorResponse(
                    "Method '{$request->method()}' not allowed. Allowed methods: " . implode(', ', $allowedMethods), 
                    405, 
                    $request, 
                    $receiver
                );
            }

            // Enhanced authentication with client token synchronization
            $authResult = $this->validateAuthentication($request, $receiver);
            if (!$authResult['success']) {
                Log::warning('Authentication failed', [
                    'receiver_id' => $receiver->id,
                    'auth_type' => $receiver->auth_type,
                    'error' => $authResult['message']
                ]);

                // Return simple error response for GET requests
                if ($request->method() === 'GET') {
                    return response()->json([
                        'sukses' => false,
                        'error' => $authResult['message']
                    ], $authResult['code']);
                }

                // Return detailed error response for other methods
                return $this->errorResponse($authResult['message'], $authResult['code'], $request, $receiver);
            }

            // Rate limiting check
            $rateLimitResult = $this->checkRateLimit($request, $receiver);
            if (!$rateLimitResult['success']) {
                Log::warning('Rate limit exceeded', [
                    'receiver_id' => $receiver->id,
                    'ip' => $request->ip(),
                    'rate_limit' => $receiver->rate_limit
                ]);

                // Return simple error response for GET requests
                if ($request->method() === 'GET') {
                    return response()->json([
                        'sukses' => false,
                        'error' => $rateLimitResult['message']
                    ], 429);
                }

                // Return detailed error response for other methods
                return $this->errorResponse($rateLimitResult['message'], 429, $request, $receiver);
            }

            // Special handling for GET-only receivers
            if ($request->method() === 'GET' && $this->isGetOnlyReceiver($receiver)) {
                Log::info('Processing GET-only receiver', ['receiver_id' => $receiver->id]);
                return $this->handleGetOnlyReceiver($request, $receiver, $authResult['client']);
            }

            // Handle other methods (POST, PUT, etc.)
            Log::info('Processing data receiver', [
                'receiver_id' => $receiver->id,
                'method' => $request->method()
            ]);
            return $this->handleDataReceiver($request, $receiver, $authResult['client']);

        } catch (\Exception $e) {
            // Log the exception
            Log::error('API Receiver error', [
                'endpoint' => $endpoint,
                'method' => $request->method(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => [
                    'ip' => $request->ip(),
                    'user_agent' => $request->userAgent(),
                    'headers' => $request->headers->all()
                ]
            ]);

            // Return simple error response for GET requests
            if ($request->method() === 'GET') {
                return response()->json([
                    'sukses' => false,
                    'error' => 'Internal server error'
                ], 500);
            }

            // Return detailed error response for other methods
            return response()->json([
                'success' => false,
                'error' => 'Internal server error',
                'message' => 'An unexpected error occurred while processing your request',
                'code' => 500,
                'received_at' => now()->toISOString(),
                'debug_info' => config('app.debug') ? [
                    'exception' => $e->getMessage(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ] : null
            ], 500);
        }
    }

    /**
     * Enhanced authentication with client token synchronization
     */
    protected function validateAuthentication(Request $request, ApiReceiver $receiver)
    {
        if ($receiver->auth_type === 'none') {
            return ['success' => true, 'client' => null];
        }

        $token = null;
        $client = null;

        // Extract token based on auth type
        if ($receiver->auth_type === 'api_key') {
            $token = $request->header('X-API-Key') ?? $request->query('api_key');
        } elseif ($receiver->auth_type === 'bearer_token') {
            $authHeader = $request->header('Authorization');
            
            // Debug logging for Bearer token
            Log::info('Bearer Token Debug', [
                'auth_header' => $authHeader,
                'receiver_id' => $receiver->id,
                'endpoint' => $receiver->endpoint,
                'api_client_id' => $receiver->api_client_id
            ]);
            
            if ($authHeader && str_starts_with($authHeader, 'Bearer ')) {
                $token = substr($authHeader, 7);
                Log::info('Bearer token extracted', [
                    'token' => $token,
                    'token_length' => strlen($token)
                ]);
            } else {
                $token = $request->query('token');
                Log::warning('Bearer token not found in header, using query parameter', [
                    'query_token' => $token
                ]);
            }
        } elseif ($receiver->auth_type === 'basic_auth') {
            // Basic Authentication validation
            $authHeader = $request->header('Authorization');
            
            // Debug logging
            Log::info('Basic Auth Debug', [
                'auth_header' => $authHeader,
                'receiver_username' => $receiver->auth_username,
                'receiver_password' => $receiver->auth_password,
                'endpoint' => $receiver->endpoint
            ]);
            
            if (!$authHeader || !str_starts_with($authHeader, 'Basic ')) {
                Log::warning('Basic Auth header missing or invalid', [
                    'auth_header' => $authHeader,
                    'endpoint' => $receiver->endpoint
                ]);
                return [
                    'success' => false,
                    'message' => 'Basic Authentication required',
                    'code' => 401
                ];
            }
            
            // Extract credentials from Authorization header
            $credentials = base64_decode(substr($authHeader, 6));
            $parts = explode(':', $credentials, 2);
            
            if (count($parts) !== 2) {
                Log::warning('Invalid Basic Auth format', [
                    'credentials' => $credentials,
                    'parts_count' => count($parts),
                    'endpoint' => $receiver->endpoint
                ]);
                return [
                    'success' => false,
                    'message' => 'Invalid Basic Authentication format',
                    'code' => 401
                ];
            }
            
            $username = $parts[0];
            $password = $parts[1];
            
            // Debug logging for credentials
            Log::info('Basic Auth Credentials Check', [
                'provided_username' => $username,
                'provided_password' => $password,
                'stored_username' => $receiver->auth_username,
                'stored_password' => $receiver->auth_password,
                'username_match' => $username === $receiver->auth_username,
                'password_match' => $password === $receiver->auth_password,
                'endpoint' => $receiver->endpoint
            ]);
            
            // Validate against receiver's stored credentials
            if ($username !== $receiver->auth_username || $password !== $receiver->auth_password) {
                Log::warning('Basic Auth credentials mismatch', [
                    'provided_username' => $username,
                    'provided_password' => $password,
                    'stored_username' => $receiver->auth_username,
                    'stored_password' => $receiver->auth_password,
                    'endpoint' => $receiver->endpoint
                ]);
                return [
                    'success' => false,
                    'message' => 'Invalid Basic Authentication credentials',
                    'code' => 401
                ];
            }
            
            Log::info('Basic Auth successful', [
                'username' => $username,
                'endpoint' => $receiver->endpoint
            ]);
            
            $token = $username; // Use username as token for further processing
        } elseif ($receiver->auth_type === 'hmac_signature') {
            // HMAC signature validation
            $signature = $request->header('X-Signature') ?? $request->header('X-HMAC-Signature');
            if (!$signature) {
                return [
                    'success' => false,
                    'message' => 'HMAC signature required',
                    'code' => 401
                ];
            }
            
            // Validate HMAC signature
            $payload = $request->getContent();
            $expectedSignature = hash_hmac('sha256', $payload, $receiver->auth_secret);
            
            if (!hash_equals($expectedSignature, $signature)) {
                return [
                    'success' => false,
                    'message' => 'Invalid HMAC signature',
                    'code' => 401
                ];
            }
            
            $token = $signature; // Use signature as token for further processing
        }

        if (!$token) {
            return [
                'success' => false,
                'message' => 'Authentication token required',
                'code' => 401
            ];
        }

        // Validate token against receiver's auth_token (skip for HMAC and Basic Auth as they're already validated)
        if ($receiver->auth_type !== 'hmac_signature' && $receiver->auth_type !== 'basic_auth' && $token !== $receiver->auth_token) {
            Log::warning('Token validation failed', [
                'receiver_id' => $receiver->id,
                'auth_type' => $receiver->auth_type,
                'provided_token' => $token,
                'receiver_auth_token' => $receiver->auth_token,
                'tokens_match' => $token === $receiver->auth_token
            ]);
            
            return [
                'success' => false,
                'message' => 'Invalid authentication token',
                'code' => 401
            ];
        }

        // Synchronize with API client if receiver is linked to a client
        if ($receiver->api_client_id) {
            $client = \App\Models\ApiClient::find($receiver->api_client_id);

            if (!$client || !$client->is_active) {
                return [
                    'success' => false,
                    'message' => 'Associated API client is inactive',
                    'code' => 403
                ];
            }

            // Check if client has expired
            if ($client->expires_at && $client->expires_at->isPast()) {
                return [
                    'success' => false,
                    'message' => 'Associated API client has expired',
                    'code' => 403
                ];
            }

            // Only enforce client token validation for bearer/api_key.
            // For basic_auth and hmac_signature we already validated above, so skip client token check.
            $authTypeRequiresClientToken = in_array($receiver->auth_type, ['bearer_token', 'api_key']);

            if ($authTypeRequiresClientToken) {
                // If receiver has a specific auth_token configured and it matches, accept without requiring it to exist in client's tokens
                $matchesReceiverToken = !empty($receiver->auth_token) && hash_equals((string) $receiver->auth_token, (string) $token);

                if (!$matchesReceiverToken) {
                    // Validate token against client's tokens
                    $validToken = $client->tokens()
                        ->where('token', $token)
                        ->where('is_revoked', false)
                        ->where(function($query) {
                            $query->whereNull('expires_at')
                                  ->orWhere('expires_at', '>', now());
                        })
                        ->first();

                    if (!$validToken) {
                        Log::warning('Token not found in client records', [
                            'receiver_id' => $receiver->id,
                            'client_id' => $client->id,
                            'provided_token' => $token,
                            'client_tokens_count' => $client->tokens()->count()
                        ]);
                        
                        return [
                            'success' => false,
                            'message' => 'Token not found or expired in client records',
                            'code' => 401
                        ];
                    }

                    // Update token last used
                    $validToken->update(['last_used_at' => now()]);
                    $client->update(['last_used_at' => now()]);
                } else {
                    // Token already matched receiver's configured token; optionally update client last used timestamp
                    $client->update(['last_used_at' => now()]);
                }
            }
        }

        return [
            'success' => true,
            'client' => $client,
            'token' => $token
        ];
    }

    /**
     * Check rate limiting
     */
    protected function checkRateLimit(Request $request, ApiReceiver $receiver)
    {
        if (!$receiver->rate_limit) {
            return ['success' => true];
        }

        $key = 'rate_limit:receiver:' . $receiver->id . ':' . $request->ip();
        $attempts = cache()->get($key, 0);

        if ($attempts >= $receiver->rate_limit) {
            return [
                'success' => false,
                'message' => 'Rate limit exceeded. Maximum ' . $receiver->rate_limit . ' requests per hour.'
            ];
        }

        cache()->put($key, $attempts + 1, now()->addHour());
        return ['success' => true];
    }

    /**
     * Check if this is a GET-only receiver
     */
    protected function isGetOnlyReceiver(ApiReceiver $receiver)
    {
        $allowedMethods = array_map('trim', explode(',', $receiver->allowed_methods));
        return count($allowedMethods) === 1 && $allowedMethods[0] === 'GET';
    }

    /**
     * Handle GET-only receiver requests
     */
    protected function handleGetOnlyReceiver(Request $request, ApiReceiver $receiver, $client = null)
    {
        try {
            Log::info('handleGetOnlyReceiver called', [
                'receiver_id' => $receiver->id,
                'method' => $request->method(),
                'endpoint' => $receiver->endpoint
            ]);

            // Check if there's body data to save (like POST)
            $bodyData = $request->getContent();
            $hasBodyData = !empty($bodyData);
            
            Log::info('Body data check', [
                'receiver_id' => $receiver->id,
                'has_body_data' => $hasBodyData,
                'body_size' => strlen($bodyData),
                'body_content' => $bodyData
            ]);
            
            // If there's body data, check if it's for fetching data from table
            if ($hasBodyData) {
                Log::info('GET request with body data detected', [
                    'receiver_id' => $receiver->id,
                    'body_size' => strlen($bodyData),
                    'content_type' => $request->header('Content-Type')
                ]);
                
                // Parse JSON body data
                $data = json_decode($bodyData, true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    // Return simple error response for GET requests
                    return response()->json([
                        'sukses' => false,
                        'error' => 'Invalid JSON in request body'
                    ], 400);
                }
                
                // Check if this is a data fetch request (contains 'data' or 'table' field)
                if (isset($data['data']) || isset($data['table'])) {
                    Log::info('Calling handleGetWithBodyData', [
                        'receiver_id' => $receiver->id,
                        'data' => $data
                    ]);
                    return $this->handleGetWithBodyData($request, $receiver, $client);
                }
                
                // If GET request body doesn't contain 'data' or 'table', return error
                // Don't save data or show data for invalid GET requests
                Log::warning('Invalid GET request body - missing data/table field', [
                        'receiver_id' => $receiver->id,
                    'received_data' => $data,
                    'message' => 'GET request body must contain "data" or "table" field to fetch data'
                ]);
                
                // Return simple error response for GET requests
                return response()->json([
                    'sukses' => false,
                    'error' => 'GET request body must contain "data" or "table" field to fetch data'
                ], 400);
                }
            
            // If no body data, return error - GET request must have body data to fetch data
            Log::warning('GET request without body data - data fetching requires body', [
                'receiver_id' => $receiver->id,
                'message' => 'GET request must contain body data with "data" or "table" field to fetch data'
            ]);

            return response()->json([
                'sukses' => false,
                'error' => 'GET request must contain body data to fetch data. Use {"data": "nama_tabel"} or {"table": "nama_tabel"}'
            ], 400);

        } catch (\Exception $e) {
            // Log error
            $this->logReceiveRequest($request, $receiver, $client, 500, 'Error: ' . $e->getMessage(), true, false, false, $e->getMessage());

            // Return simple error response for GET requests
            return response()->json([
                'sukses' => false,
                'error' => 'Processing failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Handle GET request with body data to fetch from table
     */
    protected function handleGetWithBodyData(Request $request, ApiReceiver $receiver, $client = null)
    {
        try {
            // Get data from request body
            $data = $request->all();
            
            Log::info('Processing GET with body data', [
                'receiver_id' => $receiver->id,
                'data' => $data,
                'target_table' => $receiver->target_table
            ]);

            // Check if data contains table name or other parameters
            $tableName = $data['data'] ?? $data['table'] ?? $receiver->target_table;
            $limit = $data['limit'] ?? 10;
            $offset = $data['offset'] ?? 0;
            $sort = $data['sort'] ?? 'id';
            $order = $data['order'] ?? 'desc';
            
            // Validate table name
            if (!$tableName) {
                // Return simple error response for GET requests
                return response()->json([
                    'sukses' => false,
                    'error' => 'Table name is required in request body'
                ], 400);
            }

            // Check if table exists
            if (!Schema::hasTable($tableName)) {
                // Return simple error response for GET requests
                return response()->json([
                    'sukses' => false,
                    'error' => "Table '{$tableName}' does not exist"
                ], 404);
            }

            // Build query
            $query = DB::table($tableName);
            
            // Apply filters from request body (excluding pagination and sorting params)
            $excludeParams = ['data', 'table', 'limit', 'offset', 'sort', 'order', 'api_key', 'token'];
            foreach ($data as $key => $value) {
                if (!in_array($key, $excludeParams) && $value !== null) {
                    $query->where($key, $value);
                }
            }

            // Get total count
            $totalRecords = $query->count();
            
            // Apply pagination and sorting
            $records = $query->orderBy($sort, $order)
                          ->limit($limit)
                          ->offset($offset)
                          ->get();

            // Prepare response dengan format yang diminta
                $response = [
                'sukses' => true,
                'data' => $records->toArray()
            ];

            // Log query results
            Log::info('Query results', [
                'receiver_id' => $receiver->id,
                'table' => $tableName,
                'total_records' => $totalRecords,
                'returned_records' => $records->count(),
                'response_data' => $response
            ]);

            // Add debug info if requested
            $debugMode = ($request->boolean('debug') || $request->header('X-Debug') === 'true' || config('app.debug'));
            if ($debugMode) {
                $response['debug'] = [
                    'request_body' => $data,
                    'applied_filters' => array_diff_key($data, array_flip($excludeParams)),
                    'query_info' => [
                        'table' => $tableName,
                        'total_count' => $totalRecords,
                        'returned_count' => $records->count()
                    ]
                ];
            }

            // Log the request
            $this->logReceiveRequest($request, $receiver, $client, 200, 'GET with body data successful', true, true, true);

            // Update receiver statistics
            $receiver->increment('total_received');
            $receiver->update(['last_received_at' => now()]);

            // Log response sebelum dikirim
            Log::info('Sending GET response', [
                'receiver_id' => $receiver->id,
                'response' => $response,
                'response_size' => strlen(json_encode($response))
            ]);

            // Log final response
            Log::info('Final response being sent', [
                'receiver_id' => $receiver->id,
                'response' => $response,
                'response_json' => json_encode($response)
            ]);

            // Return response langsung tanpa wrapper
            return response()->json($response, 200);

        } catch (\Exception $e) {
            Log::error('Error in GET with body data', [
                'receiver_id' => $receiver->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            $this->logReceiveRequest($request, $receiver, $client, 500, 'Error: ' . $e->getMessage(), true, false, false, $e->getMessage());

            // Return simple error response for GET requests
            return response()->json([
                'sukses' => false,
                'error' => 'Processing failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Validate GET request parameters
     */
    protected function validateGetParameters(Request $request, ApiReceiver $receiver)
    {
        $rules = [];
        $messages = [];

        // Add validation rules based on receiver configuration
        if ($receiver->validation_rules) {
            $validationRules = is_string($receiver->validation_rules) 
                ? json_decode($receiver->validation_rules, true) 
                : $receiver->validation_rules;

            if (is_array($validationRules)) {
                foreach ($validationRules as $field => $fieldRules) {
                    $fieldRule = '';
                    
                    // Handle required field
                    if (isset($fieldRules['required']) && $fieldRules['required']) {
                        $fieldRule = 'required';
                    }
                    
                    // Handle data type validation
                    if (isset($fieldRules['type'])) {
                        switch ($fieldRules['type']) {
                            case 'integer':
                                $fieldRule .= $fieldRule ? '|integer' : 'integer';
                                break;
                            case 'numeric':
                                $fieldRule .= $fieldRule ? '|numeric' : 'numeric';
                                break;
                            case 'email':
                                $fieldRule .= $fieldRule ? '|email' : 'email';
                                break;
                            case 'date':
                                $fieldRule .= $fieldRule ? '|date' : 'date';
                                break;
                            case 'string':
                                $fieldRule .= $fieldRule ? '|string' : 'string';
                                break;
                            case 'boolean':
                                $fieldRule .= $fieldRule ? '|boolean' : 'boolean';
                                break;
                            case 'array':
                                $fieldRule .= $fieldRule ? '|array' : 'array';
                                break;
                        }
                    }

                    // Handle length constraints
                    if (isset($fieldRules['max_length']) && is_numeric($fieldRules['max_length'])) {
                        $fieldRule .= $fieldRule ? '|max:' . $fieldRules['max_length'] : 'max:' . $fieldRules['max_length'];
                    }

                    if (isset($fieldRules['min_length']) && is_numeric($fieldRules['min_length'])) {
                        $fieldRule .= $fieldRule ? '|min:' . $fieldRules['min_length'] : 'min:' . $fieldRules['min_length'];
                    }

                    // Handle numeric constraints
                    if (isset($fieldRules['min_value']) && is_numeric($fieldRules['min_value'])) {
                        $fieldRule .= $fieldRule ? '|min:' . $fieldRules['min_value'] : 'min:' . $fieldRules['min_value'];
                    }

                    if (isset($fieldRules['max_value']) && is_numeric($fieldRules['max_value'])) {
                        $fieldRule .= $fieldRule ? '|max:' . $fieldRules['max_value'] : 'max:' . $fieldRules['max_value'];
                    }

                    // Handle enum/in validation
                    if (isset($fieldRules['options']) && is_array($fieldRules['options'])) {
                        $fieldRule .= $fieldRule ? '|in:' . implode(',', $fieldRules['options']) : 'in:' . implode(',', $fieldRules['options']);
                    }

                    // Handle regex validation
                    if (isset($fieldRules['pattern']) && is_string($fieldRules['pattern'])) {
                        $fieldRule .= $fieldRule ? '|regex:' . $fieldRules['pattern'] : 'regex:' . $fieldRules['pattern'];
                    }

                    // Add custom error messages
                    if (isset($fieldRules['message']) && is_string($fieldRules['message'])) {
                        $messages[$field] = $fieldRules['message'];
                    }

                    // Only add rule if it's not empty
                    if (!empty($fieldRule)) {
                        $rules[$field] = $fieldRule;
                    }
                }
            }
        }

        // Validate the request
        try {
            $validator = Validator::make($request->all(), $rules, $messages);
            
            if ($validator->fails()) {
                return [
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()->toArray(),
                    'failed_fields' => array_keys($validator->errors()->toArray())
                ];
            }

            return [
                'success' => true,
                'validated_data' => $validator->validated()
            ];
        } catch (\Exception $e) {
            Log::error('GET parameter validation error', [
                'receiver_id' => $receiver->id,
                'error' => $e->getMessage(),
                'rules' => $rules,
                'request_data' => $request->all()
            ]);

            return [
                'success' => false,
                'message' => 'Validation system error: ' . $e->getMessage(),
                'error_type' => 'validation_system_error'
            ];
        }
    }

    /**
     * Fetch data for GET receiver
     */
    protected function fetchDataForGetReceiver(Request $request, ApiReceiver $receiver)
    {
        // Get query parameters
        $params = $request->all();
        
        // Apply data filtering based on receiver configuration
        $limit = min($receiver->max_records_per_request ?? 10, 1000); // Cap at 1000
        $offset = $request->get('offset', 0);
        $sort = $request->get('sort', 'id');
        $order = $request->get('order', 'desc');

        // Basic data structure for GET response
        $data = [
            'receiver_info' => [
                'id' => $receiver->id,
                'name' => $receiver->name,
                'endpoint' => $receiver->endpoint,
                'description' => $receiver->description
            ],
            'request_info' => [
                'method' => $request->method(),
                'received_at' => now()->toISOString(),
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent()
            ],
            'parameters' => $params,
            'pagination' => [
                'limit' => $limit,
                'offset' => $offset,
                'sort' => $sort,
                'order' => $order
            ]
        ];

        // If receiver has a target table, try to fetch data from it
        if ($receiver->target_table && Schema::hasTable($receiver->target_table)) {
            try {
                $query = DB::table($receiver->target_table);
                
                // Apply filters from query parameters
                foreach ($params as $key => $value) {
                    if (!in_array($key, ['limit', 'offset', 'sort', 'order', 'api_key', 'token'])) {
                        $query->where($key, $value);
                    }
                }

                $total = $query->count();
                $records = $query->orderBy($sort, $order)
                              ->limit($limit)
                              ->offset($offset)
                              ->get();

                $data['data'] = [
                    'total_records' => $total,
                    'returned_records' => $records->count(),
                    'records' => $records->toArray()  // Convert to array for better JSON response
                ];
            } catch (\Exception $e) {
                $data['data'] = [
                    'error' => 'Unable to fetch data from target table',
                    'message' => $e->getMessage()
                ];
            }
        } else {
            $data['data'] = [
                'message' => 'No data source configured or table does not exist'
            ];
        }

        return $data;
    }

    /**
     * Format GET response based on receiver configuration
     */
    protected function formatGetResponse($data, ApiReceiver $receiver)
    {
        // Response with data display - simplify for GET requests
        $response = [
            'success' => true,
            'message' => 'Data retrieved successfully',
            'received_at' => now()->toISOString()
        ];
        
        // Extract relevant data for display
        if (isset($data['data']) && isset($data['data']['records'])) {
            // If there are records from database, show them
            $response['data'] = $data['data']['records'];
            $response['total_records'] = $data['data']['total_records'] ?? 0;
        } elseif (isset($data['data'])) {
            // If there's other data, show it
            $response['data'] = $data['data'];
        } else {
            // Fallback to show receiver info
            $response['data'] = [
                'receiver_name' => $data['receiver_info']['name'] ?? 'Unknown',
                'endpoint' => $data['receiver_info']['endpoint'] ?? 'Unknown',
                'message' => 'No data available'
            ];
        }

        // Format response based on receiver configuration
        switch ($receiver->response_format) {
            case 'xml':
                return response($this->arrayToXml($response), 200)
                    ->header('Content-Type', 'application/xml');
            
            case 'plain':
                $plainText = "Success: Data retrieved\n";
                $plainText .= "Received at: " . now()->toISOString();
                return response($plainText, 200)
                    ->header('Content-Type', 'text/plain');
            
            default: // json
                return response()->json($response, 200);
        }
    }

    /**
     * Handle regular data receiver (POST, PUT, etc.)
     */
    protected function handleDataReceiver(Request $request, ApiReceiver $receiver, $client = null)
    {
        try {
            $data = $request->all();
            
            // Log the received data
            Log::info('Processing data receiver request', [
                'receiver_id' => $receiver->id,
                'method' => $request->method(),
                'data_size' => count($data),
                'target_table' => $receiver->target_table
            ]);

            $savedToTable = false;
            $saveError = null;
            $insertedIds = [];
            $insertedData = [];
            
            // Check if user wants to skip database storage (for POST requests)
            $skipDatabaseStorage = $request->has('skip_storage') || $request->header('X-Skip-Storage') === 'true';
            
            // Save data to target table if configured and not skipped
            if ($receiver->target_table && !$skipDatabaseStorage) {
                Log::info('Starting database save process', [
                    'receiver_id' => $receiver->id,
                    'target_table' => $receiver->target_table,
                    'data_size' => count($data),
                    'skip_storage' => $skipDatabaseStorage,
                    'auto_create_table' => $receiver->auto_create_table
                ]);
                try {
                    // Check if table exists
                    Log::info('Checking table existence', [
                        'table' => $receiver->target_table,
                        'exists' => Schema::hasTable($receiver->target_table),
                        'auto_create' => $receiver->auto_create_table
                    ]);
                    
                        if (!Schema::hasTable($receiver->target_table)) {
                            $autoCreateRequested = ($receiver->auto_create_table || $request->boolean('auto_create_table') || $request->header('X-Auto-Create') === 'true');
                            if ($autoCreateRequested) {
                                Log::info('Creating table automatically', ['table' => $receiver->target_table]);
                            
                            // Auto-create table with proper structure for JSON data
                                Schema::create($receiver->target_table, function ($table) {
                                $table->id();
                                $table->integer('external_id')->nullable(); // Changed from 'id' to 'external_id'
                                $table->string('nama')->nullable();
                                $table->string('alamat')->nullable();
                                $table->string('gender')->nullable();
                                    // Use longText for broad hosting compatibility
                                    $table->longText('data')->nullable();
                                $table->timestamps();
                            });
                            Log::info('Auto-created table with JSON support', ['table' => $receiver->target_table]);
                        } else {
                            throw new \Exception("Target table '{$receiver->target_table}' does not exist and auto-create is disabled");
                        }
                    } else {
                        Log::info('Table already exists', ['table' => $receiver->target_table]);
                    }

                    // Get table columns
                    $columns = Schema::getColumnListing($receiver->target_table);
                    Log::info('Table columns detected', [
                        'table' => $receiver->target_table,
                        'columns' => $columns,
                        'columns_count' => count($columns)
                    ]);
                    
                    // Prepare data for insertion
                    $insertData = [];
                    
                    // Get JSON columns using raw query to avoid deprecated methods
                    $jsonColumns = [];
                    try {
                        $columnTypes = DB::select("
                            SELECT COLUMN_NAME, DATA_TYPE 
                            FROM INFORMATION_SCHEMA.COLUMNS 
                            WHERE TABLE_SCHEMA = DATABASE() 
                            AND TABLE_NAME = ? 
                            AND DATA_TYPE = 'json'
                        ", [$receiver->target_table]);
                        
                        foreach ($columnTypes as $column) {
                            $jsonColumns[] = $column->COLUMN_NAME;
                        }
                    } catch (\Exception $e) {
                        Log::warning('Failed to detect JSON columns', [
                            'table' => $receiver->target_table,
                            'error' => $e->getMessage()
                        ]);
                    }
                    
                    // If data is an array of records, process each one
                    if (isset($data[0]) && is_array($data[0])) {
                        Log::info('Processing multiple records', ['count' => count($data)]);
                        foreach ($data as $index => $record) {
                            $filtered = array_intersect_key($record, array_flip($columns));
                            
                            // Handle field mapping for auto-created tables
                            if ($receiver->auto_create_table && isset($filtered['id'])) {
                                $filtered['external_id'] = $filtered['id'];
                                unset($filtered['id']);
                            }
                            
                            $filtered['created_at'] = now();
                            $filtered['updated_at'] = now();
                            
                            // Handle JSON fields
                            foreach ($filtered as $key => $value) {
                                if (in_array($key, $jsonColumns)) {
                                    $filtered[$key] = is_array($value) || is_object($value) ? json_encode($value) : $value;
                                }
                            }
                            // Fallback: ensure any remaining array/object values are JSON-encoded
                            foreach ($filtered as $key => $value) {
                                if (is_array($value) || is_object($value)) {
                                    $filtered[$key] = json_encode($value);
                                }
                            }
                            
                            Log::info('Filtered record', [
                                'index' => $index,
                                'original_keys' => array_keys($record),
                                'filtered_keys' => array_keys($filtered),
                                'filtered_data' => $filtered
                            ]);
                            
                            $insertData[] = $filtered;
                        }
                    } else {
                        // Single record
                        Log::info('Processing single record', ['data_keys' => array_keys($data)]);
                        $filtered = array_intersect_key($data, array_flip($columns));
                        
                        // Handle field mapping for auto-created tables
                        if ($receiver->auto_create_table && isset($filtered['id'])) {
                            $filtered['external_id'] = $filtered['id'];
                            unset($filtered['id']);
                        }
                        
                        $filtered['created_at'] = now();
                        $filtered['updated_at'] = now();
                        
                        // Handle JSON fields
                        foreach ($filtered as $key => $value) {
                            if (in_array($key, $jsonColumns)) {
                                $filtered[$key] = is_array($value) || is_object($value) ? json_encode($value) : $value;
                            }
                        }
                        // Fallback: ensure any remaining array/object values are JSON-encoded
                        foreach ($filtered as $key => $value) {
                            if (is_array($value) || is_object($value)) {
                                $filtered[$key] = json_encode($value);
                            }
                        }
                        
                        Log::info('Filtered single record', [
                            'original_keys' => array_keys($data),
                            'filtered_keys' => array_keys($filtered),
                            'filtered_data' => $filtered
                        ]);
                        
                        $insertData[] = $filtered;
                    }

                    // Insert data
                    if (!empty($insertData)) {
                        Log::info('Attempting to insert data', [
                            'table' => $receiver->target_table,
                            'records_count' => count($insertData),
                            'columns' => $columns,
                            'json_columns' => $jsonColumns,
                            'sample_data' => array_slice($insertData, 0, 1)
                        ]);
                        
                        try {
                            foreach ($insertData as $index => $row) {
                                Log::info('Inserting row', [
                                    'row_index' => $index,
                                    'row_data' => $row,
                                    'table' => $receiver->target_table
                                ]);
                                
                                // Coba insertGetId, fallback ke insert biasa jika gagal
                                try {
                                    $id = DB::table($receiver->target_table)->insertGetId($row);
                                    $insertedIds[] = $id;
                                    Log::info('Row inserted with ID', ['id' => $id, 'row_index' => $index]);
                                } catch (\Exception $e) {
                                    // Jika insertGetId gagal (tidak ada kolom id), lakukan insert biasa
                                    DB::table($receiver->target_table)->insert($row);
                                    Log::info('Row inserted without ID', ['row_index' => $index]);
                                }
                            }
                            $savedToTable = true;
                            Log::info('Data saved to table successfully', [
                                'table' => $receiver->target_table,
                                'records_inserted' => count($insertData),
                                'inserted_ids' => $insertedIds
                            ]);
                        } catch (\Exception $e) {
                            $saveError = $e->getMessage();
                            Log::error('Failed to insert data', [
                                'table' => $receiver->target_table,
                                'error' => $e->getMessage(),
                                'trace' => $e->getTraceAsString(),
                                'data_sample' => array_slice($insertData, 0, 1)
                            ]);
                            // Fallback: save raw payload into public_data if structured insert fails
                            try {
                                $fallbackSaved = $this->saveToPublicData($data, $request);
                                if ($fallbackSaved) {
                                    $savedToTable = true;
                                    $saveError = null;
                                    Log::info('POST/PUT data saved to public_data fallback');
                                }
                            } catch (\Exception $fallbackEx) {
                                Log::warning('Fallback save failed', ['error' => $fallbackEx->getMessage()]);
                            }
                        }
                    } else {
                        Log::warning('No data to insert', [
                            'table' => $receiver->target_table,
                            'original_data' => $data,
                            'columns' => $columns
                        ]);
                    }

                } catch (\Exception $e) {
                    $saveError = $e->getMessage();
                    Log::error('Failed to save data to table', [
                        'table' => $receiver->target_table,
                        'error' => $e->getMessage(),
                        'receiver_id' => $receiver->id
                    ]);
                }
            }

            // Log the request
            $this->logReceiveRequest($request, $receiver, $client, 200, 'Success', true, true, $savedToTable, $saveError);

            // Update receiver statistics
            $receiver->increment('total_received');
            $receiver->update(['last_received_at' => now()]);

            // Prepare response; ALWAYS echo back data regardless of authentication
            $response = [
                'success' => true,
                'message' => 'Data received successfully',
                'received_at' => now()->toISOString(),
                'data' => $data // Always include data in response
            ];

            // Attach storage status in debug mode (no data echo)
            $debugStorage = ($request->boolean('debug_storage') || $request->header('X-Debug-Storage') === 'true' || config('app.debug'));
            if ($debugStorage) {
                $response['saved_to_table'] = $savedToTable;
                if ($saveError) {
                    $response['save_error'] = $saveError;
                }
            }

            // Add info if database storage was skipped
            if ($skipDatabaseStorage) {
                $response['message'] = 'Data received successfully (database storage skipped)';
                $response['storage_skipped'] = true;
            }

            // Log warning internally but don't show in response
            if (!$savedToTable && $saveError && !$skipDatabaseStorage) {
                Log::warning('Data save failed', [
                    'receiver_id' => $receiver->id,
                    'error' => $saveError,
                    'data_size' => count($data)
                ]);
            }

            return response()->json($response, 200);

        } catch (\Exception $e) {
            // Log the error
            Log::error('Data receiver processing error', [
                'receiver_id' => $receiver->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            $this->logReceiveRequest($request, $receiver, $client, 500, 'Error: ' . $e->getMessage(), true, false, false, $e->getMessage());

            // Return simple error response for GET requests, detailed for others
            if ($request->method() === 'GET') {
                return response()->json([
                    'sukses' => false,
                    'error' => 'Processing failed'
                ], 500);
            }

            // Return detailed error response for other methods
            return response()->json([
                'success' => false,
                'error' => 'Processing failed',
                'message' => 'An error occurred while processing your data',
                'exception' => config('app.debug') ? $e->getMessage() : 'Internal server error',
                'timestamp' => now()->toISOString()
            ], 500);
        }
    }

    /**
     * Fallback persistence: store raw payload into public_data table
     */
    protected function saveToPublicData($data, Request $request): bool
    {
        try {
            // Ensure table exists or create a minimal one
            if (!Schema::hasTable('public_data')) {
                Schema::create('public_data', function ($table) {
                    $table->id();
                    $table->string('name')->nullable();
                    $table->string('email')->nullable();
                    $table->text('message')->nullable();
                    $table->string('ip_address')->nullable();
                    $table->json('raw_data')->nullable();
                    $table->timestamps();
                });
            }

            $payload = is_string($data) ? $data : json_encode($data);
            DB::table('public_data')->insert([
                'name' => $request->input('name'),
                'email' => $request->input('email'),
                'message' => $request->input('message'),
                'ip_address' => $request->ip(),
                'raw_data' => $payload,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
            return true;
        } catch (\Exception $e) {
            Log::error('saveToPublicData failed', ['error' => $e->getMessage()]);
            return false;
        }
    }

    /**
     * Log receive request
     */
    protected function logReceiveRequest(Request $request, ApiReceiver $receiver, $client = null, int $statusCode, string $message, bool $authSuccess, bool $validationSuccess, bool $savedToTable, ?string $saveError = null)
    {
        $startTime = defined('LARAVEL_START') ? LARAVEL_START : microtime(true);
        $responseTime = (microtime(true) - $startTime) * 1000; // Convert to milliseconds
        
        $body = $request->method() === 'GET' ? json_encode($request->query()) : $request->getContent();
        $payloadSize = strlen($body);
        
        ApiReceiveLog::create([
            'receiver_id' => $receiver->id,
            'method' => $request->method(),
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'headers' => $request->headers->all(),
            'body' => $body,
            'query_params' => $request->query(),
            'status_code' => $statusCode,
            'response_message' => $message,
            'response_time' => $responseTime,
            'payload_size' => $payloadSize,
            'auth_success' => $authSuccess,
            'validation_success' => $validationSuccess,
            'saved_to_table' => $savedToTable,
            'save_error' => $saveError,
            'reference_id' => $client ? $client->client_id : null
        ]);
    }

    /**
     * Return error response
     */
    protected function errorResponse(string $message, int $code, Request $request, ?ApiReceiver $receiver = null, bool $authSuccess = false)
    {
        if ($receiver) {
            $this->logReceiveRequest($request, $receiver, null, $code, $message, $authSuccess, false, false, $message);
        }

        // Return simple error response for GET requests, detailed for others
        if ($request->method() === 'GET') {
            return response()->json([
                'sukses' => false,
                'error' => $message
            ], $code);
        }

        // Return detailed error response for other methods
        return response()->json([
            'success' => false,
            'error' => $message,
            'code' => $code,
            'received_at' => now()->toISOString()
        ], $code);
    }

    /**
     * Convert array to XML
     */
    protected function arrayToXml(array $data, string $rootElement = 'response')
    {
        $xml = new \SimpleXMLElement("<?xml version=\"1.0\" encoding=\"UTF-8\"?><{$rootElement}></{$rootElement}>");
        $this->arrayToXmlRecursive($data, $xml);
        return $xml->asXML();
    }

    /**
     * Recursively convert array to XML
     */
    protected function arrayToXmlRecursive(array $data, \SimpleXMLElement $xml)
    {
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                $child = $xml->addChild(is_numeric($key) ? 'item' : $key);
                $this->arrayToXmlRecursive($value, $child);
            } else {
                $xml->addChild(is_numeric($key) ? 'item' : $key, htmlspecialchars((string)$value));
            }
        }
    }

    public function logs(ApiReceiver $apiReceiver)
    {
        $logs = $apiReceiver->receiveLogs()->latest()->paginate(20);
        return view('api-receivers.logs', compact('apiReceiver', 'logs'));
    }

    public function previewPdf(ApiReceiver $apiReceiver)
    {
        try {
            // Load additional data for the PDF
            $apiReceiver->load(['receiveLogs' => function($query) {
                $query->latest()->limit(5);
            }]);
            
            // Get statistics with null checks
            $receiveLogs = $apiReceiver->receiveLogs();
            $stats = [
                'total_requests' => $receiveLogs->count(),
                'successful_requests' => $receiveLogs->where('status_code', '>=', 200)->where('status_code', '<', 300)->count(),
                'error_requests' => $receiveLogs->where('status_code', '>=', 400)->count(),
                'avg_response_time' => $receiveLogs->avg('response_time') ?? 0,
                'last_activity' => $receiveLogs->latest('created_at')->first()
            ];
            
            // Get target table structure for sample data with null checks
            $tableStructure = null;
            $sampleData = null;
            
            if ($apiReceiver->target_table) {
                $tableStructure = $this->getTableStructure($apiReceiver->target_table);
                $sampleData = $this->generateSampleData($apiReceiver->target_table, $tableStructure);
            }
            
            return view('api-receivers.preview-pdf', compact('apiReceiver', 'stats', 'tableStructure', 'sampleData'));
            
        } catch (\Exception $e) {
            Log::error('PDF preview failed for receiver ' . $apiReceiver->id, [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()->with('error', 'Failed to generate PDF preview: ' . $e->getMessage());
        }
    }

    public function exportPdf(ApiReceiver $apiReceiver)
    {
        try {
            // Load additional data for the PDF
            $apiReceiver->load(['receiveLogs' => function($query) {
                $query->latest()->limit(5);
            }]);
            
            // Get statistics
            $stats = [
                'total_requests' => $apiReceiver->receiveLogs()->count(),
                'successful_requests' => $apiReceiver->receiveLogs()->where('status_code', '>=', 200)->where('status_code', '<', 300)->count(),
                'error_requests' => $apiReceiver->receiveLogs()->where('status_code', '>=', 400)->count(),
                'avg_response_time' => $apiReceiver->receiveLogs()->avg('response_time'),
                'last_activity' => $apiReceiver->receiveLogs()->latest('created_at')->first()
            ];
            
            // Get target table structure for sample data
            $tableStructure = $this->getTableStructure($apiReceiver->target_table);
            $sampleData = $this->generateSampleData($apiReceiver->target_table, $tableStructure);
            
            $pdf = Pdf::loadView('api-receivers.show-pdf', compact('apiReceiver', 'stats', 'tableStructure', 'sampleData'));
            
            // Set PDF options for better quality
            $pdf->setPaper('a4', 'portrait');
            $pdf->setOptions([
                'isHtml5ParserEnabled' => true,
                'isRemoteEnabled' => true,
                'defaultFont' => 'DejaVu Sans'
            ]);
            
            return $pdf->download('api-receiver-' . $apiReceiver->id . '-' . $apiReceiver->name . '.pdf');
            
        } catch (\Exception $e) {
            Log::error('PDF export failed for receiver ' . $apiReceiver->id, [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()->with('error', 'Failed to generate PDF: ' . $e->getMessage());
        }
    }

    /**
     * Get table structure for target table
     */
    protected function getTableStructure($tableName)
    {
        try {
            if (!Schema::hasTable($tableName)) {
                return null;
            }
            
            $columns = Schema::getColumnListing($tableName);
            $structure = [];
            
            foreach ($columns as $column) {
                $type = Schema::getColumnType($tableName, $column);
                $structure[$column] = $type;
            }
            
            return $structure;
        } catch (\Exception $e) {
            Log::error("Error getting table structure for {$tableName}: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Generate sample data based on table structure
     */
    protected function generateSampleData($tableName, $structure)
    {
        if (!$structure) {
            return null;
        }
        
        $sampleData = [];
        
        foreach ($structure as $column => $type) {
            $columnLower = strtolower($column);
            
            switch ($type) {
                case 'string':
                case 'varchar':
                case 'text':
                    if (str_contains($columnLower, 'name')) {
                        $sampleData[$column] = 'John Doe';
                    } elseif (str_contains($columnLower, 'email')) {
                        $sampleData[$column] = 'john.doe@example.com';
                    } elseif (str_contains($columnLower, 'phone') || str_contains($columnLower, 'tel')) {
                        $sampleData[$column] = '+62812345678';
                    } elseif (str_contains($columnLower, 'address')) {
                        $sampleData[$column] = 'Jl. Contoh No. 123, Jakarta';
                    } elseif (str_contains($columnLower, 'description')) {
                        $sampleData[$column] = 'Sample description for this field';
                    } elseif (str_contains($columnLower, 'title')) {
                        $sampleData[$column] = 'Sample Title';
                    } elseif (str_contains($columnLower, 'code')) {
                        $sampleData[$column] = 'SAMPLE001';
                    } elseif (str_contains($columnLower, 'status')) {
                        $sampleData[$column] = 'active';
                    } elseif (str_contains($columnLower, 'type')) {
                        $sampleData[$column] = 'default';
                    } elseif (str_contains($columnLower, 'category')) {
                        $sampleData[$column] = 'general';
                    } elseif (str_contains($columnLower, 'url') || str_contains($columnLower, 'link')) {
                        $sampleData[$column] = 'https://example.com';
                    } elseif (str_contains($columnLower, 'note') || str_contains($columnLower, 'comment')) {
                        $sampleData[$column] = 'Sample note or comment';
                    } else {
                        $sampleData[$column] = 'sample_value';
                    }
                    break;
                    
                case 'integer':
                case 'bigint':
                case 'int':
                    if (str_contains($columnLower, 'id')) {
                        $sampleData[$column] = 12345;
                    } elseif (str_contains($columnLower, 'amount') || str_contains($columnLower, 'price') || str_contains($columnLower, 'cost')) {
                        $sampleData[$column] = 150000;
                    } elseif (str_contains($columnLower, 'quantity') || str_contains($columnLower, 'qty')) {
                        $sampleData[$column] = 2;
                    } elseif (str_contains($columnLower, 'age')) {
                        $sampleData[$column] = 25;
                    } elseif (str_contains($columnLower, 'year')) {
                        $sampleData[$column] = 2024;
                    } elseif (str_contains($columnLower, 'month')) {
                        $sampleData[$column] = 6;
                    } elseif (str_contains($columnLower, 'day')) {
                        $sampleData[$column] = 15;
                    } elseif (str_contains($columnLower, 'count') || str_contains($columnLower, 'total')) {
                        $sampleData[$column] = 100;
                    } else {
                        $sampleData[$column] = 100;
                    }
                    break;
                    
                case 'decimal':
                case 'float':
                case 'double':
                    if (str_contains($columnLower, 'price') || str_contains($columnLower, 'amount') || str_contains($columnLower, 'cost')) {
                        $sampleData[$column] = 99.99;
                    } elseif (str_contains($columnLower, 'rate') || str_contains($columnLower, 'percentage')) {
                        $sampleData[$column] = 85.5;
                    } elseif (str_contains($columnLower, 'score') || str_contains($columnLower, 'rating')) {
                        $sampleData[$column] = 4.5;
                    } else {
                        $sampleData[$column] = 99.99;
                    }
                    break;
                    
                case 'boolean':
                    if (str_contains($columnLower, 'is_') || str_contains($columnLower, 'has_') || str_contains($columnLower, 'can_')) {
                        $sampleData[$column] = true;
                    } elseif (str_contains($columnLower, 'active') || str_contains($columnLower, 'enabled')) {
                        $sampleData[$column] = true;
                    } else {
                        $sampleData[$column] = true;
                    }
                    break;
                    
                case 'datetime':
                case 'timestamp':
                    if (str_contains($columnLower, 'created') || str_contains($columnLower, 'added')) {
                        $sampleData[$column] = now()->subDays(5)->toISOString();
                    } elseif (str_contains($columnLower, 'updated') || str_contains($columnLower, 'modified')) {
                        $sampleData[$column] = now()->subHours(2)->toISOString();
                    } elseif (str_contains($columnLower, 'expired') || str_contains($columnLower, 'expiry')) {
                        $sampleData[$column] = now()->addDays(30)->toISOString();
                    } else {
                        $sampleData[$column] = now()->toISOString();
                    }
                    break;
                    
                case 'json':
                    if (str_contains($columnLower, 'meta') || str_contains($columnLower, 'data')) {
                        $sampleData[$column] = ['key' => 'value', 'nested' => ['data' => 'example']];
                    } elseif (str_contains($columnLower, 'settings') || str_contains($columnLower, 'config')) {
                        $sampleData[$column] = ['enabled' => true, 'options' => ['option1', 'option2']];
                    } else {
                        $sampleData[$column] = ['key' => 'value', 'nested' => ['data' => 'example']];
                    }
                    break;
                    
                default:
                    $sampleData[$column] = 'sample_value';
                    break;
            }
        }
        
        return $sampleData;
    }
} 