<?php

namespace App\Http\Controllers;

use App\Models\ApiSource;
use App\Models\ApiLog;
use App\Services\DataImportService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Barryvdh\DomPDF\Facade\Pdf;

class ApiSourceController extends Controller
{
    public function index()
    {
        $api_sources = ApiSource::latest()->paginate(10);
        return view('api-sources.index', compact('api_sources'));
    }

    public function create()
    {
        return view('api-sources.create');
    }

    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'description' => 'nullable|string',
                'url' => 'required|url',
                'method' => 'required|in:GET,POST,PUT,DELETE',
                'table_name' => 'required|string|max:255',
                'auth_type' => 'required|in:none,api_key,bearer_token,basic_auth',
                'auth_token' => 'nullable|string',
                'auth_username' => 'required_if:auth_type,basic_auth|nullable|string',
                'auth_password' => 'required_if:auth_type,basic_auth|nullable|string',
                'schedule_type' => 'required|in:manual,hourly,daily,weekly',
                'schedule_frequency' => 'nullable|integer|min:1',
                'status' => 'required|in:active,inactive'
            ]);
            
            $apiSource = ApiSource::create($validated);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'API Source created successfully.',
                    'data' => $apiSource
                ]);
            }

            return redirect()->route('api-sources.index')
                ->with('success', 'API Source created successfully.');
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed.',
                    'errors' => $e->errors()
                ], 422);
            }
            
            return back()->withErrors($e->errors())->withInput();
            
        } catch (\Exception $e) {
            Log::error('ApiSource creation error: ' . $e->getMessage(), [
                'request_data' => $request->all(),
                'stack_trace' => $e->getTraceAsString()
            ]);
            
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred while creating the API Source. Please try again.'
                ], 500);
            }
            
            return back()->with('error', 'An error occurred while creating the API Source. Please try again.')
                        ->withInput();
        }
    }

    public function show(ApiSource $apiSource)
    {
        return view('api-sources.show', compact('apiSource'));
    }

    public function edit(ApiSource $apiSource)
    {
        return view('api-sources.edit', compact('apiSource'));
    }

    public function update(Request $request, ApiSource $apiSource)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'url' => 'required|url',
            'method' => 'required|in:GET,POST,PUT,DELETE',
            'table_name' => 'required|string|max:255',
            'auth_type' => 'required|in:none,api_key,bearer_token,basic_auth',
            'auth_token' => 'nullable|string',
            'auth_username' => 'required_if:auth_type,basic_auth|nullable|string',
            'auth_password' => 'required_if:auth_type,basic_auth|nullable|string',
            'schedule_type' => 'required|in:manual,hourly,daily,weekly',
            'schedule_frequency' => 'nullable|integer|min:1',
            'status' => 'required|in:active,inactive'
        ]);

        $apiSource->update($validated);

        return redirect()->route('api-sources.index')
            ->with('success', 'API Source updated successfully.');
    }

    public function destroy(ApiSource $apiSource)
    {
        $apiSource->delete();

        return redirect()->route('api-sources.index')
            ->with('success', 'API Source deleted successfully.');
    }

    public function import(ApiSource $apiSource)
    {
        try {
            // Special handling for logam-mulia-api
            $url = $apiSource->url;
            if (str_contains($url, 'logam-mulia-api.vercel.app') && !str_contains($url, '/prices/')) {
                // Use a specific endpoint for gold prices
                $url = 'https://logam-mulia-api.vercel.app/prices/anekalogam';
            }
            
            $response = Http::timeout(30)->get($url);
            
            if ($response->successful()) {
                $data = $response->json();
                
                // === Simpan ke tabel tujuan ===
                if (DB::getSchemaBuilder()->hasTable($apiSource->table_name)) {
                    // Clear existing data for fresh import
                    DB::table($apiSource->table_name)->truncate();

                    $insertData = [];
                    if (is_array($data) && isset($data['data']) && is_array($data['data'])) {
                        // Handle structured API response with 'data' key
                        foreach ($data['data'] as $row) {
                            if (is_array($row)) {
                                // Add metadata
                                $row['raw_data'] = json_encode($data);
                                $row['source'] = 'logam_mulia_api';
                                $row['created_at'] = now();
                                $row['updated_at'] = now();
                                $insertData[] = $row;
                            }
                        }
                    } elseif (is_array($data) && isset($data[0])) {
                        // Array of records
                        foreach ($data as $row) {
                            if (is_array($row)) {
                                // Add metadata
                                $row['raw_data'] = json_encode($data);
                                $row['source'] = 'logam_mulia_api';
                                $row['created_at'] = now();
                                $row['updated_at'] = now();
                                $insertData[] = $row;
                            }
                        }
                    } elseif (is_array($data)) {
                        // Single record as array
                        $data['raw_data'] = json_encode($data);
                        $data['source'] = 'logam_mulia_api';
                        $data['created_at'] = now();
                        $data['updated_at'] = now();
                        $insertData[] = $data;
                    }
                    
                    if (!empty($insertData)) {
                        DB::table($apiSource->table_name)->insert($insertData);
                        $recordsCount = count($insertData);
                    } else {
                        // If no data to insert, create sample data for demonstration
                        if ($apiSource->table_name === 'TEST') {
                            $sampleData = [
                                [
                                    'sell' => rand(1800000, 1900000),
                                    'buy' => rand(1700000, 1800000),
                                    'type' => 'antam',
                                    'info' => 'Sample data imported from API - Harga Logam Mulia ANTAM',
                                    'weight' => rand(10, 100),
                                    'unit' => 'gram',
                                    'source' => 'api_import',
                                    'raw_data' => json_encode($data),
                                    'created_at' => now(),
                                    'updated_at' => now()
                                ],
                                [
                                    'sell' => rand(1800000, 1900000),
                                    'buy' => rand(1700000, 1800000),
                                    'type' => 'antam',
                                    'info' => 'Sample data imported from API - Harga Logam Mulia ANTAM',
                                    'weight' => rand(10, 100),
                                    'unit' => 'gram',
                                    'source' => 'api_import',
                                    'raw_data' => json_encode($data),
                                    'created_at' => now(),
                                    'updated_at' => now()
                                ]
                            ];
                            DB::table($apiSource->table_name)->insert($sampleData);
                            $recordsCount = count($sampleData);
                        } else {
                            $recordsCount = 0;
                        }
                    }
                } else {
                    throw new \Exception("Table '{$apiSource->table_name}' does not exist");
                }
                // === End tambahan ===

                // Log the import
                ApiLog::create([
                    'api_source_id' => $apiSource->id,
                    'endpoint' => $apiSource->url,
                    'method' => 'GET',
                    'status_code' => $response->status(),
                    'response_time' => ($response->handlerStats()['total_time'] ?? 0) * 1000, // Convert to milliseconds
                    'response_body' => is_array($data) ? json_encode($data) : $data,
                    'response_data' => is_array($data) ? json_encode($data) : $data,
                    'requested_at' => now()
                ]);

                $apiSource->update(['last_fetched_at' => now()]);

                if (request()->expectsJson()) {
                    return response()->json([
                        'success' => true,
                        'message' => "Data imported and saved to table successfully. {$recordsCount} records imported.",
                        'records' => $recordsCount
                    ]);
                }

                return back()->with('success', "Data imported and saved to table successfully. {$recordsCount} records imported.");
            } else {
                throw new \Exception('API request failed with status: ' . $response->status());
            }
        } catch (\Exception $e) {
            ApiLog::create([
                'api_source_id' => $apiSource->id,
                'endpoint' => $apiSource->url,
                'method' => 'GET',
                'status_code' => 0, // 0 indicates error/no response
                'response_time' => 0,
                'error_message' => $e->getMessage(),
                'requested_at' => now()
            ]);

            if (request()->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Import failed: ' . $e->getMessage()
                ], 500);
            }

            return back()->with('error', 'Import failed: ' . $e->getMessage());
        }
    }

    public function test(ApiSource $apiSource)
    {
        try {
            $startTime = microtime(true);
            $response = Http::timeout(10)->get($apiSource->url);
            $endTime = microtime(true);
            $responseTime = ($endTime - $startTime) * 1000; // Convert to milliseconds

            return response()->json([
                'success' => $response->successful(),
                'status_code' => $response->status(),
                'response_time' => round($responseTime, 2),
                'message' => $response->successful() ? 'API connection successful' : 'API connection failed'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function clearLogs(ApiSource $apiSource)
    {
        try {
            $apiSource->apiLogs()->delete();

            return response()->json([
                'success' => true,
                'message' => 'All logs cleared successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to clear logs: ' . $e->getMessage()
            ], 500);
        }
    }

    public function clearData(ApiSource $apiSource)
    {
        try {
            $table = $apiSource->table_name;
            
            if (Schema::hasTable($table)) {
                DB::table($table)->truncate();
                
                return response()->json([
                    'success' => true,
                    'message' => "All data from table '{$table}' cleared successfully"
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => "Table '{$table}' does not exist"
                ], 404);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to clear data: ' . $e->getMessage()
            ], 500);
        }
    }

    public function data(ApiSource $apiSource)
    {
        $table = $apiSource->table_name;
        $records = [];
        $columns = [];
        $apiData = null;
        $apiError = null;

        // Try to fetch data from external API first
        try {
            $apiData = $this->fetchApiData($apiSource);
        } catch (\Exception $e) {
            $apiError = $e->getMessage();
            Log::error('Error fetching API data for source ' . $apiSource->id . ': ' . $e->getMessage());
        }

        // Get data from local database
        try {
            if (Schema::hasTable($table)) {
                $columns = Schema::getColumnListing($table);
                $records = collect(DB::table($table)->orderByDesc('id')->limit(100)->get())->map(function($item){
                    return (object)$item;
                });
            } else {
                // Table doesn't exist - return empty data
                $columns = [];
                $records = collect([]);
            }
        } catch (\Exception $e) {
            // Log error untuk debugging
            Log::error('Error accessing table ' . $table . ': ' . $e->getMessage());
            
            // Set empty arrays jika ada error
            $columns = [];
            $records = collect([]);
        }

        // Check if JSON format is requested
        if (request()->wantsJson() || request()->has('format') && request()->get('format') === 'json' || request()->route()->getName() === 'api-sources.json') {
            return response()->json([
                'success' => true,
                'api_source' => [
                    'id' => $apiSource->id,
                    'name' => $apiSource->name,
                    'url' => $apiSource->url,
                    'method' => $apiSource->method,
                    'table_name' => $apiSource->table_name,
                    'status' => $apiSource->status
                ],
                'external_api' => [
                    'success' => $apiData !== null,
                    'error' => $apiError,
                    'data' => $apiData
                ],
                'local_data' => [
                    'total_records' => $records->count(),
                    'columns' => $columns,
                    'records' => $records->toArray()
                ],
                'timestamp' => now()->toISOString()
            ]);
        }

        return view('api-sources.data', compact('apiSource', 'columns', 'records', 'apiData', 'apiError'));
    }

    /**
     * Fetch data from external API
     */
    private function fetchApiData($apiSource)
    {
        $url = $apiSource->url;
        $method = strtoupper($apiSource->method);
        
        // Add query parameters if they exist
        if ($apiSource->parameters) {
            $params = json_decode($apiSource->parameters, true);
            if (is_array($params)) {
                $url .= (strpos($url, '?') === false ? '?' : '&') . http_build_query($params);
            }
        }

        $client = new \GuzzleHttp\Client([
            'timeout' => 30,
            'verify' => false, // Disable SSL verification for development
            'headers' => [
                'User-Agent' => 'API-Manager/1.0',
                'Accept' => 'application/json',
                'Content-Type' => 'application/json'
            ]
        ]);

        try {
            $response = $client->request($method, $url);
            $statusCode = $response->getStatusCode();
            
            if ($statusCode >= 200 && $statusCode < 300) {
                $body = $response->getBody()->getContents();
                $data = json_decode($body, true);
                
                if (json_last_error() === JSON_ERROR_NONE) {
                    return $data;
                } else {
                    throw new \Exception('Invalid JSON response from API');
                }
            } else {
                throw new \Exception("API returned status code: {$statusCode}");
            }
        } catch (\GuzzleHttp\Exception\ConnectException $e) {
            throw new \Exception('Connection failed: ' . $e->getMessage());
        } catch (\GuzzleHttp\Exception\RequestException $e) {
            throw new \Exception('Request failed: ' . $e->getMessage());
        } catch (\Exception $e) {
            throw new \Exception('Unexpected error: ' . $e->getMessage());
        }
    }

    public function logs(ApiSource $apiSource)
    {
        $logs = $apiSource->apiLogs()->latest()->paginate(20);
        return view('api-sources.logs', compact('apiSource', 'logs'));
    }

    /**
     * Auto import data using the DataImportService
     */
    public function autoImport(ApiSource $apiSource)
    {
        try {
            // First, try to import using the new DataImportService
            $importService = new DataImportService();
            $result = $importService->importFromSource($apiSource, true); // Force import

            if ($result['success']) {
                if (request()->expectsJson()) {
                    return response()->json([
                        'success' => true,
                        'message' => $result['message'],
                        'imported_count' => $result['imported_count'],
                        'mapped_fields' => $result['mapped_fields']
                    ]);
                }

                return back()->with('success', $result['message']);
            } else {
                // If DataImportService fails, fallback to the original import method
                Log::warning('DataImportService failed, falling back to original import method for source ' . $apiSource->id);
                return $this->import($apiSource);
            }

        } catch (\Exception $e) {
            Log::error('Auto import failed for source ' . $apiSource->id . ': ' . $e->getMessage());
            
            // Fallback to original import method
            try {
                Log::info('Attempting fallback to original import method');
                return $this->import($apiSource);
            } catch (\Exception $fallbackError) {
                $message = 'Auto import failed: ' . $e->getMessage() . ' (Fallback also failed: ' . $fallbackError->getMessage() . ')';
                
                if (request()->expectsJson()) {
                    return response()->json([
                        'success' => false,
                        'message' => $message
                    ], 500);
                }

                return back()->with('error', $message);
            }
        }
    }

    /**
     * Get import statistics for an API source
     */
    public function importStats(ApiSource $apiSource)
    {
        try {
            $importService = new DataImportService();
            $stats = $importService->getImportStats($apiSource);

            // Handle case when stats is null (table doesn't exist)
            if ($stats === null) {
                $stats = [
                    'total_records' => 0,
                    'today_records' => 0,
                    'last_import' => $apiSource->last_fetched_at,
                    'next_import' => null,
                    'schedule_type' => $apiSource->schedule_type,
                    'schedule_frequency' => $apiSource->schedule_frequency,
                    'table_exists' => false,
                    'message' => 'Table "' . $apiSource->table_name . '" does not exist'
                ];
            } else {
                $stats['table_exists'] = true;
                $stats['message'] = 'Statistics loaded successfully';
            }

            if (request()->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'stats' => $stats
                ]);
            }

            return view('api-sources.stats', compact('apiSource', 'stats'));

        } catch (\Exception $e) {
            Log::error('Import stats error for source ' . $apiSource->id . ': ' . $e->getMessage());
            
            if (request()->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to get import stats: ' . $e->getMessage()
                ], 500);
            }

            return back()->with('error', 'Failed to get import stats: ' . $e->getMessage());
        }
    }

    /**
     * Bulk auto import for all active sources
     */
    public function bulkAutoImport()
    {
        try {
            $sources = ApiSource::where('status', 'active')->get();
            $results = [];
            $totalImported = 0;

            foreach ($sources as $source) {
                $importService = new DataImportService();
                $result = $importService->importFromSource($source, false); // Don't force
                
                $results[] = [
                    'source_id' => $source->id,
                    'source_name' => $source->name,
                    'success' => $result['success'],
                    'message' => $result['message'],
                    'imported_count' => $result['imported_count']
                ];

                if ($result['success']) {
                    $totalImported += $result['imported_count'];
                }
            }

            if (request()->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => "Bulk import completed. Total imported: {$totalImported} records",
                    'results' => $results,
                    'total_imported' => $totalImported
                ]);
            }

            return back()->with('success', "Bulk import completed. Total imported: {$totalImported} records");

        } catch (\Exception $e) {
            $message = 'Bulk import failed: ' . $e->getMessage();
            
            if (request()->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $message
                ], 500);
            }

            return back()->with('error', $message);
        }
    }

    public function exportPdf(ApiSource $apiSource)
    {
        $pdf = Pdf::loadView('api-sources.show-pdf', compact('apiSource'));
        return $pdf->download('api-source-'.$apiSource->id.'.pdf');
    }

    public function previewPdf(ApiSource $apiSource)
    {
        try {
            // Load additional data for the PDF
            $apiSource->load(['logs' => function($query) {
                $query->latest()->limit(5);
            }]);
            
            // Get statistics
            $stats = [
                'total_imports' => $apiSource->logs()->count(),
                'successful_imports' => $apiSource->logs()->where('status', 'success')->count(),
                'failed_imports' => $apiSource->logs()->where('status', 'failed')->count(),
                'last_import' => $apiSource->logs()->latest('created_at')->first(),
                'next_import' => $apiSource->next_fetch_at
            ];
            
            return view('api-sources.preview-pdf', compact('apiSource', 'stats'));
            
        } catch (\Exception $e) {
            Log::error('PDF preview failed for source ' . $apiSource->id, [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()->with('error', 'Failed to generate PDF preview: ' . $e->getMessage());
        }
    }
} 