<?php

namespace App\Http\Controllers;

use App\Models\ApiTarget;
use App\Models\ApiSendLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ApiTargetController extends Controller
{
    public function index()
    {
        $api_targets = ApiTarget::latest()->paginate(10);
        return view('api-targets.index', compact('api_targets'));
    }

    public function create()
    {
        try {
            // Get all database tables
            $tables = collect(DB::select('SHOW TABLES'))
                ->map(function ($table) {
                    $tableName = array_values((array) $table)[0];
                    // Filter out Laravel system tables if needed
                    return $tableName;
                })
                ->filter(function ($tableName) {
                    // Exclude migration and cache tables
                    return !in_array($tableName, ['migrations', 'cache', 'sessions', 'cache_locks']);
                })
                ->values()
                ->toArray();
        } catch (\Exception $e) {
            // If database query fails, provide a fallback with common table names
            $tables = ['users', 'api_sources', 'api_targets', 'api_receivers', 'api_transceivers'];
        }

        return view('api-targets.create', compact('tables'));
    }

    public function store(Request $request)
    {
        // Debug logging
        Log::info('ApiTarget store method called', [
            'method' => $request->method(),
            'url' => $request->url(),
            'data' => $request->all(),
            'headers' => $request->headers->all()
        ]);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'base_url' => 'required|url',
            'endpoint' => 'required|string',
            'method' => 'required|in:GET,POST,PUT,DELETE',
            'source_table' => 'required|string|max:255',
            'auth_type' => 'required|in:none,api_key,bearer_token,basic_auth',
            'auth_token' => 'nullable|string',
            'auth_username' => 'nullable|string',
            'auth_password' => 'nullable|string',
            'schedule_type' => 'required|in:manual,hourly,daily,weekly',
            'schedule_frequency' => 'nullable|integer|min:1',
            'is_active' => 'boolean'
        ]);

        $validated['created_by'] = auth()->id();
        $validated['is_active'] = $request->has('is_active');
        
        ApiTarget::create($validated);

        return redirect()->route('api-targets.index')
            ->with('success', 'API Target created successfully.');
    }

    public function show(ApiTarget $apiTarget)
    {
        return view('api-targets.show', compact('apiTarget'));
    }

    public function edit(ApiTarget $apiTarget)
    {
        // Get all database tables
        $tables = collect(DB::select('SHOW TABLES'))
            ->map(function ($table) {
                $tableName = array_values((array) $table)[0];
                // Filter out Laravel system tables if needed
                return $tableName;
            })
            ->filter(function ($tableName) {
                // Exclude migration and cache tables
                return !in_array($tableName, ['migrations', 'cache', 'sessions', 'cache_locks']);
            })
            ->values()
            ->toArray();

        return view('api-targets.edit', compact('apiTarget', 'tables'));
    }

    public function update(Request $request, ApiTarget $apiTarget)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'base_url' => 'required|url',
            'endpoint' => 'required|string',
            'method' => 'required|in:GET,POST,PUT,DELETE',
            'source_table' => 'required|string|max:255',
            'auth_type' => 'required|in:none,api_key,bearer_token,basic_auth',
            'auth_token' => 'nullable|string',
            'auth_username' => 'nullable|string',
            'auth_password' => 'nullable|string',
            'schedule_type' => 'required|in:manual,hourly,daily,weekly',
            'schedule_frequency' => 'nullable|integer|min:1',
            'is_active' => 'boolean'
        ]);

        $validated['is_active'] = $request->has('is_active');
        
        $apiTarget->update($validated);

        return redirect()->route('api-targets.index')
            ->with('success', 'API Target updated successfully.');
    }

    public function destroy(ApiTarget $apiTarget)
    {
        Log::info('Destroy method called for API Target', [
            'id' => $apiTarget->id,
            'name' => $apiTarget->name,
            'method' => request()->method(),
            'url' => request()->url(),
            'is_ajax' => request()->ajax(),
            'wants_json' => request()->wantsJson()
        ]);

        try {
            // Check if there are any related records that might prevent deletion
            $relatedLogs = $apiTarget->sendLogs()->count();
            Log::info('Related logs count', ['count' => $relatedLogs]);

            $apiTarget->delete();
            
            Log::info('API Target deleted successfully', [
                'id' => $apiTarget->id,
                'name' => $apiTarget->name
            ]);

            // Return JSON response for AJAX requests
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'API Target deleted successfully.',
                    'id' => $apiTarget->id,
                    'name' => $apiTarget->name
                ]);
            }

            return redirect()->route('api-targets.index')
                ->with('success', 'API Target deleted successfully.');
        } catch (\Exception $e) {
            Log::error('Error deleting API Target', [
                'id' => $apiTarget->id,
                'name' => $apiTarget->name,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            // Return JSON response for AJAX requests
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error deleting API Target: ' . $e->getMessage(),
                    'id' => $apiTarget->id,
                    'name' => $apiTarget->name
                ], 500);
            }

            return redirect()->route('api-targets.index')
                ->with('error', 'Error deleting API Target: ' . $e->getMessage());
        }
    }

    public function send(ApiTarget $apiTarget)
    {
        try {
            // Get data from source table
            $data = $this->getDataFromSourceTable($apiTarget->source_table);
            
            if (empty($data)) {
                return back()->with('warning', 'No data found in source table: ' . $apiTarget->source_table);
            }
            
            // Prepare headers based on auth type
            $headers = $this->prepareHeaders($apiTarget);
            
            // Send request
            $response = Http::withHeaders($headers)
                ->timeout(30)
                ->{strtolower($apiTarget->method)}($apiTarget->url, $data);
            
            if ($response->successful()) {
                ApiSendLog::create([
                    'api_target_id' => $apiTarget->id,
                    'success' => true,
                    'status_code' => $response->status(),
                    'response_time' => $response->handlerStats()['total_time'] ?? 0,
                    'payload' => $data,
                    'records_sent' => is_array($data) ? count($data) : 1,
                    'sent_at' => now()
                ]);

                $apiTarget->update(['last_sent_at' => now()]);

                return back()->with('success', 'Data sent successfully. Response: ' . $response->body());
            } else {
                throw new \Exception('API request failed with status: ' . $response->status() . ' - ' . $response->body());
            }
        } catch (\Exception $e) {
            ApiSendLog::create([
                'api_target_id' => $apiTarget->id,
                'success' => false,
                'error_message' => $e->getMessage(),
                'payload' => $data ?? [],
                'sent_at' => now()
            ]);

            return back()->with('error', 'Send failed: ' . $e->getMessage());
        }
    }

    /**
     * Get data from source table
     */
    protected function getDataFromSourceTable($sourceTable)
    {
        try {
            if (!DB::getSchemaBuilder()->hasTable($sourceTable)) {
                throw new \Exception("Table '{$sourceTable}' does not exist");
            }
            
            // Get all data from the table
            $data = DB::table($sourceTable)->get();
            
            // Convert to array and handle timestamps
            return $data->map(function ($item) {
                $array = (array) $item;
                // Convert timestamps to ISO format
                if (isset($array['created_at'])) {
                    $array['created_at'] = \Carbon\Carbon::parse($array['created_at'])->toISOString();
                }
                if (isset($array['updated_at'])) {
                    $array['updated_at'] = \Carbon\Carbon::parse($array['updated_at'])->toISOString();
                }
                return $array;
            })->toArray();
            
        } catch (\Exception $e) {
            Log::error("Error fetching data from table {$sourceTable}: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Prepare headers based on authentication type
     */
    protected function prepareHeaders($apiTarget)
    {
        $headers = [
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
            'User-Agent' => 'API8-Hubnet-Integration/1.0'
        ];

        switch ($apiTarget->auth_type) {
            case 'basic_auth':
                if ($apiTarget->auth_username && $apiTarget->auth_password) {
                    $headers['Authorization'] = 'Basic ' . base64_encode($apiTarget->auth_username . ':' . $apiTarget->auth_password);
                }
                break;
            case 'bearer_token':
                if ($apiTarget->auth_token) {
                    $headers['Authorization'] = 'Bearer ' . $apiTarget->auth_token;
                }
                break;
            case 'api_key':
                if ($apiTarget->auth_token) {
                    $headers['X-API-Key'] = $apiTarget->auth_token;
                }
                break;
        }

        return $headers;
    }

    public function logs(ApiTarget $apiTarget)
    {
        $logs = $apiTarget->sendLogs()->latest()->paginate(20);
        return view('api-targets.logs', compact('apiTarget', 'logs'));
    }

    public function logDetails($logId)
    {
        $log = ApiSendLog::findOrFail($logId);
        
        return response()->json([
            'success' => true,
            'log' => [
                'id' => $log->id,
                'success' => $log->success,
                'status_code' => $log->status_code,
                'response_time' => $log->response_time,
                'response_body' => $log->response_body,
                'error_message' => $log->error_message,
                'payload' => $log->payload,
                'records_sent' => $log->records_sent,
                'retry_count' => $log->retry_count,
                'sent_at' => $log->sent_at ? $log->sent_at->format('Y-m-d H:i:s') : null,
                'completed_at' => $log->completed_at ? $log->completed_at->format('Y-m-d H:i:s') : null,
                'batch_id' => $log->batch_id,
            ]
        ]);
    }

    public function testConnection(Request $request)
    {
        try {
            $response = Http::timeout(10)->get($request->base_url);
            return response()->json([
                'success' => $response->successful(),
                'status' => $response->status(),
                'message' => $response->successful() ? 'Connection successful' : 'Connection failed'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function tableFields(Request $request)
    {
        // Return sample table fields (in real implementation, this would query the actual table)
        return response()->json([
            'fields' => [
                'id' => 'integer',
                'name' => 'string',
                'email' => 'string',
                'created_at' => 'datetime',
                'updated_at' => 'datetime'
            ]
        ]);
    }

    /**
     * Show documentation for API Target
     */
    public function documentation(ApiTarget $apiTarget)
    {
        // Generate documentation based on target configuration
        $documentation = $this->generateDocumentation($apiTarget);
        
        return view('api-targets.documentation', compact('apiTarget', 'documentation'));
    }

    /**
     * Generate documentation for API Target
     */
    protected function generateDocumentation($apiTarget)
    {
        $doc = [
            'basic_info' => [
                'name' => $apiTarget->name,
                'description' => $apiTarget->description,
                'base_url' => $apiTarget->base_url,
                'endpoint' => $apiTarget->endpoint,
                'full_url' => $apiTarget->url,
                'method' => $apiTarget->method,
                'source_table' => $apiTarget->source_table,
                'auth_type' => $apiTarget->auth_type,
                'schedule_type' => $apiTarget->schedule_type,
                'is_active' => $apiTarget->is_active,
                'created_at' => $apiTarget->created_at,
                'last_sent_at' => $apiTarget->last_sent_at,
            ],
            'authentication' => $this->getAuthDocumentation($apiTarget),
            'data_format' => $this->getDataFormatDocumentation($apiTarget),
            'usage_examples' => $this->getUsageExamples($apiTarget),
            'troubleshooting' => $this->getTroubleshootingGuide($apiTarget),
            'logs' => $apiTarget->sendLogs()->latest()->take(5)->get(),
        ];

        return $doc;
    }

    /**
     * Get authentication documentation
     */
    protected function getAuthDocumentation($apiTarget)
    {
        $auth = [
            'type' => $apiTarget->auth_type,
            'description' => '',
            'headers' => [],
            'example' => ''
        ];

        switch ($apiTarget->auth_type) {
            case 'basic_auth':
                $auth['description'] = 'Basic Authentication menggunakan username dan password';
                $auth['headers'] = [
                    'Authorization' => 'Basic ' . base64_encode($apiTarget->auth_username . ':' . $apiTarget->auth_password)
                ];
                $auth['example'] = 'curl -X POST "' . $apiTarget->url . '" -H "Authorization: Basic ' . base64_encode($apiTarget->auth_username . ':' . $apiTarget->auth_password) . '"';
                break;
            case 'bearer_token':
                $auth['description'] = 'Bearer Token Authentication';
                $auth['headers'] = [
                    'Authorization' => 'Bearer ' . $apiTarget->auth_token
                ];
                $auth['example'] = 'curl -X POST "' . $apiTarget->url . '" -H "Authorization: Bearer ' . $apiTarget->auth_token . '"';
                break;
            case 'api_key':
                $auth['description'] = 'API Key Authentication';
                $auth['headers'] = [
                    'X-API-Key' => $apiTarget->auth_token
                ];
                $auth['example'] = 'curl -X POST "' . $apiTarget->url . '" -H "X-API-Key: ' . $apiTarget->auth_token . '"';
                break;
            default:
                $auth['description'] = 'Tidak ada autentikasi';
                $auth['example'] = 'curl -X POST "' . $apiTarget->url . '"';
        }

        return $auth;
    }

    /**
     * Get data format documentation
     */
    protected function getDataFormatDocumentation($apiTarget)
    {
        $format = [
            'content_type' => 'application/json',
            'source_table' => $apiTarget->source_table,
            'sample_data' => $this->getSampleData($apiTarget->source_table),
            'field_mapping' => $apiTarget->field_mapping ? json_decode($apiTarget->field_mapping, true) : null,
        ];

        return $format;
    }

    /**
     * Get sample data from source table
     */
    protected function getSampleData($sourceTable)
    {
        try {
            if (DB::getSchemaBuilder()->hasTable($sourceTable)) {
                $data = DB::table($sourceTable)->limit(3)->get();
                return $data->map(function ($item) {
                    return (array) $item;
                })->toArray();
            }
        } catch (\Exception $e) {
            Log::error("Error getting sample data from table {$sourceTable}: " . $e->getMessage());
        }

        return [];
    }

    /**
     * Get usage examples
     */
    protected function getUsageExamples($apiTarget)
    {
        $examples = [
            'curl' => $this->getCurlExample($apiTarget),
            'php' => $this->getPhpExample($apiTarget),
            'javascript' => $this->getJavascriptExample($apiTarget),
            'python' => $this->getPythonExample($apiTarget),
        ];

        return $examples;
    }

    /**
     * Get cURL example
     */
    protected function getCurlExample($apiTarget)
    {
        $headers = $this->prepareHeaders($apiTarget);
        $headerString = '';
        
        foreach ($headers as $key => $value) {
            if ($key !== 'Content-Type' && $key !== 'Accept') {
                $headerString .= " -H \"{$key}: {$value}\"";
            }
        }

        return "curl -X {$apiTarget->method} \"{$apiTarget->url}\" \\\n" .
               "  -H \"Content-Type: application/json\" \\\n" .
               "  -H \"Accept: application/json\"{$headerString} \\\n" .
               "  -d '{\"sample\": \"data\"}'";
    }

    /**
     * Get PHP example
     */
    protected function getPhpExample($apiTarget)
    {
        $headers = $this->prepareHeaders($apiTarget);
        $headerString = '';
        
        foreach ($headers as $key => $value) {
            $headerString .= "    '{$key}' => '{$value}',\n";
        }

        return "<?php\n" .
               "\$url = '{$apiTarget->url}';\n" .
               "\$data = ['sample' => 'data'];\n" .
               "\n" .
               "\$ch = curl_init();\n" .
               "curl_setopt(\$ch, CURLOPT_URL, \$url);\n" .
               "curl_setopt(\$ch, CURLOPT_RETURNTRANSFER, true);\n" .
               "curl_setopt(\$ch, CURLOPT_CUSTOMREQUEST, '{$apiTarget->method}');\n" .
               "curl_setopt(\$ch, CURLOPT_POSTFIELDS, json_encode(\$data));\n" .
               "curl_setopt(\$ch, CURLOPT_HTTPHEADER, [\n" .
               "    'Content-Type: application/json',\n" .
               "    'Accept: application/json',\n" .
               $headerString .
               "]);\n" .
               "\n" .
               "\$response = curl_exec(\$ch);\n" .
               "curl_close(\$ch);\n" .
               "echo \$response;";
    }

    /**
     * Get JavaScript example
     */
    protected function getJavascriptExample($apiTarget)
    {
        $headers = $this->prepareHeaders($apiTarget);
        $headerString = '';
        
        foreach ($headers as $key => $value) {
            $headerString .= "  '{$key}': '{$value}',\n";
        }

        return "const url = '{$apiTarget->url}';\n" .
               "const data = { sample: 'data' };\n" .
               "\n" .
               "fetch(url, {\n" .
               "  method: '{$apiTarget->method}',\n" .
               "  headers: {\n" .
               "    'Content-Type': 'application/json',\n" .
               "    'Accept': 'application/json',\n" .
               $headerString .
               "  },\n" .
               "  body: JSON.stringify(data)\n" .
               "})\n" .
               ".then(response => response.json())\n" .
               ".then(data => console.log(data))\n" .
               ".catch(error => console.error('Error:', error));";
    }

    /**
     * Get Python example
     */
    protected function getPythonExample($apiTarget)
    {
        $headers = $this->prepareHeaders($apiTarget);
        $headerString = '';
        
        foreach ($headers as $key => $value) {
            $headerString .= "    '{$key}': '{$value}',\n";
        }

        return "import requests\n" .
               "import json\n" .
               "\n" .
               "url = '{$apiTarget->url}'\n" .
               "data = {'sample': 'data'}\n" .
               "headers = {\n" .
               "    'Content-Type': 'application/json',\n" .
               "    'Accept': 'application/json',\n" .
               $headerString .
               "}\n" .
               "\n" .
               "response = requests.request('{$apiTarget->method}', url, headers=headers, json=data)\n" .
               "print(response.text)";
    }

    /**
     * Get troubleshooting guide
     */
    protected function getTroubleshootingGuide($apiTarget)
    {
        return [
            'common_issues' => [
                [
                    'issue' => 'Connection Timeout',
                    'cause' => 'Server tidak dapat diakses atau terlalu lambat',
                    'solution' => 'Periksa koneksi internet dan URL endpoint'
                ],
                [
                    'issue' => 'Authentication Error (401)',
                    'cause' => 'Kredensial autentikasi salah atau expired',
                    'solution' => 'Periksa username/password atau token'
                ],
                [
                    'issue' => 'Data Validation Error (400)',
                    'cause' => 'Format data tidak sesuai dengan yang diharapkan',
                    'solution' => 'Periksa struktur data dan field mapping'
                ],
                [
                    'issue' => 'Server Error (500)',
                    'cause' => 'Error internal pada server tujuan',
                    'solution' => 'Hubungi administrator server tujuan'
                ]
            ],
            'checklist' => [
                'URL endpoint dapat diakses',
                'Kredensial autentikasi valid',
                'Format data sesuai skema',
                'Network connectivity stabil',
                'Rate limit tidak terlampaui'
            ]
        ];
    }
} 