<?php

namespace App\Http\Controllers;

use App\Models\ApiToken;
use App\Models\ApiClient;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ApiTokenController extends Controller
{
    public function __construct()
    {
        // Remove auth middleware
    }

    public function index()
    {
        $tokens = ApiToken::with('client')->latest()->paginate(10);
        return view('api-tokens.index', compact('tokens'));
    }

    public function create()
    {
        $clients = ApiClient::where('is_active', true)->get();
        return view('api-tokens.create', compact('clients'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'api_client_id' => 'required|exists:api_clients,id',
            'name' => 'required|string|max:255',
            'abilities' => 'nullable|array',
            'expires_at' => 'nullable|date|after:today'
        ]);

        // Generate tokens
        $token = $this->generateUniqueToken();
        $refreshToken = $this->generateUniqueToken();
        
        // Create token record
        $apiToken = ApiToken::create([
            'api_client_id' => $validated['api_client_id'],
            'name' => $validated['name'],
            'token' => $token,
            'refresh_token' => $refreshToken,
            'abilities' => $validated['abilities'] ?? ['*'],
            'expires_at' => $validated['expires_at'],
            'is_revoked' => false
        ]);

        return redirect()->route('api-tokens.index')
            ->with('success', 'API Token created successfully.')
            ->with('new_token', $token);
    }

    public function show(ApiToken $apiToken)
    {
        $apiToken->load('client');
        return view('api-tokens.show', compact('apiToken'));
    }

    public function edit(ApiToken $apiToken)
    {
        $clients = ApiClient::where('is_active', true)->get();
        return view('api-tokens.edit', compact('apiToken', 'clients'));
    }

    public function update(Request $request, ApiToken $apiToken)
    {
        $validated = $request->validate([
            'api_client_id' => 'required|exists:api_clients,id',
            'name' => 'required|string|max:255',
            'abilities' => 'nullable|array',
            'expires_at' => 'nullable|date|after:today',
            'is_revoked' => 'boolean'
        ]);

        $validated['is_revoked'] = $request->has('is_revoked');
        $validated['abilities'] = $validated['abilities'] ?? ['*'];

        $apiToken->update($validated);

        return redirect()->route('api-tokens.index')
            ->with('success', 'API Token updated successfully.');
    }

    public function destroy(ApiToken $apiToken)
    {
        $apiToken->delete();

        return redirect()->route('api-tokens.index')
            ->with('success', 'API Token deleted successfully.');
    }

    public function revoke(ApiToken $apiToken)
    {
        $apiToken->update(['is_revoked' => true]);
        return redirect()->route('api-tokens.index')->with('success', 'API Token revoked successfully.');
    }

    public function regenerate(ApiToken $apiToken)
    {
        $newToken = $this->generateUniqueToken();
        $apiToken->update([
            'token' => $newToken,
            'is_revoked' => false,
            'last_used_at' => null
        ]);

        return back()->with('success', 'API Token regenerated successfully.')
                    ->with('new_token', $newToken);
    }

    private function generateUniqueToken()
    {
        do {
            $token = 'at_' . Str::random(40);
        } while (ApiToken::where('token', $token)->exists());

        return $token;
    }
} 