<?php

namespace App\Http\Controllers;

use App\Models\ApiClient;
use App\Models\ApiReceiver;
use App\Models\ApiTarget;
use App\Models\ApiTransceiver;
use App\Models\ApiTransceiverLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use GuzzleHttp\Client;

class ApiTransceiverController extends Controller
{
    public function index()
    {
        $api_transceivers = ApiTransceiver::with(['apiClient', 'receiver', 'target'])
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('api-transceivers.index', compact('api_transceivers'));
    }

    public function create()
    {
        $apiClients = ApiClient::where('is_active', true)->orderBy('name')->get();
        $apiReceivers = ApiReceiver::where('is_active', true)->orderBy('name')->get();
        $apiTargets = ApiTarget::where('is_active', true)->orderBy('name')->get();

        return view('api-transceivers.create', compact('apiClients', 'apiReceivers', 'apiTargets'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'api_client_id' => 'required|exists:api_clients,id',
            'api_receiver_id' => 'required|exists:api_receivers,id',
            'api_target_id' => 'required|exists:api_targets,id',
            'direction' => 'required|in:unidirectional,bidirectional',
            'schedule_type' => 'required|in:manual,interval,cron',
            'sync_interval' => 'required_if:schedule_type,interval|nullable|integer|min:1|max:1440',
            'cron_expression' => 'required_if:schedule_type,cron|nullable|string',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
            'auto_sync' => 'boolean',
        ]);

        try {
            $transceiver = new ApiTransceiver();
            $transceiver->name = $validated['name'];
            $transceiver->api_client_id = $validated['api_client_id'];
            $transceiver->api_receiver_id = $validated['api_receiver_id'];
            $transceiver->api_target_id = $validated['api_target_id'];
            $transceiver->direction = $validated['direction'];
            $transceiver->schedule_type = $validated['schedule_type'];
            
            if ($validated['schedule_type'] === 'interval') {
                $transceiver->sync_interval = $validated['sync_interval'];
                $transceiver->cron_expression = null;
            } elseif ($validated['schedule_type'] === 'cron') {
                $transceiver->sync_interval = null;
                $transceiver->cron_expression = $validated['cron_expression'];
            } else {
                $transceiver->sync_interval = null;
                $transceiver->cron_expression = null;
            }
            
            $transceiver->description = $validated['description'] ?? null;
            $transceiver->is_active = $validated['is_active'] ?? false;
            $transceiver->auto_sync = $validated['auto_sync'] ?? false;
            
            $transceiver->save();

            return redirect()
                ->route('api-transceivers.index')
                ->with('success', 'API Transceiver created successfully.');
                
        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'Failed to create API Transceiver: ' . $e->getMessage());
        }
    }

    public function show(ApiTransceiver $apiTransceiver)
    {
        $apiTransceiver->load(['apiClient', 'receiver', 'target']);
        return view('api-transceivers.show', compact('apiTransceiver'));
    }

    public function edit(ApiTransceiver $apiTransceiver)
    {
        $apiClients = ApiClient::where('is_active', true)->orderBy('name')->get();
        $apiReceivers = ApiReceiver::where('is_active', true)->orderBy('name')->get();
        $apiTargets = ApiTarget::where('is_active', true)->orderBy('name')->get();

        return view('api-transceivers.edit', compact('apiTransceiver', 'apiClients', 'apiReceivers', 'apiTargets'));
    }

    public function update(Request $request, ApiTransceiver $apiTransceiver)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'api_client_id' => 'required|exists:api_clients,id',
            'api_receiver_id' => 'required|exists:api_receivers,id',
            'api_target_id' => 'required|exists:api_targets,id',
            'direction' => 'required|in:unidirectional,bidirectional',
            'schedule_type' => 'required|in:manual,interval,cron',
            'sync_interval' => 'required_if:schedule_type,interval|nullable|integer|min:1|max:1440',
            'cron_expression' => 'required_if:schedule_type,cron|nullable|string',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
            'auto_sync' => 'boolean',
        ]);

        $apiTransceiver->update($validated);

        return redirect()
            ->route('api-transceivers.show', $apiTransceiver)
            ->with('success', 'API Transceiver updated successfully.');
    }

    public function destroy(ApiTransceiver $apiTransceiver)
    {
        $apiTransceiver->delete();
        return redirect()
            ->route('api-transceivers.index')
            ->with('success', 'API Transceiver deleted successfully.');
    }

    public function test(ApiTransceiver $apiTransceiver)
    {
        try {
            // Simulate forwarding data from source to target
            ApiTransceiverLog::create([
                'api_transceiver_id' => $apiTransceiver->id,
                'status' => 'success',
                'message' => 'Test forwarding completed successfully',
                'records_processed' => 1,
                'execution_time' => 0.5
            ]);

            return back()->with('success', 'Test completed successfully.');
        } catch (\Exception $e) {
            ApiTransceiverLog::create([
                'api_transceiver_id' => $apiTransceiver->id,
                'status' => 'error',
                'message' => 'Test failed: ' . $e->getMessage(),
                'records_processed' => 0,
                'execution_time' => 0
            ]);

            return back()->with('error', 'Test failed: ' . $e->getMessage());
        }
    }

    public function toggleStatus(Request $request, ApiTransceiver $apiTransceiver)
    {
        $apiTransceiver->update(['is_active' => !$apiTransceiver->is_active]);

        $status = $apiTransceiver->is_active ? 'activated' : 'deactivated';
        $message = "API Transceiver {$status} successfully.";
        
        if ($request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => $message,
                'is_active' => $apiTransceiver->is_active
            ]);
        }
        
        return back()->with('success', $message);
    }

    public function logs(ApiTransceiver $apiTransceiver)
    {
        $logs = $apiTransceiver->logs()->latest()->paginate(20);
        return view('api-transceivers.logs', compact('apiTransceiver', 'logs'));
    }

    public function sync(Request $request, ApiTransceiver $apiTransceiver)
    {
        \Log::info('[SYNC-DEBUG] Sync requested', [
            'transceiver_id' => $apiTransceiver->id,
            'receiver_id' => $apiTransceiver->api_receiver_id,
            'target_id' => $apiTransceiver->api_target_id,
            'route_params' => $request->route()->parameters(),
        ]);
        $success = false;
        $message = '';
        
        try {
            $receiver = $apiTransceiver->receiver;
            $target = $apiTransceiver->target;
            \Log::info('[SYNC-DEBUG] Receiver/Target loaded', [
                'receiver' => $receiver ? $receiver->toArray() : null,
                'target' => $target ? $target->toArray() : null,
            ]);
            
            // Validate that receiver and target exist
            if (!$receiver) {
                throw new \Exception('No receiver configured for this transceiver');
            }
            
            if (!$target) {
                throw new \Exception('No target configured for this transceiver');
            }
            
            $startTime = microtime(true);
            $successCount = 0;
            $errorCount = 0;
            
            // Check if receiver has a target table configured
            if (!$receiver->target_table) {
                // If no target table, simulate a successful sync for demo purposes
                $message = "Sync completed successfully. No data to process (target table not configured).";
                $success = true;
            } else {
                // Step 1: Check if target table exists
                if (!DB::getSchemaBuilder()->hasTable($receiver->target_table)) {
                    throw new \Exception("Target table '{$receiver->target_table}' does not exist");
                }
                
                // Step 2: Get data from receiver
                $receivedData = DB::table($receiver->target_table)
                    ->where('processed', false)
                    ->orWhereNull('processed')
                    ->limit(100)
                    ->get();
                    
                // Step 3: Send to target
                $client = new Client(['timeout' => 30, 'connect_timeout' => 10]);
                
                foreach ($receivedData as $data) {
                    try {
                        $requestData = (array) $data;
                        
                        $response = $client->request($target->method ?? 'POST', $target->url, [
                            'json' => $requestData,
                            'headers' => [
                                'Content-Type' => 'application/json',
                                'Accept' => 'application/json',
                                'Authorization' => $target->auth_token ? 'Bearer ' . $target->auth_token : null
                            ]
                        ]);
                        
                        if ($response->getStatusCode() >= 200 && $response->getStatusCode() < 300) {
                            // Mark as processed if the table has these columns
                            if (DB::getSchemaBuilder()->hasColumn($receiver->target_table, 'processed')) {
                                DB::table($receiver->target_table)
                                    ->where('id', $data->id)
                                    ->update(['processed' => true, 'processed_at' => now()]);
                            }
                                
                            $successCount++;
                        }
                    } catch (\Exception $e) {
                        $errorCount++;
                        Log::error("Sync error for record {$data->id}: " . $e->getMessage());
                    }
                }
                
                // Step 4: If bidirectional, get data from target
                if ($apiTransceiver->direction === 'bidirectional' && isset($target->callback_url) && $target->callback_url) {
                    try {
                        $targetResponse = $client->request('GET', $target->callback_url, [
                            'headers' => [
                                'Accept' => 'application/json',
                                'Authorization' => $target->auth_token ? 'Bearer ' . $target->auth_token : null
                            ]
                        ]);
                        
                        if ($targetResponse->getStatusCode() === 200) {
                            $targetData = json_decode($targetResponse->getBody(), true);
                            
                            // Process target data
                            if (is_array($targetData)) {
                                foreach ($targetData as $item) {
                                    if (is_array($item)) {
                                        $item['created_at'] = now();
                                        $item['updated_at'] = now();
                                        DB::table($receiver->target_table)->insert($item);
                                    }
                                }
                            }
                        }
                    } catch (\Exception $e) {
                        Log::error("Bidirectional sync error: " . $e->getMessage());
                    }
                }
                
                $message = "Sync completed successfully. Processed $successCount records" . 
                          ($errorCount > 0 ? " with $errorCount errors" : "") . ".";
                $success = true;
            }
            
            $executionTime = microtime(true) - $startTime;
            
            // Log the sync operation
            ApiTransceiverLog::create([
                'api_transceiver_id' => $apiTransceiver->id,
                'status' => 'success',
                'message' => $message,
                'records_processed' => $successCount + $errorCount,
                'execution_time' => $executionTime,
                'details' => json_encode([
                    'success_count' => $successCount,
                    'error_count' => $errorCount,
                    'bidirectional' => $apiTransceiver->direction === 'bidirectional'
                ])
            ]);

        } catch (\Exception $e) {
            $message = 'Failed to sync: ' . $e->getMessage();
            $success = false;
            
            Log::error("Transceiver sync failed: " . $e->getMessage(), [
                'transceiver_id' => $apiTransceiver->id,
                'trace' => $e->getTraceAsString()
            ]);
            
            ApiTransceiverLog::create([
                'api_transceiver_id' => $apiTransceiver->id,
                'status' => 'error',
                'message' => $message,
                'records_processed' => 0,
                'execution_time' => 0,
                'details' => json_encode(['error' => $e->getMessage()])
            ]);
        }
        \Log::info('[SYNC-DEBUG] Sync finished', [
            'transceiver_id' => $apiTransceiver->id,
            'success' => $success,
            'message' => $message,
        ]);

        if ($request->wantsJson()) {
            return response()->json([
                'success' => $success,
                'message' => $message
            ]);
        }

        return redirect()->back()
            ->with($success ? 'success' : 'error', $message);
    }
} 