<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Models\ApiSource;
use App\Models\ApiTarget;
use App\Models\ApiReceiver;
use App\Models\ApiLog;
use App\Models\ApiReceiveLog;
use App\Models\ApiSendLog;
use App\Models\ApiTransceiver;
use App\Models\ApiClient;

class DashboardController extends Controller
{
    public function index()
    {
        try {
            // Calculate base counts
            $apiSourceCount = ApiSource::count();
            $apiTargetCount = ApiTarget::count();
            $apiReceiverCount = ApiReceiver::count();
            
            $stats = [
                'api_sources' => $apiSourceCount,
                'api_targets' => $apiTargetCount,
                'api_receivers' => $apiReceiverCount,
                'total_requests' => ApiLog::count() + ApiReceiveLog::count() + ApiSendLog::count(),
                'active_sources' => ApiSource::where('status', 'active')->count(),
                'active_targets' => ApiTarget::where('is_active', true)->count(),
                'active_receivers' => ApiReceiver::where('is_active', true)->count(),
                'get_endpoints' => ApiReceiver::where('allowed_methods', 'like', '%GET%')->count(),
                'post_endpoints' => ApiReceiver::where('allowed_methods', 'like', '%POST%')->count(),
                'total_apis' => $apiSourceCount + $apiTargetCount + $apiReceiverCount,
                'total_clients' => ApiClient::count(),
                'total_transceivers' => ApiTransceiver::count(),
                'active_transceivers' => ApiTransceiver::where('is_active', true)->count(),
                'system_status' => 'Online',
                'access_mode' => 'Public',
                'success_rate' => $this->calculateSuccessRate(),
                'error_rate' => $this->calculateErrorRate(),
                'today_requests' => $this->getTodayRequests(),
                // Tambahan statistik
                'uptime' => $this->getUptime(),
                'memory_usage' => $this->getMemoryUsage(),
                'php_version' => PHP_VERSION,
                'laravel_version' => app()->version(),
            ];
        } catch (\Exception $e) {
            // Fallback stats if there's an error
            $stats = $this->getDefaultStats();
            $stats['system_status'] = 'Error';
            
            // Log the error for debugging
            Log::error('Dashboard stats calculation error: ' . $e->getMessage());
        }

        // Ensure all required keys exist with default values
        $stats = array_merge($this->getDefaultStats(), $stats);

        // Debug: Log the stats array to see what's being passed (disabled in production)
        // Log::info('Dashboard stats: ', $stats);

        $recent_logs = ApiLog::with('apiSource')
            ->latest()
            ->limit(10)
            ->get();

        $api_sources = ApiSource::latest()->limit(5)->get();
        $api_targets = ApiTarget::latest()->limit(5)->get();
        $api_receivers = ApiReceiver::latest()->limit(5)->get();

        return view('dashboard.index', compact('stats', 'recent_logs', 'api_sources', 'api_targets', 'api_receivers'));
    }

    private function getDefaultStats()
    {
        return [
            'api_sources' => 0,
            'api_targets' => 0,
            'api_receivers' => 0,
            'total_requests' => 0,
            'active_sources' => 0,
            'active_targets' => 0,
            'active_receivers' => 0,
            'get_endpoints' => 0,
            'post_endpoints' => 0,
            'total_apis' => 0,
            'total_clients' => 0,
            'total_transceivers' => 0,
            'active_transceivers' => 0,
            'system_status' => 'Unknown',
            'access_mode' => 'Public',
            'success_rate' => 0,
            'error_rate' => 0,
            'today_requests' => 0,
        ];
    }

    private function calculateSuccessRate()
    {
        $totalRequests = ApiLog::count() + ApiReceiveLog::count() + ApiSendLog::count();
        if ($totalRequests === 0) {
            return 100;
        }

        $successfulRequests = 
            ApiLog::where('status_code', '>=', 200)->where('status_code', '<', 300)->count() +
            ApiReceiveLog::where('status_code', '>=', 200)->where('status_code', '<', 300)->count() +
            ApiSendLog::where('status_code', '>=', 200)->where('status_code', '<', 300)->count();

        return round(($successfulRequests / $totalRequests) * 100, 1);
    }

    private function calculateErrorRate()
    {
        $totalRequests = ApiLog::count() + ApiReceiveLog::count() + ApiSendLog::count();
        if ($totalRequests === 0) {
            return 0;
        }

        $errorRequests = 
            ApiLog::where('status_code', '>=', 400)->count() +
            ApiReceiveLog::where('status_code', '>=', 400)->count() +
            ApiSendLog::where('status_code', '>=', 400)->count();

        return round(($errorRequests / $totalRequests) * 100, 1);
    }

    private function getTodayRequests()
    {
        return 
            ApiLog::whereDate('created_at', today())->count() +
            ApiReceiveLog::whereDate('created_at', today())->count() +
            ApiSendLog::whereDate('created_at', today())->count();
    }

    // Tambahan fungsi statistik
    private function getUptime()
    {
        // Sederhana: waktu server berjalan sejak boot (bisa diganti dengan uptime server sebenarnya)
        if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
            return 'N/A'; // Uptime server di Windows tidak mudah didapat
        }
        $uptime = @file_get_contents('/proc/uptime');
        if ($uptime !== false) {
            $seconds = (int)floatval(explode(' ', $uptime)[0]);
            $hours = floor($seconds / 3600);
            $minutes = floor(($seconds % 3600) / 60);
            return $hours . ' jam ' . $minutes . ' menit';
        }
        return 'Unknown';
    }

    private function getMemoryUsage()
    {
        $bytes = memory_get_usage(true);
        $units = ['B', 'KB', 'MB', 'GB'];
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        return round($bytes, 2) . ' ' . $units[$i];
    }
} 