<?php

namespace App\Http\Controllers;

use App\Models\GetOnlyReceiver;
use App\Models\ApiToken;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class GetOnlyReceiverController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        try {
            // Check if table exists
            if (!Schema::hasTable('get_only_receivers')) {
                return view('get-only-receivers.index', [
                    'receivers' => collect([]),
                    'stats' => [
                        'total_receivers' => 0,
                        'active_receivers' => 0,
                        'total_requests' => 0,
                        'total_success' => 0,
                        'total_errors' => 0,
                    ],
                    'message' => 'Database table not found. Please run migrations.'
                ]);
            }

            $receivers = GetOnlyReceiver::with('apiToken')
                ->latest()
                ->paginate(10);

            $stats = [
                'total_receivers' => GetOnlyReceiver::count(),
                'active_receivers' => GetOnlyReceiver::where('is_active', true)->count(),
                'total_requests' => GetOnlyReceiver::sum('total_requests'),
                'total_success' => GetOnlyReceiver::sum('total_success'),
                'total_errors' => GetOnlyReceiver::sum('total_errors'),
            ];

            return view('get-only-receivers.index', compact('receivers', 'stats'));
            
        } catch (\Exception $e) {
            Log::error('Error in GetOnlyReceiverController::index', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return view('get-only-receivers.index', [
                'receivers' => collect([]),
                'stats' => [
                    'total_receivers' => 0,
                    'active_receivers' => 0,
                    'total_requests' => 0,
                    'total_success' => 0,
                    'total_errors' => 0,
                ],
                'error' => 'Database error: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $apiTokens = ApiToken::where('is_revoked', false)
            ->where(function($query) {
                $query->whereNull('expires_at')
                      ->orWhere('expires_at', '>', now());
            })
            ->get();
        $tables = $this->getDatabaseTables();
        
        return view('get-only-receivers.create', compact('apiTokens', 'tables'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // PRE-PROCESS REQUEST DATA
        
        // 1. Handle array fields from comma-separated strings
        if ($request->has('response_fields') && is_string($request->response_fields)) {
            $fields = array_filter(array_map('trim', explode(',', $request->response_fields)));
            $request->merge(['response_fields' => $fields]);
        }

        if ($request->has('cors_origins') && is_string($request->cors_origins)) {
            $origins = array_filter(array_map('trim', explode(',', $request->cors_origins)));
            $request->merge(['cors_origins' => $origins]);
        }

        // 2. Handle boolean fields properly (checkboxes that are unchecked don't send values)
        $booleanFields = ['enable_pagination', 'require_token', 'enable_cors', 'enable_cache', 'is_active'];
        foreach ($booleanFields as $field) {
            if (!$request->has($field)) {
                $request->merge([$field => false]);
            } else {
                $request->merge([$field => (bool)$request->input($field)]);
            }
        }

        // 3. Set default values for optional fields
        if (!$request->has('description') || empty($request->description)) {
            $request->merge(['description' => null]);
        }

        // 4. Generate endpoint from name if empty
        if (!$request->has('endpoint') || empty($request->endpoint)) {
            $endpoint = Str::slug($request->name) . '-' . Str::random(6);
            $request->merge(['endpoint' => $endpoint]);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'endpoint' => 'required|string|max:255|unique:get_only_receivers,endpoint|regex:/^[a-zA-Z0-9\-_]+$/',
            'data_source' => 'required|in:table,api,custom',
            'source_table' => 'required_if:data_source,table|nullable|string',
            'source_api_url' => 'required_if:data_source,api|nullable|url',
            'source_api_headers' => 'nullable|array',
            'custom_query' => 'required_if:data_source,custom|nullable|string',
            'api_token_id' => 'nullable|exists:api_tokens,id',
            'require_token' => 'boolean',
            'allowed_tokens' => 'nullable|array',
            'allowed_tokens.*' => 'exists:api_tokens,id',
            'response_format' => 'required|in:json,xml,csv',
            'response_fields' => 'nullable|array',
            'default_filters' => 'nullable|array',
            'allowed_filters' => 'nullable|array',
            'default_limit' => 'required|integer|min:1|max:1000',
            'max_limit' => 'required|integer|min:1|max:10000',
            'enable_pagination' => 'boolean',
            'rate_limit_per_minute' => 'required|integer|min:1|max:1000',
            'rate_limit_per_hour' => 'required|integer|min:1|max:10000',
            'allowed_ips' => 'nullable|array',
            'enable_cors' => 'boolean',
            'cors_origins' => 'nullable|array',
            'enable_cache' => 'boolean',
            'cache_duration' => 'required|integer|min:60|max:86400',
            'validation_rules' => 'nullable|array',
            'required_parameters' => 'nullable|array',
            'optional_parameters' => 'nullable|array',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $data = $validator->validated();
            $data['created_by'] = auth()->user()->name ?? 'System';
            
            // Generate unique endpoint if empty
            if (empty($data['endpoint'])) {
                $data['endpoint'] = Str::slug($data['name']) . '-' . Str::random(6);
            }

            // Validate source table exists if data_source is table
            if ($data['data_source'] === 'table' && $data['source_table']) {
                if (!Schema::hasTable($data['source_table'])) {
                    return redirect()->back()
                        ->withErrors(['source_table' => 'Table does not exist in database'])
                        ->withInput();
                }
            }

            // Validate custom query if data_source is custom
            if ($data['data_source'] === 'custom' && $data['custom_query']) {
                try {
                    // Test query syntax
                    DB::select("EXPLAIN " . $data['custom_query']);
                } catch (\Exception $e) {
                    return redirect()->back()
                        ->withErrors(['custom_query' => 'Invalid SQL query: ' . $e->getMessage()])
                        ->withInput();
                }
            }

            $receiver = GetOnlyReceiver::create($data);

            Log::info('GetOnlyReceiver created successfully', [
                'id' => $receiver->id,
                'name' => $receiver->name,
                'endpoint' => $receiver->endpoint
            ]);

            return redirect()->route('get-only-receivers.index')
                ->with('success', 'GET-Only Receiver berhasil dibuat!');

        } catch (\Exception $e) {
            Log::error('Error creating GetOnlyReceiver', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()
                ->withErrors(['error' => 'Gagal membuat receiver: ' . $e->getMessage()])
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(GetOnlyReceiver $get_only_receiver)
    {
        $getOnlyReceiver = $get_only_receiver;
        $getOnlyReceiver->load('apiToken');
        
        // Get recent access logs (you might want to create a separate log table)
        $recentStats = [
            'today_requests' => 0, // Implement if you have access logs
            'success_rate' => $getOnlyReceiver->total_requests > 0 
                ? round(($getOnlyReceiver->total_success / $getOnlyReceiver->total_requests) * 100, 2) 
                : 0,
        ];

        return view('get-only-receivers.show', compact('getOnlyReceiver', 'recentStats'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(GetOnlyReceiver $get_only_receiver)
    {
        $getOnlyReceiver = $get_only_receiver;
        $apiTokens = ApiToken::where('is_revoked', false)
            ->where(function($query) {
                $query->whereNull('expires_at')
                      ->orWhere('expires_at', '>', now());
            })
            ->get();
        $tables = $this->getDatabaseTables();
        
        return view('get-only-receivers.edit', compact('getOnlyReceiver', 'apiTokens', 'tables'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, GetOnlyReceiver $get_only_receiver)
    {
        $getOnlyReceiver = $get_only_receiver;
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'endpoint' => 'required|string|max:255|regex:/^[a-zA-Z0-9\-_]+$/|unique:get_only_receivers,endpoint,' . $getOnlyReceiver->id,
            'data_source' => 'required|in:table,api,custom',
            'source_table' => 'required_if:data_source,table|nullable|string',
            'source_api_url' => 'required_if:data_source,api|nullable|url',
            'source_api_headers' => 'nullable|array',
            'custom_query' => 'required_if:data_source,custom|nullable|string',
            'api_token_id' => 'nullable|exists:api_tokens,id',
            'require_token' => 'boolean',
            'allowed_tokens' => 'nullable|array',
            'allowed_tokens.*' => 'exists:api_tokens,id',
            'response_format' => 'required|in:json,xml,csv',
            'response_fields' => 'nullable|array',
            'default_filters' => 'nullable|array',
            'allowed_filters' => 'nullable|array',
            'default_limit' => 'required|integer|min:1|max:1000',
            'max_limit' => 'required|integer|min:1|max:10000',
            'enable_pagination' => 'boolean',
            'rate_limit_per_minute' => 'required|integer|min:1|max:1000',
            'rate_limit_per_hour' => 'required|integer|min:1|max:10000',
            'allowed_ips' => 'nullable|array',
            'enable_cors' => 'boolean',
            'cors_origins' => 'nullable|array',
            'enable_cache' => 'boolean',
            'cache_duration' => 'required|integer|min:60|max:86400',
            'validation_rules' => 'nullable|array',
            'required_parameters' => 'nullable|array',
            'optional_parameters' => 'nullable|array',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $data = $validator->validated();

            // Validate source table exists if data_source is table
            if ($data['data_source'] === 'table' && $data['source_table']) {
                if (!Schema::hasTable($data['source_table'])) {
                    return redirect()->back()
                        ->withErrors(['source_table' => 'Table does not exist in database'])
                        ->withInput();
                }
            }

            // Validate custom query if data_source is custom
            if ($data['data_source'] === 'custom' && $data['custom_query']) {
                try {
                    // Test query syntax
                    DB::select("EXPLAIN " . $data['custom_query']);
                } catch (\Exception $e) {
                    return redirect()->back()
                        ->withErrors(['custom_query' => 'Invalid SQL query: ' . $e->getMessage()])
                        ->withInput();
                }
            }

            $getOnlyReceiver->update($data);

            Log::info('GetOnlyReceiver updated successfully', [
                'id' => $getOnlyReceiver->id,
                'name' => $getOnlyReceiver->name
            ]);

            return redirect()->route('get-only-receivers.show', $getOnlyReceiver)
                ->with('success', 'GET-Only Receiver berhasil diupdate!');

        } catch (\Exception $e) {
            Log::error('Error updating GetOnlyReceiver', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()
                ->withErrors(['error' => 'Gagal mengupdate receiver: ' . $e->getMessage()])
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(GetOnlyReceiver $get_only_receiver)
    {
        try {
            $getOnlyReceiver = $get_only_receiver;
            $name = $getOnlyReceiver->name;
            $getOnlyReceiver->delete();

            Log::info('GetOnlyReceiver deleted successfully', [
                'name' => $name
            ]);

            return redirect()->route('get-only-receivers.index')
                ->with('success', "GET-Only Receiver '{$name}' berhasil dihapus!");

        } catch (\Exception $e) {
            Log::error('Error deleting GetOnlyReceiver', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()
                ->withErrors(['error' => 'Gagal menghapus receiver: ' . $e->getMessage()]);
        }
    }

    /**
     * Handle GET requests to receiver endpoint
     */
    public function receive(Request $request, $endpoint)
    {
        $startTime = microtime(true);
        
        try {
            // Find receiver
            $receiver = GetOnlyReceiver::where('endpoint', $endpoint)
                ->where('is_active', true)
                ->first();

            if (!$receiver) {
                return $this->errorResponse('Receiver not found', 404, $startTime);
            }

            // Check rate limiting
            $rateLimitKey = 'get_receiver_' . $receiver->id . '_' . $request->ip();
            
            if (RateLimiter::tooManyAttempts($rateLimitKey . '_minute', $receiver->rate_limit_per_minute)) {
                return $this->errorResponse('Rate limit exceeded (per minute)', 429, $startTime);
            }
            
            if (RateLimiter::tooManyAttempts($rateLimitKey . '_hour', $receiver->rate_limit_per_hour)) {
                return $this->errorResponse('Rate limit exceeded (per hour)', 429, $startTime);
            }

            // Increment rate limit counters
            RateLimiter::hit($rateLimitKey . '_minute', 60);
            RateLimiter::hit($rateLimitKey . '_hour', 3600);

            // Check IP whitelist
            if ($receiver->allowed_ips && !in_array($request->ip(), $receiver->allowed_ips)) {
                return $this->errorResponse('IP not allowed', 403, $startTime);
            }

            // Validate token
            $token = $request->bearerToken() ?: $request->get('token') ?: $request->get('api_key');
            
            if ($receiver->require_token) {
                if (!$token) {
                    return $this->errorResponse('Token required', 401, $startTime);
                }

                if (!$receiver->isTokenValid($token)) {
                    return $this->errorResponse('Invalid token', 401, $startTime);
                }
            }

            // Get query parameters
            $parameters = $request->query();
            unset($parameters['token'], $parameters['api_key']);

            // Validate parameters
            $validationErrors = $receiver->validateParameters($parameters);
            if (!empty($validationErrors)) {
                return $this->errorResponse('Validation failed: ' . implode(', ', $validationErrors), 400, $startTime);
            }

            // Get pagination parameters
            $limit = min((int)$request->get('limit', $receiver->default_limit), $receiver->max_limit);
            $offset = max(0, (int)$request->get('offset', 0));

            // Get data
            $result = $receiver->getData($parameters, $limit, $offset);

            // Update statistics
            $responseTime = (microtime(true) - $startTime) * 1000; // milliseconds
            $receiver->updateStats($responseTime, true);

            // Prepare response
            $response = [
                'success' => true,
                'data' => $result['data'],
                'meta' => [
                    'total' => $result['total'],
                    'limit' => $result['limit'],
                    'offset' => $result['offset'],
                    'has_more' => $result['has_more'],
                    'response_time' => round($responseTime, 2) . 'ms'
                ]
            ];

            // Set CORS headers if enabled
            $responseObj = response()->json($response);
            
            if ($receiver->enable_cors) {
                $origins = $receiver->cors_origins ?: ['*'];
                $responseObj->header('Access-Control-Allow-Origin', implode(', ', $origins));
                $responseObj->header('Access-Control-Allow-Methods', 'GET, OPTIONS');
                $responseObj->header('Access-Control-Allow-Headers', 'Content-Type, Authorization');
            }

            return $responseObj;

        } catch (\Exception $e) {
            Log::error('Error in GetOnlyReceiver', [
                'endpoint' => $endpoint,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            $responseTime = (microtime(true) - $startTime) * 1000;
            if (isset($receiver)) {
                $receiver->updateStats($responseTime, false);
            }

            return $this->errorResponse('Internal server error', 500, $startTime);
        }
    }

    /**
     * Test receiver endpoint
     */
    public function test(Request $request, GetOnlyReceiver $get_only_receiver)
    {
        try {
            $getOnlyReceiver = $get_only_receiver;
            $parameters = $request->get('parameters', []);
            $limit = min((int)$request->get('limit', 10), 10); // Max 10 for testing

            $result = $getOnlyReceiver->getData($parameters, $limit, 0);

            return response()->json([
                'success' => true,
                'message' => 'Test successful',
                'data' => $result,
                'endpoint_url' => url('/api/get/' . $getOnlyReceiver->endpoint)
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Test failed: ' . $e->getMessage()
            ], 400);
        }
    }

    /**
     * Get database tables
     */
    private function getDatabaseTables()
    {
        try {
            $tables = DB::select('SHOW TABLES');
            $tableNames = [];
            
            foreach ($tables as $table) {
                $tableArray = (array) $table;
                $tableNames[] = array_values($tableArray)[0];
            }
            
            return $tableNames;
        } catch (\Exception $e) {
            return [];
        }
    }

    /**
     * Return error response
     */
    private function errorResponse($message, $code, $startTime)
    {
        $responseTime = (microtime(true) - $startTime) * 1000;
        
        return response()->json([
            'success' => false,
            'error' => $message,
            'meta' => [
                'response_time' => round($responseTime, 2) . 'ms'
            ]
        ], $code);
    }
}
