<?php

namespace App\Http\Controllers;

use App\Models\PostOnlyReceiver;
use App\Models\PostOnlyReceiveLog;
use App\Models\ApiClient;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class PostOnlyReceiverController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $receivers = PostOnlyReceiver::with('apiClient')
            ->latest()
            ->paginate(10);

        return view('post-only-receivers.index', compact('receivers'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $apiClients = ApiClient::where('is_active', true)->get();
        $tables = $this->getDatabaseTables();
        return view('post-only-receivers.create', compact('apiClients', 'tables'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        Log::info('PostOnlyReceiver store attempt', [
            'request_data' => $request->all(),
            'user_agent' => $request->userAgent(),
            'ip' => $request->ip()
        ]);

        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'description' => 'nullable|string',
                'endpoint' => 'required|string|max:255|unique:post_only_receivers,endpoint',
                'target_table' => 'required|string|max:255',
                'api_client_id' => 'nullable|exists:api_clients,id',
                'require_token' => 'boolean',
                'field_mapping' => 'nullable|array',
                'validation_rules' => 'nullable|array',
                'auto_create_table' => 'boolean',
                'allowed_ips' => 'nullable|array',
                'rate_limit' => 'nullable|integer|min:1',
                'is_active' => 'boolean',
                'sample_data' => 'nullable|array'
            ]);

            // Handle custom table name
            if ($request->has('target_table_select') && $request->target_table_select === '__custom__') {
                $validated['target_table'] = $request->custom_table_name;
                
                // Validate custom table name
                if (empty($request->custom_table_name)) {
                    return redirect()->back()
                        ->withInput()
                        ->with('error', 'Nama tabel custom harus diisi.');
                }
                
                // Validate table name format
                if (!preg_match('/^[a-zA-Z_][a-zA-Z0-9_]*$/', $request->custom_table_name)) {
                    return redirect()->back()
                        ->withInput()
                        ->with('error', 'Nama tabel hanya boleh menggunakan huruf, angka, dan underscore. Harus dimulai dengan huruf atau underscore.');
                }
            }

            // Convert checkbox values to proper booleans
            $validated['require_token'] = $request->has('require_token');
            $validated['auto_create_table'] = $request->has('auto_create_table');
            $validated['is_active'] = $request->has('is_active');

            // Generate unique endpoint if not provided
            if (empty($validated['endpoint'])) {
                $validated['endpoint'] = Str::slug($validated['name']) . '-' . Str::random(6);
            }

            // Validate API client requirement
            if ($validated['require_token'] && !$validated['api_client_id']) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'API Client harus dipilih jika token diperlukan.');
            }

            Log::info('PostOnlyReceiver validated data', ['validated' => $validated]);

            $receiver = PostOnlyReceiver::create($validated);

            Log::info('PostOnlyReceiver created successfully', ['id' => $receiver->id]);

            return redirect()->route('post-only-receivers.index')
                ->with('success', 'POST-Only Receiver berhasil dibuat.');

        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('PostOnlyReceiver validation failed', [
                'errors' => $e->errors(),
                'request_data' => $request->all()
            ]);
            throw $e;
        } catch (\Exception $e) {
            Log::error('PostOnlyReceiver creation failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Gagal membuat POST-Only Receiver: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(PostOnlyReceiver $postOnlyReceiver)
    {
        $postOnlyReceiver->load('apiClient');
        $recentLogs = $postOnlyReceiver->receiveLogs()
            ->with(['apiClient', 'apiToken'])
            ->latest()
            ->take(10)
            ->get();

        return view('post-only-receivers.show', compact('postOnlyReceiver', 'recentLogs'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(PostOnlyReceiver $postOnlyReceiver)
    {
        $apiClients = ApiClient::where('is_active', true)->get();
        $tables = $this->getDatabaseTables();
        return view('post-only-receivers.edit', compact('postOnlyReceiver', 'apiClients', 'tables'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, PostOnlyReceiver $postOnlyReceiver)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'description' => 'nullable|string',
                'endpoint' => 'required|string|max:255|unique:post_only_receivers,endpoint,' . $postOnlyReceiver->id,
                'target_table' => 'required|string|max:255',
                'api_client_id' => 'nullable|exists:api_clients,id',
                'require_token' => 'boolean',
                'field_mapping' => 'nullable|array',
                'validation_rules' => 'nullable|array',
                'auto_create_table' => 'boolean',
                'allowed_ips' => 'nullable|array',
                'rate_limit' => 'nullable|integer|min:1',
                'is_active' => 'boolean',
                'sample_data' => 'nullable|array'
            ]);

            // Handle custom table name
            if ($request->has('target_table_select') && $request->target_table_select === '__custom__') {
                $validated['target_table'] = $request->custom_table_name;
                
                // Validate custom table name
                if (empty($request->custom_table_name)) {
                    return redirect()->back()
                        ->withInput()
                        ->with('error', 'Nama tabel custom harus diisi.');
                }
                
                // Validate table name format
                if (!preg_match('/^[a-zA-Z_][a-zA-Z0-9_]*$/', $request->custom_table_name)) {
                    return redirect()->back()
                        ->withInput()
                        ->with('error', 'Nama tabel hanya boleh menggunakan huruf, angka, dan underscore. Harus dimulai dengan huruf atau underscore.');
                }
            }

            // Convert checkbox values to proper booleans
            $validated['require_token'] = $request->has('require_token');
            $validated['auto_create_table'] = $request->has('auto_create_table');
            $validated['is_active'] = $request->has('is_active');

            // Validate API client requirement
            if ($validated['require_token'] && !$validated['api_client_id']) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'API Client harus dipilih jika token diperlukan.');
            }

            $postOnlyReceiver->update($validated);

            return redirect()->route('post-only-receivers.index')
                ->with('success', 'POST-Only Receiver berhasil diperbarui.');

        } catch (\Exception $e) {
            Log::error('PostOnlyReceiver update failed', [
                'error' => $e->getMessage(),
                'request_data' => $request->all()
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Gagal memperbarui POST-Only Receiver: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(PostOnlyReceiver $postOnlyReceiver)
    {
        try {
            $postOnlyReceiver->delete();

            return redirect()->route('post-only-receivers.index')
                ->with('success', 'POST-Only Receiver berhasil dihapus.');

        } catch (\Exception $e) {
            Log::error('PostOnlyReceiver deletion failed', [
                'error' => $e->getMessage(),
                'receiver_id' => $postOnlyReceiver->id
            ]);

            return redirect()->back()
                ->with('error', 'Gagal menghapus POST-Only Receiver: ' . $e->getMessage());
        }
    }

    /**
     * Handle POST data reception
     */
    public function receive(Request $request, $endpoint)
    {
        $startTime = microtime(true);
        
        // Find the receiver
        $receiver = PostOnlyReceiver::where('endpoint', $endpoint)
            ->where('is_active', true)
            ->first();

        if (!$receiver) {
            return $this->logAndReturnError($request, null, 'Receiver tidak ditemukan', 404, $startTime);
        }

        // Check rate limiting
        $rateLimitResult = $this->checkRateLimit($request, $receiver);
        if (!$rateLimitResult['success']) {
            return $this->logAndReturnError($request, $receiver, $rateLimitResult['message'], 429, $startTime);
        }

        // Check IP whitelist
        if ($receiver->allowed_ips && !in_array($request->ip(), $receiver->allowed_ips)) {
            return $this->logAndReturnError($request, $receiver, 'IP tidak diizinkan', 403, $startTime);
        }

        // Validate token
        $token = $this->extractToken($request);
        $tokenResult = $receiver->validateToken($token);
        
        if (!$tokenResult['success']) {
            return $this->logAndReturnError($request, $receiver, $tokenResult['message'], $tokenResult['code'], $startTime, false);
        }

        // Get POST data
        $data = $request->all();
        $payloadSize = strlen(json_encode($data));

        // Validate data
        $validationResult = $receiver->validateData($data);
        if (!$validationResult['success']) {
            return $this->logAndReturnError($request, $receiver, $validationResult['message'], 400, $startTime, true, false, $validationResult['errors'] ?? null);
        }

        // Save to target table
        $saveResult = $receiver->saveToTargetTable($validationResult['data']);
        if (!$saveResult['success']) {
            return $this->logAndReturnError($request, $receiver, $saveResult['message'], 500, $startTime, true, true, null, $saveResult['message']);
        }

        // Update statistics
        $receiver->incrementReceived();
        $receiver->incrementSuccess();

        // Log successful request
        $this->logRequest($request, $receiver, 200, 'Data berhasil diterima', $startTime, $payloadSize, true, true, true, null, null, $tokenResult['client'], $tokenResult['token']);

        // Return only success status
        return response()->json([
            'success' => true
        ], 200);
    }

    /**
     * Extract token from request
     */
    protected function extractToken(Request $request)
    {
        $authHeader = $request->header('Authorization');
        if ($authHeader && str_starts_with($authHeader, 'Bearer ')) {
            return substr($authHeader, 7);
        }
        
        return $request->header('X-API-Key') ?? $request->query('token');
    }

    /**
     * Check rate limiting
     */
    protected function checkRateLimit(Request $request, PostOnlyReceiver $receiver)
    {
        if (!$receiver->rate_limit) {
            return ['success' => true];
        }

        $key = 'rate_limit:post_receiver:' . $receiver->id . ':' . $request->ip();
        $attempts = Cache::get($key, 0);

        if ($attempts >= $receiver->rate_limit) {
            return [
                'success' => false,
                'message' => 'Rate limit terlampaui. Maksimal ' . $receiver->rate_limit . ' requests per jam.'
            ];
        }

        Cache::put($key, $attempts + 1, now()->addHour());
        return ['success' => true];
    }

    /**
     * Log request and return error response
     */
    protected function logAndReturnError(Request $request, $receiver, $message, $code, $startTime, $tokenValid = false, $validationSuccess = false, $validationErrors = null, $saveError = null)
    {
        $responseTime = microtime(true) - $startTime;
        $payloadSize = strlen(json_encode($request->all()));

        if ($receiver) {
            $this->logRequest($request, $receiver, $code, $message, $startTime, $payloadSize, $tokenValid, $validationSuccess, false, $validationErrors, $saveError);
            $receiver->incrementReceived();
            $receiver->incrementFailed();
        }

        return response()->json([
            'success' => false,
            'error' => $message,
            'code' => $code,
            'timestamp' => now()->toISOString()
        ], $code);
    }

    /**
     * Log request
     */
    protected function logRequest(Request $request, PostOnlyReceiver $receiver, $statusCode, $message, $startTime, $payloadSize, $tokenValid = false, $validationSuccess = false, $savedToTable = false, $validationErrors = null, $saveError = null, $client = null, $token = null)
    {
        $responseTime = microtime(true) - $startTime;

        PostOnlyReceiveLog::create([
            'receiver_id' => $receiver->id,
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'headers' => $request->headers->all(),
            'body' => json_encode($request->all()),
            'status_code' => $statusCode,
            'response_message' => $message,
            'response_time' => $responseTime,
            'payload_size' => $payloadSize,
            'token_valid' => $tokenValid,
            'validation_success' => $validationSuccess,
            'validation_errors' => $validationErrors,
            'saved_to_table' => $savedToTable,
            'save_error' => $saveError,
            'client_id' => $client ? $client->id : null,
            'token_id' => $token ? $token->id : null
        ]);
    }

    /**
     * Show logs for a specific receiver
     */
    public function logs(PostOnlyReceiver $postOnlyReceiver)
    {
        $logs = $postOnlyReceiver->receiveLogs()
            ->with(['apiClient', 'apiToken'])
            ->latest()
            ->paginate(50);

        return view('post-only-receivers.logs', compact('postOnlyReceiver', 'logs'));
    }

    /**
     * Show receiver statistics
     */
    public function stats(PostOnlyReceiver $postOnlyReceiver)
    {
        $stats = [
            'total_received' => $postOnlyReceiver->total_received,
            'total_success' => $postOnlyReceiver->total_success,
            'total_failed' => $postOnlyReceiver->total_failed,
            'success_rate' => $postOnlyReceiver->success_rate,
            'last_received' => $postOnlyReceiver->last_received_at,
            'recent_logs' => $postOnlyReceiver->receiveLogs()
                ->latest()
                ->take(10)
                ->get()
        ];

        return response()->json($stats);
    }

    /**
     * Get database tables
     */
    private function getDatabaseTables()
    {
        try {
            $tables = DB::select('SHOW TABLES');
            $tableNames = [];
            
            foreach ($tables as $table) {
                $tableArray = (array) $table;
                $tableNames[] = array_values($tableArray)[0];
            }
            
            return $tableNames;
        } catch (\Exception $e) {
            return [];
        }
    }
} 