<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\ApiSource;
use App\Models\ApiTarget;
use App\Models\ApiReceiver;
use App\Models\ApiLog;
use App\Models\ApiReceiveLog;
use App\Models\ApiSendLog;

class SystemController extends Controller
{
    public function stats()
    {
        try {
            $stats = [
                'api_count' => ApiSource::count(),
                'request_count' => ApiLog::count() + ApiReceiveLog::count() + ApiSendLog::count(),
                'uptime' => '24h', // You can implement actual uptime calculation
                'active_sources' => ApiSource::where('status', 'active')->count(),
                'active_targets' => ApiTarget::where('is_active', true)->count(),
                'active_receivers' => ApiReceiver::where('is_active', true)->count(),
                'total_endpoints' => ApiSource::count() + ApiTarget::count() + ApiReceiver::count(),
                'success_rate' => $this->calculateSuccessRate(),
                'system_status' => 'online',
                'memory_usage' => $this->getMemoryUsage(),
                'php_version' => PHP_VERSION,
                'laravel_version' => app()->version(),
                'last_updated' => now()->toISOString(),
            ];

            return response()->json($stats);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to retrieve system statistics',
                'message' => $e->getMessage(),
                'api_count' => 0,
                'request_count' => 0,
                'uptime' => '0h',
                'system_status' => 'error'
            ], 500);
        }
    }

    private function calculateSuccessRate()
    {
        try {
            $totalRequests = ApiLog::count();
            if ($totalRequests === 0) {
                return 100;
            }
            
            $successfulRequests = ApiLog::where('response_status', '>=', 200)
                                        ->where('response_status', '<', 300)
                                        ->count();
            
            return round(($successfulRequests / $totalRequests) * 100, 2);
        } catch (\Exception $e) {
            return 0;
        }
    }

    private function getMemoryUsage()
    {
        try {
            $bytes = memory_get_usage(true);
            $units = ['B', 'KB', 'MB', 'GB'];
            
            for ($i = 0; $bytes > 1024; $i++) {
                $bytes /= 1024;
            }
            
            return round($bytes, 2) . ' ' . $units[$i];
        } catch (\Exception $e) {
            return 'Unknown';
        }
    }

    public function health()
    {
        try {
            $health = [
                'status' => 'healthy',
                'database' => $this->checkDatabaseConnection(),
                'storage' => $this->checkStorageWritable(),
                'cache' => $this->checkCacheConnection(),
                'timestamp' => now()->toISOString(),
            ];

            $allHealthy = collect($health)->except(['status', 'timestamp'])->every(fn($status) => $status === 'ok');
            $health['status'] = $allHealthy ? 'healthy' : 'degraded';

            return response()->json($health);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'unhealthy',
                'error' => $e->getMessage(),
                'timestamp' => now()->toISOString(),
            ], 500);
        }
    }

    private function checkDatabaseConnection()
    {
        try {
            DB::connection()->getPdo();
            return 'ok';
        } catch (\Exception $e) {
            return 'error';
        }
    }

    private function checkStorageWritable()
    {
        try {
            $testFile = storage_path('app/health_check.txt');
            file_put_contents($testFile, 'test');
            unlink($testFile);
            return 'ok';
        } catch (\Exception $e) {
            return 'error';
        }
    }

    private function checkCacheConnection()
    {
        try {
            cache()->put('health_check', 'test', 1);
            cache()->forget('health_check');
            return 'ok';
        } catch (\Exception $e) {
            return 'error';
        }
    }
} 