<?php

namespace App\Http\Controllers;

use App\Models\Transceiver;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;

class TransceiverController extends Controller
{
    public function index()
    {
        $transceivers = Transceiver::latest()->paginate(10);
        return view('api-transceivers.index', compact('transceivers'));
    }

    public function create()
    {
        $tables = collect(DB::select('SHOW TABLES'))
            ->map(function ($table) {
                return array_values((array) $table)[0];
            })
            ->filter(fn($t) => !in_array($t, ['migrations', 'cache', 'sessions', 'cache_locks']))
            ->values()->toArray();
        return view('api-transceivers.create', compact('tables'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'endpoint' => 'required|string',
            'method' => 'required|in:POST,PUT',
            'source_table' => 'required|string|max:255',
            'auth_type' => 'required|in:none,api_key,bearer_token,basic_auth',
            'auth_token' => 'nullable|string',
            'auth_username' => 'nullable|string',
            'auth_password' => 'nullable|string',
            'is_active' => 'boolean'
        ]);
        $validated['created_by'] = auth()->id();
        $validated['is_active'] = $request->has('is_active');
        Transceiver::create($validated);
        return redirect()->route('transceivers.index')->with('success', 'Transceiver created successfully.');
    }

    public function show(Transceiver $transceiver)
    {
        return view('api-transceivers.show', compact('transceiver'));
    }

    public function edit(Transceiver $transceiver)
    {
        $tables = collect(DB::select('SHOW TABLES'))
            ->map(function ($table) {
                return array_values((array) $table)[0];
            })
            ->filter(fn($t) => !in_array($t, ['migrations', 'cache', 'sessions', 'cache_locks']))
            ->values()->toArray();
        return view('api-transceivers.edit', compact('transceiver', 'tables'));
    }

    public function update(Request $request, Transceiver $transceiver)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'endpoint' => 'required|string',
            'method' => 'required|in:POST,PUT',
            'source_table' => 'required|string|max:255',
            'auth_type' => 'required|in:none,api_key,bearer_token,basic_auth',
            'auth_token' => 'nullable|string',
            'auth_username' => 'nullable|string',
            'auth_password' => 'nullable|string',
            'is_active' => 'boolean'
        ]);
        $validated['is_active'] = $request->has('is_active');
        $transceiver->update($validated);
        return redirect()->route('transceivers.index')->with('success', 'Transceiver updated successfully.');
    }

    public function destroy(Transceiver $transceiver)
    {
        $transceiver->delete();
        return redirect()->route('transceivers.index')->with('success', 'Transceiver deleted successfully.');
    }

    public function send(Request $request, $id)
    {
        $transceiver = Transceiver::findOrFail($id);
        if (!$transceiver->is_active) {
            return response()->json(['success' => false, 'message' => 'Transceiver is not active'], 403);
        }
        // Endpoint tujuan (harus diisi di field endpoint)
        $url = $transceiver->endpoint;
        $method = strtoupper($transceiver->method);
        if (!in_array($request->method(), ['POST', 'PUT'])) {
            return response()->json(['success' => false, 'message' => 'Only POST and PUT allowed'], 405);
        }
        // Headers
        $headers = [
            'Content-Type' => 'application/json',
        ];
        // Auth
        if ($transceiver->auth_type === 'api_key' && $transceiver->auth_token) {
            $headers['X-API-KEY'] = $transceiver->auth_token;
        } elseif ($transceiver->auth_type === 'bearer_token' && $transceiver->auth_token) {
            $headers['Authorization'] = 'Bearer ' . $transceiver->auth_token;
        } elseif ($transceiver->auth_type === 'basic_auth' && $transceiver->auth_username && $transceiver->auth_password) {
            $basic = base64_encode($transceiver->auth_username . ':' . $transceiver->auth_password);
            $headers['Authorization'] = 'Basic ' . $basic;
        }
        // Kirim data ke endpoint eksternal
        try {
            $response = Http::withHeaders($headers)
                ->timeout(30)
                ->send($request->method(), $url, [
                    'json' => $request->all()
                ]);
            return response()->json([
                'success' => $response->successful(),
                'message' => $response->successful() ? 'Data sent successfully' : 'Failed to send data',
                'external_response' => $response->json(),
                'status' => $response->status(),
            ], $response->status());
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error sending data: ' . $e->getMessage(),
            ], 500);
        }
    }
} 