<?php

namespace App\Jobs;

use App\Models\ApiCallbackLog;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ProcessCallbackRetries implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Create a new job instance.
     */
    public function __construct()
    {
        //
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        try {
            // Get failed callbacks that need retry
            $failedCallbacks = ApiCallbackLog::where('status', 'failed')
                ->where('retry_count', '<', 'max_retries')
                ->where('next_retry_at', '<=', now())
                ->limit(50)
                ->get();

            if ($failedCallbacks->isEmpty()) {
                Log::info('No callback retries needed');
                return;
            }

            Log::info('Processing callback retries', ['count' => $failedCallbacks->count()]);

            foreach ($failedCallbacks as $callbackLog) {
                $this->retryCallback($callbackLog);
            }

        } catch (\Exception $e) {
            Log::error('Error processing callback retries', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Retry a failed callback
     */
    protected function retryCallback(ApiCallbackLog $callbackLog)
    {
        try {
            $callbackLog->update(['status' => 'retrying']);

            $payload = json_decode($callbackLog->payload_sent, true);
            $headers = $callbackLog->headers_sent ?? [];

            $startTime = microtime(true);
            $response = \Illuminate\Support\Facades\Http::withHeaders($headers)
                ->timeout(30)
                ->{strtolower($callbackLog->method)}($callbackLog->callback_url, $payload);

            $responseTime = (microtime(true) - $startTime) * 1000;

            if ($response->successful()) {
                $callbackLog->markAsSuccess(
                    $response->status(),
                    $response->body(),
                    $response->headers(),
                    $responseTime
                );

                $receiver = $callbackLog->apiReceiver;
                $receiver->increment('total_callbacks_sent');
                $receiver->update(['last_callback_at' => now()]);

                Log::info('Callback retry successful', [
                    'callback_log_id' => $callbackLog->id,
                    'retry_count' => $callbackLog->retry_count,
                    'response_code' => $response->status()
                ]);
            } else {
                $this->handleRetryFailure($callbackLog, $response->status(), $response->body());
            }

        } catch (\Exception $e) {
            $this->handleRetryException($callbackLog, $e);
        }
    }

    /**
     * Handle retry failure
     */
    protected function handleRetryFailure(ApiCallbackLog $callbackLog, int $statusCode, string $responseBody)
    {
        if ($callbackLog->retry_count + 1 < $callbackLog->max_retries) {
            // Schedule another retry
            $callbackLog->update([
                'status' => 'failed',
                'retry_count' => $callbackLog->retry_count + 1,
                'next_retry_at' => now()->addSeconds(($callbackLog->retry_count + 1) * 60), // Exponential backoff
                'error_message' => "HTTP {$statusCode} response from callback URL",
                'error_details' => $responseBody,
                'response_code' => $statusCode
            ]);

            Log::warning('Callback retry failed, will retry again', [
                'callback_log_id' => $callbackLog->id,
                'retry_count' => $callbackLog->retry_count,
                'next_retry_at' => $callbackLog->next_retry_at,
                'status_code' => $statusCode
            ]);
        } else {
            // Max retries reached
            $callbackLog->markAsFailed(
                "All retry attempts failed. Last HTTP {$statusCode}",
                $responseBody,
                $statusCode
            );
            
            $callbackLog->apiReceiver->increment('total_callbacks_failed');

            Log::error('Callback retry exhausted', [
                'callback_log_id' => $callbackLog->id,
                'final_status_code' => $statusCode,
                'total_retries' => $callbackLog->retry_count
            ]);
        }
    }

    /**
     * Handle retry exception
     */
    protected function handleRetryException(ApiCallbackLog $callbackLog, \Exception $e)
    {
        if ($callbackLog->retry_count + 1 < $callbackLog->max_retries) {
            // Schedule another retry
            $callbackLog->update([
                'status' => 'failed',
                'retry_count' => $callbackLog->retry_count + 1,
                'next_retry_at' => now()->addSeconds(($callbackLog->retry_count + 1) * 60),
                'error_message' => $e->getMessage(),
                'error_details' => $e->getTraceAsString()
            ]);

            Log::warning('Callback retry exception, will retry again', [
                'callback_log_id' => $callbackLog->id,
                'retry_count' => $callbackLog->retry_count,
                'next_retry_at' => $callbackLog->next_retry_at,
                'error' => $e->getMessage()
            ]);
        } else {
            // Max retries reached
            $callbackLog->markAsFailed(
                "All retry attempts failed: " . $e->getMessage(),
                $e->getTraceAsString()
            );
            
            $callbackLog->apiReceiver->increment('total_callbacks_failed');

            Log::error('Callback retry exception exhausted', [
                'callback_log_id' => $callbackLog->id,
                'final_error' => $e->getMessage(),
                'total_retries' => $callbackLog->retry_count
            ]);
        }
    }
}
