<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ApiCallbackLog extends Model
{
    use HasFactory;

    protected $fillable = [
        'api_receiver_id',
        'api_receive_log_id',
        'api_client_id',
        'callback_url',
        'method',
        'headers_sent',
        'payload_sent',
        'payload_size',
        'response_code',
        'response_body',
        'response_headers',
        'response_time',
        'status',
        'retry_count',
        'max_retries',
        'next_retry_at',
        'error_message',
        'error_details',
        'sent_at',
        'completed_at'
    ];

    protected $casts = [
        'headers_sent' => 'array',
        'response_headers' => 'array',
        'payload_size' => 'integer',
        'response_code' => 'integer',
        'response_time' => 'float',
        'retry_count' => 'integer',
        'max_retries' => 'integer',
        'next_retry_at' => 'datetime',
        'sent_at' => 'datetime',
        'completed_at' => 'datetime'
    ];

    public function apiReceiver()
    {
        return $this->belongsTo(ApiReceiver::class);
    }

    public function apiReceiveLog()
    {
        return $this->belongsTo(ApiReceiveLog::class);
    }

    public function apiClient()
    {
        return $this->belongsTo(ApiClient::class);
    }

    // Scopes
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeFailed($query)
    {
        return $query->where('status', 'failed');
    }

    public function scopeNeedsRetry($query)
    {
        return $query->where('status', 'failed')
                    ->where('retry_count', '<', 'max_retries')
                    ->where('next_retry_at', '<=', now());
    }

    // Helper methods
    public function canRetry()
    {
        return $this->status === 'failed' && 
               $this->retry_count < $this->max_retries;
    }

    public function markAsSuccess($responseCode, $responseBody = null, $responseHeaders = null, $responseTime = null)
    {
        $this->update([
            'status' => 'success',
            'response_code' => $responseCode,
            'response_body' => $responseBody,
            'response_headers' => $responseHeaders,
            'response_time' => $responseTime,
            'completed_at' => now()
        ]);
    }

    public function markAsFailed($errorMessage, $errorDetails = null, $responseCode = null)
    {
        $this->update([
            'status' => 'failed',
            'error_message' => $errorMessage,
            'error_details' => $errorDetails,
            'response_code' => $responseCode,
            'completed_at' => now()
        ]);
    }

    public function scheduleRetry()
    {
        if ($this->canRetry()) {
            $this->update([
                'status' => 'retrying',
                'retry_count' => $this->retry_count + 1,
                'next_retry_at' => now()->addSeconds($this->retry_count * 60) // Exponential backoff
            ]);
        }
    }
}
