<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ApiReceiver extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'description',
        'endpoint',
        'target_table',
        'auth_type',
        'auth_token',
        'auth_secret',
        'auth_username',
        'auth_password',
        'allowed_methods',
        'field_mapping',
        'rate_limit',
        'auto_create_table',
        'is_active',
        'allowed_ips',
        'allowed_headers',
        'total_received',
        'sample_data',
        'validation_rules',
        // Data processing fields
        'duplicate_handling',
        'unique_fields',
        'data_filter',
        'max_records_per_request',
        // Scheduling fields
        'enable_scheduling',
        'schedule_type',
        'schedule_interval',
        'schedule_time',
        'active_hours_start',
        'active_hours_end',
        'timezone',
        'last_processed_at',
        'total_processed',
        'total_skipped',
        'total_errors',
        // Bidirectional support
        'is_bidirectional',
        'response_format',
        'success_response_template',
        'error_response_template',
        'api_client_id',
        'callback_url',
        'callback_method',
        'auto_acknowledge',
        'retry_attempts',
        'retry_delay',
        'total_callbacks_sent',
        'total_callbacks_failed',
        'last_callback_at',
        'last_received_at'
    ];

    protected $casts = [
        'field_mapping' => 'array',
        'is_active' => 'boolean',
        'auto_create_table' => 'boolean',
        'allowed_ips' => 'array',
        'allowed_headers' => 'array',
        'sample_data' => 'array',
        'validation_rules' => 'array',
        'total_received' => 'integer',
        'rate_limit' => 'integer',
        'last_received_at' => 'datetime',
        // Data processing casts
        'enable_scheduling' => 'boolean',
        'schedule_interval' => 'integer',
        'schedule_time' => 'datetime:H:i',
        'active_hours_start' => 'datetime:H:i',
        'active_hours_end' => 'datetime:H:i',
        'last_processed_at' => 'datetime',
        'total_processed' => 'integer',
        'total_skipped' => 'integer',
        'total_errors' => 'integer',
        // Bidirectional casts
        'is_bidirectional' => 'boolean',
        'auto_acknowledge' => 'boolean',
        'retry_attempts' => 'integer',
        'retry_delay' => 'integer',
        'total_callbacks_sent' => 'integer',
        'total_callbacks_failed' => 'integer',
        'last_callback_at' => 'datetime'
    ];

    public function receiveLogs()
    {
        return $this->hasMany(ApiReceiveLog::class, 'receiver_id');
    }

    public function callbackLogs()
    {
        return $this->hasMany(ApiCallbackLog::class);
    }

    public function apiClient()
    {
        return $this->belongsTo(ApiClient::class);
    }

    public function transceivers()
    {
        return $this->hasMany(ApiTransceiver::class, 'api_receiver_id');
    }

    /**
     * Get validation rules for GET parameters
     */
    public function getValidationRules()
    {
        if (!$this->validation_rules) {
            return [];
        }

        return is_string($this->validation_rules) 
            ? json_decode($this->validation_rules, true) 
            : $this->validation_rules;
    }

    /**
     * Check if receiver requires client token synchronization
     */
    public function requiresClientSync()
    {
        return $this->api_client_id && $this->auth_type !== 'none';
    }

    /**
     * Return true jika receiver hanya mengizinkan GET
     */
    public function isGetOnly()
    {
        $methods = is_array($this->allowed_methods)
            ? $this->allowed_methods
            : explode(',', $this->allowed_methods);
        $methods = array_map('trim', $methods);
        return count($methods) === 1 && strtoupper($methods[0]) === 'GET';
    }
} 