<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ApiToken extends Model
{
    use HasFactory;

    protected $fillable = [
        'api_client_id',
        'name',
        'token',
        'refresh_token',
        'expires_at',
        'scope',
        'is_revoked',
        'last_used_at',
        // virtual mass-assignable attribute mapped via mutator
        'abilities',
    ];

    protected $casts = [
        'is_revoked' => 'boolean',
        'expires_at' => 'datetime',
        'last_used_at' => 'datetime'
    ];

    /**
     * Accessor: expose "abilities" as an array derived from scope
     */
    public function getAbilitiesAttribute(): array
    {
        $scope = $this->attributes['scope'] ?? null;
        if ($scope === null || $scope === '' || $scope === 'default') {
            return ['*'];
        }

        // If already JSON array
        if (is_string($scope) && str_starts_with($scope, '[')) {
            $decoded = json_decode($scope, true);
            if (is_array($decoded)) {
                return $decoded;
            }
        }

        // Comma separated fallback
        if (is_string($scope)) {
            $parts = array_values(array_filter(array_map('trim', explode(',', $scope)), fn ($v) => $v !== ''));
            return count($parts) ? $parts : ['*'];
        }

        return ['*'];
    }

    /**
     * Mutator: when setting "abilities", persist into "scope"
     */
    public function setAbilitiesAttribute($value): void
    {
        // Normalize to array of strings
        $abilities = $value;
        if (is_string($abilities)) {
            $abilities = array_values(array_filter(array_map('trim', explode(',', $abilities)), fn ($v) => $v !== ''));
        }
        if (!is_array($abilities)) {
            $abilities = ['*'];
        }

        // If full access only, keep compact
        if (count($abilities) === 1 && $abilities[0] === '*') {
            $this->attributes['scope'] = '*';
            return;
        }

        // Store as comma-separated for compatibility
        $this->attributes['scope'] = implode(',', $abilities);
    }

    public function client()
    {
        return $this->belongsTo(ApiClient::class, 'api_client_id');
    }
} 