<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ApiTransceiverLog extends Model
{
    use HasFactory;

    protected $fillable = [
        'api_transceiver_id',
        'api_receiver_log_id',
        'api_send_log_id',
        'source_data',
        'transformed_data',
        'target_response',
        'response_time',
        'success',
        'error_message',
        'processed_at'
    ];

    protected $casts = [
        'source_data' => 'array',
        'transformed_data' => 'array',
        'target_response' => 'array',
        'success' => 'boolean',
        'processed_at' => 'datetime'
    ];

    /**
     * Relationship with API transceiver
     */
    public function apiTransceiver(): BelongsTo
    {
        return $this->belongsTo(ApiTransceiver::class);
    }

    /**
     * Relationship with API receiver log
     */
    public function apiReceiveLog(): BelongsTo
    {
        return $this->belongsTo(ApiReceiveLog::class, 'api_receiver_log_id');
    }

    /**
     * Relationship with API send log
     */
    public function apiSendLog(): BelongsTo
    {
        return $this->belongsTo(ApiSendLog::class, 'api_send_log_id');
    }

    /**
     * Log transceiver operation
     */
    public static function logOperation(
        ApiTransceiver $transceiver,
        ?ApiReceiveLog $receiveLog,
        ?ApiSendLog $sendLog,
        array $sourceData,
        array $transformedData,
        $targetResponse,
        int $responseTime,
        bool $success,
        ?string $errorMessage = null
    ): self {
        return self::create([
            'api_transceiver_id' => $transceiver->id,
            'api_receiver_log_id' => $receiveLog?->id,
            'api_send_log_id' => $sendLog?->id,
            'source_data' => $sourceData,
            'transformed_data' => $transformedData,
            'target_response' => $targetResponse,
            'response_time' => $responseTime,
            'success' => $success,
            'error_message' => $errorMessage,
            'processed_at' => now()
        ]);
    }

    /**
     * Scope for successful operations
     */
    public function scopeSuccessful($query)
    {
        return $query->where('success', true);
    }

    /**
     * Scope for failed operations
     */
    public function scopeFailed($query)
    {
        return $query->where('success', false);
    }

    /**
     * Scope for specific transceiver
     */
    public function scopeForTransceiver($query, int $transceiverId)
    {
        return $query->where('api_transceiver_id', $transceiverId);
    }

    /**
     * Scope for date range
     */
    public function scopeDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('processed_at', [$startDate, $endDate]);
    }
} 