<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class GetOnlyReceiver extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'description',
        'endpoint',
        'data_source',
        'source_table',
        'source_api_url',
        'source_api_headers',
        'custom_query',
        'api_token_id',
        'require_token',
        'allowed_tokens',
        'response_format',
        'response_fields',
        'default_filters',
        'allowed_filters',
        'default_limit',
        'max_limit',
        'enable_pagination',
        'rate_limit_per_minute',
        'rate_limit_per_hour',
        'allowed_ips',
        'enable_cors',
        'cors_origins',
        'enable_cache',
        'cache_duration',
        'cache_key_prefix',
        'validation_rules',
        'required_parameters',
        'optional_parameters',
        'is_active',
        'total_requests',
        'total_success',
        'total_errors',
        'last_accessed_at',
        'avg_response_time',
        'metadata',
        'created_by'
    ];

    protected $casts = [
        'source_api_headers' => 'array',
        'allowed_tokens' => 'array',
        'response_fields' => 'array',
        'default_filters' => 'array',
        'allowed_filters' => 'array',
        'allowed_ips' => 'array',
        'cors_origins' => 'array',
        'validation_rules' => 'array',
        'required_parameters' => 'array',
        'optional_parameters' => 'array',
        'metadata' => 'array',
        'require_token' => 'boolean',
        'enable_pagination' => 'boolean',
        'enable_cors' => 'boolean',
        'enable_cache' => 'boolean',
        'is_active' => 'boolean',
        'last_accessed_at' => 'datetime',
        'avg_response_time' => 'decimal:3'
    ];

    /**
     * Relasi ke ApiToken
     */
    public function apiToken()
    {
        return $this->belongsTo(ApiToken::class);
    }

    /**
     * Get semua token yang diizinkan (bukan relationship)
     */
    public function getAllowedApiTokens()
    {
        if (!$this->allowed_tokens) {
            return collect();
        }
        
        return ApiToken::whereIn('id', $this->allowed_tokens)->get();
    }

    /**
     * Relationship ke allowed tokens (many-to-many through JSON)
     */
    public function allowedTokens()
    {
        return $this->belongsToMany(ApiToken::class, null, null, null, null, null, 'allowed_tokens');
    }

    /**
     * Cek apakah token valid untuk receiver ini
     */
    public function isTokenValid($token)
    {
        if (!$this->require_token) {
            return true;
        }

        // Cek token utama
        if ($this->api_token_id && $this->apiToken && $this->apiToken->token === $token) {
            return true;
        }

        // Cek token yang diizinkan
        if ($this->allowed_tokens) {
            $allowedTokens = ApiToken::whereIn('id', $this->allowed_tokens)
                ->where('token', $token)
                ->where('is_revoked', false)
                ->where(function($query) {
                    $query->whereNull('expires_at')
                          ->orWhere('expires_at', '>', now());
                })
                ->exists();
            
            if ($allowedTokens) {
                return true;
            }
        }

        return false;
    }

    /**
     * Generate cache key untuk response
     */
    public function generateCacheKey($parameters = [])
    {
        $prefix = $this->cache_key_prefix ?: 'get_receiver_';
        $paramString = http_build_query($parameters);
        return $prefix . $this->id . '_' . md5($paramString);
    }

    /**
     * Get data berdasarkan konfigurasi receiver
     */
    public function getData($parameters = [], $limit = null, $offset = 0)
    {
        $limit = $limit ?: $this->default_limit;
        $limit = min($limit, $this->max_limit);

        // Cek cache jika enabled
        if ($this->enable_cache) {
            $cacheKey = $this->generateCacheKey(array_merge($parameters, ['limit' => $limit, 'offset' => $offset]));
            $cached = Cache::get($cacheKey);
            if ($cached) {
                return $cached;
            }
        }

        $data = [];
        $total = 0;

        try {
            switch ($this->data_source) {
                case 'table':
                    $result = $this->getDataFromTable($parameters, $limit, $offset);
                    break;
                case 'api':
                    $result = $this->getDataFromApi($parameters, $limit, $offset);
                    break;
                case 'custom':
                    $result = $this->getDataFromCustomQuery($parameters, $limit, $offset);
                    break;
                default:
                    throw new \Exception('Invalid data source');
            }

            $data = $result['data'];
            $total = $result['total'];

            // Filter response fields jika ditentukan
            if ($this->response_fields) {
                $data = $this->filterResponseFields($data);
            }

            $response = [
                'data' => $data,
                'total' => $total,
                'limit' => $limit,
                'offset' => $offset,
                'has_more' => ($offset + $limit) < $total
            ];

            // Cache response jika enabled
            if ($this->enable_cache) {
                Cache::put($cacheKey, $response, $this->cache_duration);
            }

            return $response;

        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Get data dari table database
     */
    protected function getDataFromTable($parameters, $limit, $offset)
    {
        if (!$this->source_table) {
            throw new \Exception('Source table not specified');
        }

        $query = DB::table($this->source_table);

        // Apply default filters
        if ($this->default_filters) {
            foreach ($this->default_filters as $field => $value) {
                $query->where($field, $value);
            }
        }

        // Apply user filters
        $allowedFilters = $this->allowed_filters ?: [];
        foreach ($parameters as $key => $value) {
            if (in_array($key, $allowedFilters) && $value !== null && $value !== '') {
                if (is_array($value)) {
                    $query->whereIn($key, $value);
                } else {
                    $query->where($key, 'LIKE', "%{$value}%");
                }
            }
        }

        $total = $query->count();
        $data = $query->limit($limit)->offset($offset)->get()->toArray();

        return ['data' => $data, 'total' => $total];
    }

    /**
     * Get data dari API eksternal
     */
    protected function getDataFromApi($parameters, $limit, $offset)
    {
        if (!$this->source_api_url) {
            throw new \Exception('Source API URL not specified');
        }

        // Implementasi untuk fetch data dari API eksternal
        // Ini bisa disesuaikan dengan kebutuhan spesifik
        $headers = $this->source_api_headers ?: [];
        
        // Tambahkan parameter ke URL
        $url = $this->source_api_url;
        if (!empty($parameters)) {
            $url .= '?' . http_build_query(array_merge($parameters, [
                'limit' => $limit,
                'offset' => $offset
            ]));
        }

        // Fetch data (implementasi sederhana)
        $response = file_get_contents($url, false, stream_context_create([
            'http' => [
                'method' => 'GET',
                'header' => implode("\r\n", array_map(function($k, $v) {
                    return "$k: $v";
                }, array_keys($headers), $headers))
            ]
        ]));

        $data = json_decode($response, true);
        
        return [
            'data' => $data['data'] ?? $data,
            'total' => $data['total'] ?? count($data['data'] ?? $data)
        ];
    }

    /**
     * Get data dari custom query
     */
    protected function getDataFromCustomQuery($parameters, $limit, $offset)
    {
        if (!$this->custom_query) {
            throw new \Exception('Custom query not specified');
        }

        // Replace parameters dalam query
        $query = $this->custom_query;
        foreach ($parameters as $key => $value) {
            $query = str_replace(":{$key}", $value, $query);
        }

        // Add LIMIT dan OFFSET
        $query .= " LIMIT {$limit} OFFSET {$offset}";

        $data = DB::select($query);
        
        // Get total count (simplified)
        $countQuery = "SELECT COUNT(*) as total FROM ({$this->custom_query}) as subquery";
        foreach ($parameters as $key => $value) {
            $countQuery = str_replace(":{$key}", $value, $countQuery);
        }
        $totalResult = DB::select($countQuery);
        $total = $totalResult[0]->total ?? 0;

        return ['data' => $data, 'total' => $total];
    }

    /**
     * Filter response fields
     */
    protected function filterResponseFields($data)
    {
        if (!$this->response_fields || empty($data)) {
            return $data;
        }

        return array_map(function($item) {
            $filtered = [];
            foreach ($this->response_fields as $field) {
                if (isset($item[$field])) {
                    $filtered[$field] = $item[$field];
                }
            }
            return $filtered;
        }, $data);
    }

    /**
     * Update statistik
     */
    public function updateStats($responseTime, $success = true)
    {
        $this->increment('total_requests');
        
        if ($success) {
            $this->increment('total_success');
        } else {
            $this->increment('total_errors');
        }

        // Update average response time
        $newAvg = (($this->avg_response_time * ($this->total_requests - 1)) + $responseTime) / $this->total_requests;
        $this->update([
            'avg_response_time' => $newAvg,
            'last_accessed_at' => now()
        ]);
    }

    /**
     * Validate parameters
     */
    public function validateParameters($parameters)
    {
        $errors = [];

        // Check required parameters
        if ($this->required_parameters) {
            foreach ($this->required_parameters as $param) {
                if (!isset($parameters[$param]) || $parameters[$param] === null || $parameters[$param] === '') {
                    $errors[] = "Parameter '{$param}' is required";
                }
            }
        }

        // Validate using validation rules
        if ($this->validation_rules && !empty($parameters)) {
            $validator = Validator::make($parameters, $this->validation_rules);
            if ($validator->fails()) {
                $errors = array_merge($errors, $validator->errors()->all());
            }
        }

        return $errors;
    }
}
