<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

class PostOnlyReceiver extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'endpoint',
        'target_table',
        'description',
        'api_client_id',
        'require_token',
        'field_mapping',
        'validation_rules',
        'auto_create_table',
        'allowed_ips',
        'rate_limit',
        'is_active',
        'last_received_at',
        'total_received',
        'total_success',
        'total_failed',
        'sample_data'
    ];

    protected $casts = [
        'field_mapping' => 'array',
        'validation_rules' => 'array',
        'allowed_ips' => 'array',
        'sample_data' => 'array',
        'auto_create_table' => 'boolean',
        'require_token' => 'boolean',
        'is_active' => 'boolean',
        'last_received_at' => 'datetime',
        'total_received' => 'integer',
        'total_success' => 'integer',
        'total_failed' => 'integer',
        'rate_limit' => 'integer'
    ];

    /**
     * Relationship dengan API Client
     */
    public function apiClient()
    {
        return $this->belongsTo(ApiClient::class);
    }

    /**
     * Relationship dengan receive logs
     */
    public function receiveLogs()
    {
        return $this->hasMany(PostOnlyReceiveLog::class, 'receiver_id');
    }

    /**
     * Validasi token dari API client
     */
    public function validateToken($token)
    {
        if (!$this->require_token) {
            return ['success' => true, 'client' => null, 'token' => null];
        }

        if (!$token) {
            return [
                'success' => false,
                'message' => 'Token diperlukan untuk mengakses endpoint ini',
                'code' => 401
            ];
        }

        if (!$this->api_client_id) {
            return [
                'success' => false,
                'message' => 'Receiver tidak dikonfigurasi dengan API client',
                'code' => 500
            ];
        }

        $client = $this->apiClient;
        if (!$client || !$client->is_active) {
            return [
                'success' => false,
                'message' => 'API client tidak aktif',
                'code' => 403
            ];
        }

        // Check if client has expired
        if ($client->expires_at && $client->expires_at->isPast()) {
            return [
                'success' => false,
                'message' => 'API client telah expired',
                'code' => 403
            ];
        }

        // Validate token against client's tokens
        $validToken = $client->tokens()
            ->where('token', $token)
            ->where('is_revoked', false)
            ->where(function($query) {
                $query->whereNull('expires_at')
                      ->orWhere('expires_at', '>', now());
            })
            ->first();

        if (!$validToken) {
            return [
                'success' => false,
                'message' => 'Token tidak valid atau telah expired',
                'code' => 401
            ];
        }

        // Update token last used
        $validToken->update(['last_used_at' => now()]);
        $client->update(['last_used_at' => now()]);

        return [
            'success' => true,
            'client' => $client,
            'token' => $validToken
        ];
    }

    /**
     * Validasi data berdasarkan validation rules
     */
    public function validateData($data)
    {
        if (!$this->validation_rules) {
            return ['success' => true, 'data' => $data];
        }

        $rules = [];
        $messages = [];

        foreach ($this->validation_rules as $field => $fieldRules) {
            $rule = [];
            
            if (isset($fieldRules['required']) && $fieldRules['required']) {
                $rule[] = 'required';
            }
            
            if (isset($fieldRules['type'])) {
                switch ($fieldRules['type']) {
                    case 'string':
                        $rule[] = 'string';
                        break;
                    case 'integer':
                        $rule[] = 'integer';
                        break;
                    case 'numeric':
                        $rule[] = 'numeric';
                        break;
                    case 'email':
                        $rule[] = 'email';
                        break;
                    case 'date':
                        $rule[] = 'date';
                        break;
                    case 'boolean':
                        $rule[] = 'boolean';
                        break;
                    case 'array':
                        $rule[] = 'array';
                        break;
                }
            }
            
            if (isset($fieldRules['max_length'])) {
                $rule[] = 'max:' . $fieldRules['max_length'];
            }
            
            if (isset($fieldRules['min_length'])) {
                $rule[] = 'min:' . $fieldRules['min_length'];
            }
            
            if (!empty($rule)) {
                $rules[$field] = implode('|', $rule);
            }
        }

        try {
            $validator = validator($data, $rules, $messages);
            
            if ($validator->fails()) {
                return [
                    'success' => false,
                    'message' => 'Validasi gagal: ' . implode(', ', $validator->errors()->all()),
                    'errors' => $validator->errors()->toArray()
                ];
            }

            return ['success' => true, 'data' => $validator->validated()];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Error validasi: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Simpan data ke target table
     */
    public function saveToTargetTable($data)
    {
        try {
            // Check if table exists
            if (!Schema::hasTable($this->target_table)) {
                if ($this->auto_create_table && $this->sample_data) {
                    $this->createTargetTable();
                } else {
                    return [
                        'success' => false,
                        'message' => 'Target table tidak ditemukan: ' . $this->target_table
                    ];
                }
            }

            // Apply field mapping if configured
            if ($this->field_mapping) {
                $mappedData = [];
                foreach ($this->field_mapping as $sourceField => $targetField) {
                    if (isset($data[$sourceField])) {
                        $mappedData[$targetField] = $data[$sourceField];
                    }
                }
                $data = array_merge($data, $mappedData);
            }

            // Add timestamps
            $data['created_at'] = now();
            $data['updated_at'] = now();

            // Insert data
            DB::table($this->target_table)->insert($data);

            return ['success' => true, 'message' => 'Data berhasil disimpan'];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Gagal menyimpan data: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Create target table berdasarkan sample data
     */
    protected function createTargetTable()
    {
        if (!$this->sample_data) {
            throw new \Exception('Sample data tidak tersedia untuk membuat table');
        }

        Schema::create($this->target_table, function ($table) {
            $table->id();
            
            foreach ($this->sample_data as $field => $value) {
                $type = $this->getFieldType($value);
                
                switch ($type) {
                    case 'integer':
                        $table->integer($field)->nullable();
                        break;
                    case 'float':
                        $table->decimal($field, 10, 2)->nullable();
                        break;
                    case 'boolean':
                        $table->boolean($field)->nullable();
                        break;
                    case 'date':
                        $table->date($field)->nullable();
                        break;
                    case 'datetime':
                        $table->timestamp($field)->nullable();
                        break;
                    case 'json':
                        $table->json($field)->nullable();
                        break;
                    default:
                        $table->string($field, 500)->nullable();
                }
            }
            
            $table->timestamps();
        });
    }

    /**
     * Determine field type from sample value
     */
    protected function getFieldType($value)
    {
        if (is_int($value)) {
            return 'integer';
        } elseif (is_float($value)) {
            return 'float';
        } elseif (is_bool($value)) {
            return 'boolean';
        } elseif (is_array($value)) {
            return 'json';
        } elseif (is_string($value)) {
            // Check if it's a date
            if (strtotime($value) !== false) {
                if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $value)) {
                    return 'date';
                } elseif (preg_match('/^\d{4}-\d{2}-\d{2}[T\s]\d{2}:\d{2}:\d{2}/', $value)) {
                    return 'datetime';
                }
            }
            return 'string';
        }
        
        return 'string';
    }

    /**
     * Increment statistics
     */
    public function incrementReceived()
    {
        $this->increment('total_received');
        $this->update(['last_received_at' => now()]);
    }

    public function incrementSuccess()
    {
        $this->increment('total_success');
    }

    public function incrementFailed()
    {
        $this->increment('total_failed');
    }

    /**
     * Get success rate
     */
    public function getSuccessRateAttribute()
    {
        if ($this->total_received == 0) {
            return 0;
        }
        
        return round(($this->total_success / $this->total_received) * 100, 2);
    }
} 