<?php

require_once 'vendor/autoload.php';

// Bootstrap Laravel
$app = require_once 'bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

try {
    echo "🔍 Checking API Source ID 4...\n\n";
    
    // Find API source with ID 4
    $apiSource = App\Models\ApiSource::find(4);
    
    if (!$apiSource) {
        echo "❌ API Source with ID 4 not found.\n";
        echo "📋 Available API Sources:\n";
        $sources = App\Models\ApiSource::all();
        foreach ($sources as $source) {
            echo "   - ID: {$source->id}, Name: {$source->name}, Table: {$source->table_name}\n";
        }
        exit;
    }
    
    echo "✅ Found API Source ID 4:\n";
    echo "   - Name: {$apiSource->name}\n";
    echo "   - URL: {$apiSource->url}\n";
    echo "   - Table: {$apiSource->table_name}\n";
    echo "   - Status: {$apiSource->status}\n";
    echo "   - Last Fetched: " . ($apiSource->last_fetched_at ? $apiSource->last_fetched_at->format('Y-m-d H:i:s') : 'Never') . "\n\n";
    
    // Check existing logs
    $totalLogs = $apiSource->apiLogs()->count();
    $successLogs = $apiSource->apiLogs()->where('status', 'success')->count();
    $errorLogs = $apiSource->apiLogs()->where('status', 'error')->count();
    
    echo "📊 Current Logs Status:\n";
    echo "   - Total Logs: {$totalLogs}\n";
    echo "   - Success Logs: {$successLogs}\n";
    echo "   - Error Logs: {$errorLogs}\n\n";
    
    if ($successLogs == 0) {
        echo "⚠️  No successful import logs found. Creating test data...\n";
        
        // Create sample successful import logs with realistic data
        $sampleData = [
            [
                'id' => 1,
                'name' => 'Ahmad Dhani',
                'email' => 'ahmad@example.com',
                'phone' => '+62812345678',
                'address' => 'Jakarta, Indonesia',
                'status' => 'active',
                'created_at' => '2025-01-15 10:30:00',
                'updated_at' => '2025-01-15 10:30:00'
            ],
            [
                'id' => 2,
                'name' => 'Siti Nurhaliza',
                'email' => 'siti@example.com',
                'phone' => '+62812345679',
                'address' => 'Bandung, Indonesia',
                'status' => 'active',
                'created_at' => '2025-01-15 11:45:00',
                'updated_at' => '2025-01-15 11:45:00'
            ],
            [
                'id' => 3,
                'name' => 'Budi Santoso',
                'email' => 'budi@example.com',
                'phone' => '+62812345680',
                'address' => 'Surabaya, Indonesia',
                'status' => 'inactive',
                'created_at' => '2025-01-15 14:20:00',
                'updated_at' => '2025-01-15 14:20:00'
            ],
            [
                'id' => 4,
                'name' => 'Dewi Sartika',
                'email' => 'dewi@example.com',
                'phone' => '+62812345681',
                'address' => 'Yogyakarta, Indonesia',
                'status' => 'active',
                'created_at' => '2025-01-15 16:10:00',
                'updated_at' => '2025-01-15 16:10:00'
            ],
            [
                'id' => 5,
                'name' => 'Rudi Hartono',
                'email' => 'rudi@example.com',
                'phone' => '+62812345682',
                'address' => 'Medan, Indonesia',
                'status' => 'pending',
                'created_at' => '2025-01-15 18:30:00',
                'updated_at' => '2025-01-15 18:30:00'
            ]
        ];
        
        // Create multiple import logs to simulate different import sessions
        for ($i = 1; $i <= 3; $i++) {
            $importDate = now()->subDays($i);
            $recordsCount = count($sampleData);
            
            $log = App\Models\ApiLog::create([
                'api_source_id' => $apiSource->id,
                'endpoint' => $apiSource->url,
                'method' => 'GET',
                'status_code' => 200,
                'response_code' => 200,
                'status' => 'success',
                'response_time' => rand(100, 500),
                'response_body' => json_encode($sampleData),
                'records_imported' => $recordsCount,
                'requested_at' => $importDate,
                'created_at' => $importDate,
                'updated_at' => $importDate
            ]);
            
            echo "✅ Created import log #{$i} with ID: {$log->id} (Date: {$importDate->format('Y-m-d H:i:s')})\n";
        }
        
        // Update last_fetched_at
        $apiSource->update(['last_fetched_at' => now()]);
        
        echo "\n🎉 Test data created successfully!\n";
        echo "📋 Summary:\n";
        echo "   - Created 3 successful import logs\n";
        echo "   - Each log contains 5 sample records\n";
        echo "   - Total records: " . (count($sampleData) * 3) . "\n";
        echo "   - Updated last_fetched_at timestamp\n\n";
        
    } else {
        echo "✅ Found {$successLogs} successful import logs. Showing latest data:\n";
        
        $latestLog = $apiSource->apiLogs()->where('status', 'success')->latest()->first();
        if ($latestLog) {
            echo "   - Latest Import: {$latestLog->created_at->format('Y-m-d H:i:s')}\n";
            echo "   - Records: {$latestLog->records_imported}\n";
            echo "   - Response Time: {$latestLog->response_time}ms\n";
            
            $data = json_decode($latestLog->response_body, true);
            if (is_array($data)) {
                echo "   - Data Sample: " . (isset($data[0]) ? json_encode($data[0]) : json_encode($data)) . "\n";
            }
        }
    }
    
    echo "\n🌐 You can now visit: https://copilotproduction.site/api-sources/4/data\n";
    echo "🔄 The page should now show real imported data instead of sample data.\n";
    
} catch (Exception $e) {
    echo "❌ ERROR: " . $e->getMessage() . "\n";
    echo "📝 Stack trace:\n" . $e->getTraceAsString() . "\n";
} 