<?php
// Direct table creation for POST-Only Receivers using raw SQL
use Illuminate\Support\Facades\DB;

echo "<h2>🗄️ Creating Tables Directly with SQL</h2>";

try {
    // Database connection test
    echo "<h3>Step 1: Database Connection Test</h3>";
    
    $host = env('DB_HOST', 'localhost');
    $database = env('DB_DATABASE', 'copg1493_api');
    $username = env('DB_USERNAME', 'copg1493');
    $password = env('DB_PASSWORD', '');
    
    echo "<p><strong>Database:</strong> $database</p>";
    echo "<p><strong>Host:</strong> $host</p>";
    echo "<p><strong>Username:</strong> $username</p>";
    
    // Try Laravel DB connection first
    try {
        $pdo = DB::connection()->getPdo();
        echo "<p style='color: green;'>✅ Laravel DB connection: OK</p>";
    } catch (Exception $e) {
        echo "<p style='color: red;'>❌ Laravel DB connection failed: " . $e->getMessage() . "</p>";
        
        // Try direct PDO connection
        try {
            $pdo = new PDO("mysql:host=$host;dbname=$database", $username, $password);
            echo "<p style='color: green;'>✅ Direct PDO connection: OK</p>";
        } catch (Exception $e2) {
            echo "<p style='color: red;'>❌ Direct PDO connection failed: " . $e2->getMessage() . "</p>";
            throw new Exception("Cannot connect to database");
        }
    }
    
    // Step 2: Check current tables
    echo "<h3>Step 2: Current Tables Check</h3>";
    
    $tables = DB::select("SHOW TABLES LIKE 'post_only_%'");
    if (empty($tables)) {
        echo "<p style='color: orange;'>⚠️ No post_only_* tables found</p>";
    } else {
        echo "<p style='color: green;'>✅ Found existing post_only_* tables:</p>";
        foreach ($tables as $table) {
            $tableName = array_values((array)$table)[0];
            echo "<p>- $tableName</p>";
        }
    }
    
    // Step 3: Create post_only_receivers table
    echo "<h3>Step 3: Creating post_only_receivers Table</h3>";
    
    $createPostOnlyReceiversSQL = "
    CREATE TABLE IF NOT EXISTS `post_only_receivers` (
        `id` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        `name` varchar(255) NOT NULL COMMENT 'Nama receiver yang mudah dibaca',
        `endpoint` varchar(255) NOT NULL COMMENT 'URL path endpoint (contoh: webhook-data)',
        `target_table` varchar(255) NOT NULL COMMENT 'Nama tabel tujuan penyimpanan data',
        `description` text DEFAULT NULL COMMENT 'Deskripsi receiver',
        `api_client_id` bigint(20) unsigned DEFAULT NULL COMMENT 'ID API client yang terkait',
        `require_token` tinyint(1) NOT NULL DEFAULT 1 COMMENT 'Apakah memerlukan token untuk akses',
        `field_mapping` json DEFAULT NULL COMMENT 'Mapping field dari request ke database',
        `validation_rules` json DEFAULT NULL COMMENT 'Aturan validasi untuk data POST',
        `auto_create_table` tinyint(1) NOT NULL DEFAULT 0 COMMENT 'Auto create table dari sample data',
        `allowed_ips` json DEFAULT NULL COMMENT 'IP whitelist (JSON array)',
        `rate_limit` int(11) NOT NULL DEFAULT 1000 COMMENT 'Requests per hour limit',
        `is_active` tinyint(1) NOT NULL DEFAULT 1 COMMENT 'Status aktif receiver',
        `last_received_at` timestamp NULL DEFAULT NULL COMMENT 'Waktu terakhir menerima data',
        `total_received` int(11) NOT NULL DEFAULT 0 COMMENT 'Total data yang diterima',
        `total_success` int(11) NOT NULL DEFAULT 0 COMMENT 'Total data berhasil diproses',
        `total_failed` int(11) NOT NULL DEFAULT 0 COMMENT 'Total data gagal diproses',
        `sample_data` json DEFAULT NULL COMMENT 'Contoh data untuk dokumentasi',
        `created_at` timestamp NULL DEFAULT NULL,
        `updated_at` timestamp NULL DEFAULT NULL,
        PRIMARY KEY (`id`),
        UNIQUE KEY `post_only_receivers_endpoint_unique` (`endpoint`),
        KEY `post_only_receivers_endpoint_is_active_index` (`endpoint`,`is_active`),
        KEY `post_only_receivers_api_client_id_index` (`api_client_id`),
        KEY `post_only_receivers_target_table_index` (`target_table`),
        KEY `post_only_receivers_require_token_index` (`require_token`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    ";
    
    try {
        DB::statement($createPostOnlyReceiversSQL);
        echo "<p style='color: green;'>✅ post_only_receivers table created successfully</p>";
    } catch (Exception $e) {
        echo "<p style='color: red;'>❌ Failed to create post_only_receivers table: " . $e->getMessage() . "</p>";
    }
    
    // Step 4: Create post_only_receive_logs table
    echo "<h3>Step 4: Creating post_only_receive_logs Table</h3>";
    
    $createPostOnlyReceiveLogsSQL = "
    CREATE TABLE IF NOT EXISTS `post_only_receive_logs` (
        `id` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        `receiver_id` bigint(20) unsigned NOT NULL,
        `ip_address` varchar(45) DEFAULT NULL,
        `user_agent` text DEFAULT NULL,
        `headers` json DEFAULT NULL,
        `body` longtext DEFAULT NULL,
        `status_code` int(11) NOT NULL,
        `response_message` text DEFAULT NULL,
        `response_time` decimal(8,3) DEFAULT NULL COMMENT 'Response time in seconds',
        `payload_size` int(11) DEFAULT NULL COMMENT 'Payload size in bytes',
        `token_valid` tinyint(1) NOT NULL DEFAULT 0,
        `validation_success` tinyint(1) NOT NULL DEFAULT 0,
        `validation_errors` json DEFAULT NULL,
        `saved_to_table` tinyint(1) NOT NULL DEFAULT 0,
        `save_error` text DEFAULT NULL,
        `client_id` bigint(20) unsigned DEFAULT NULL,
        `token_id` bigint(20) unsigned DEFAULT NULL,
        `created_at` timestamp NULL DEFAULT NULL,
        `updated_at` timestamp NULL DEFAULT NULL,
        PRIMARY KEY (`id`),
        KEY `post_only_receive_logs_receiver_id_created_at_index` (`receiver_id`,`created_at`),
        KEY `post_only_receive_logs_status_code_index` (`status_code`),
        KEY `post_only_receive_logs_token_valid_index` (`token_valid`),
        KEY `post_only_receive_logs_validation_success_index` (`validation_success`),
        KEY `post_only_receive_logs_saved_to_table_index` (`saved_to_table`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    ";
    
    try {
        DB::statement($createPostOnlyReceiveLogsSQL);
        echo "<p style='color: green;'>✅ post_only_receive_logs table created successfully</p>";
    } catch (Exception $e) {
        echo "<p style='color: red;'>❌ Failed to create post_only_receive_logs table: " . $e->getMessage() . "</p>";
    }
    
    // Step 5: Verify tables were created
    echo "<h3>Step 5: Verifying Table Creation</h3>";
    
    $verifyTables = ['post_only_receivers', 'post_only_receive_logs'];
    
    foreach ($verifyTables as $table) {
        try {
            $result = DB::select("SELECT COUNT(*) as count FROM $table");
            echo "<p style='color: green;'>✅ Table <strong>$table</strong> exists and accessible (Records: " . $result[0]->count . ")</p>";
            
            // Show table structure
            $columns = DB::select("DESCRIBE $table");
            echo "<p><strong>Columns in $table:</strong></p>";
            echo "<ul>";
            foreach ($columns as $column) {
                echo "<li>{$column->Field} ({$column->Type})</li>";
            }
            echo "</ul>";
            
        } catch (Exception $e) {
            echo "<p style='color: red;'>❌ Table <strong>$table</strong> verification failed: " . $e->getMessage() . "</p>";
        }
    }
    
    // Step 6: Test PostOnlyReceiver model
    echo "<h3>Step 6: Testing PostOnlyReceiver Model</h3>";
    
    try {
        $count = \App\Models\PostOnlyReceiver::count();
        echo "<p style='color: green;'>✅ PostOnlyReceiver model working (Records: $count)</p>";
        
        // Try to create a test record
        $testReceiver = \App\Models\PostOnlyReceiver::create([
            'name' => 'Test Receiver',
            'endpoint' => 'test-endpoint-' . time(),
            'target_table' => 'test_table',
            'description' => 'Test receiver created by setup script',
            'require_token' => false,
            'is_active' => true
        ]);
        
        echo "<p style='color: green;'>✅ Test record created successfully (ID: {$testReceiver->id})</p>";
        
        // Delete test record
        $testReceiver->delete();
        echo "<p style='color: green;'>✅ Test record deleted successfully</p>";
        
    } catch (Exception $e) {
        echo "<p style='color: red;'>❌ PostOnlyReceiver model test failed: " . $e->getMessage() . "</p>";
    }
    
    echo "<hr>";
    echo "<h3 style='color: blue;'>🎯 Summary</h3>";
    echo "<p>Database tables have been created successfully using direct SQL.</p>";
    
} catch (Exception $e) {
    echo "<p style='color: red;'>❌ Error: " . $e->getMessage() . "</p>";
}

echo "<hr>";
echo "<h3>📋 Next Steps:</h3>";
echo "<ol>";
echo "<li>Try accessing: <a href='/post-only-receivers'>POST-Only Receivers</a></li>";
echo "<li>If still getting errors, check Laravel logs</li>";
echo "<li>Verify database permissions</li>";
echo "</ol>";

echo "<p><strong>Table creation completed at:</strong> " . date('Y-m-d H:i:s') . "</p>";
?> 