<?php

require_once 'vendor/autoload.php';

$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\ApiTarget;
use App\Models\User;
use Illuminate\Support\Facades\DB;

echo "=== CREATE API TARGET EXAMPLES ===\n\n";

try {
    // Get or create a user for created_by field
    $user = User::first();
    if (!$user) {
        echo "❌ No user found! Please create a user first.\n";
        exit(1);
    }
    
    echo "✓ Using user: {$user->name} (ID: {$user->id})\n\n";
    
    // Example 1: API Target with Bearer Token Authentication
    echo "1. Creating API Target with Bearer Token Authentication...\n";
    $bearerTarget = ApiTarget::create([
        'name' => 'HubNet Integration API',
        'description' => 'API target untuk integrasi dengan sistem HubNet menggunakan Bearer Token',
        'base_url' => 'https://api.hubnet.com',
        'endpoint' => '/v1/data/sync',
        'method' => 'POST',
        'source_table' => 'users',
        'auth_type' => 'bearer_token',
        'auth_token' => 'sk_live_51ABC123DEF456GHI789JKL012MNO345PQR678STU901VWX234YZA567BCD890',
        'auth_username' => null,
        'auth_password' => null,
        'schedule_type' => 'interval',
        'schedule_frequency' => 120, // Every 2 hours (120 minutes)
        'is_active' => true,
        'created_by' => $user->id
    ]);
    
    echo "✓ Bearer Token API Target created successfully!\n";
    echo "  - ID: {$bearerTarget->id}\n";
    echo "  - Name: {$bearerTarget->name}\n";
    echo "  - URL: {$bearerTarget->url}\n";
    echo "  - Auth: Bearer Token\n";
    echo "  - Schedule: Every 2 hours (interval)\n\n";
    
    // Example 2: API Target with API Key Authentication
    echo "2. Creating API Target with API Key Authentication...\n";
    $apiKeyTarget = ApiTarget::create([
        'name' => 'Payment Gateway API',
        'description' => 'API target untuk payment gateway menggunakan API Key',
        'base_url' => 'https://api.paymentgateway.com',
        'endpoint' => '/transactions',
        'method' => 'POST',
        'source_table' => 'orders',
        'auth_type' => 'api_key',
        'auth_token' => 'pk_live_1234567890abcdefghijklmnopqrstuvwxyz',
        'auth_username' => null,
        'auth_password' => null,
        'schedule_type' => 'manual',
        'schedule_frequency' => null,
        'is_active' => true,
        'created_by' => $user->id
    ]);
    
    echo "✓ API Key API Target created successfully!\n";
    echo "  - ID: {$apiKeyTarget->id}\n";
    echo "  - Name: {$apiKeyTarget->name}\n";
    echo "  - URL: {$apiKeyTarget->url}\n";
    echo "  - Auth: API Key\n";
    echo "  - Schedule: Manual\n\n";
    
    // Example 3: API Target with Basic Authentication
    echo "3. Creating API Target with Basic Authentication...\n";
    $basicAuthTarget = ApiTarget::create([
        'name' => 'Legacy System API',
        'description' => 'API target untuk sistem legacy menggunakan Basic Auth',
        'base_url' => 'https://legacy-api.company.com',
        'endpoint' => '/api/v1/users',
        'method' => 'PUT',
        'source_table' => 'users',
        'auth_type' => 'basic_auth',
        'auth_token' => null,
        'auth_username' => 'api_user',
        'auth_password' => 'secure_password_123',
        'schedule_type' => 'interval',
        'schedule_frequency' => 1440, // Every day (1440 minutes)
        'is_active' => true,
        'created_by' => $user->id
    ]);
    
    echo "✓ Basic Auth API Target created successfully!\n";
    echo "  - ID: {$basicAuthTarget->id}\n";
    echo "  - Name: {$basicAuthTarget->name}\n";
    echo "  - URL: {$basicAuthTarget->url}\n";
    echo "  - Auth: Basic Auth (username: api_user)\n";
    echo "  - Schedule: Daily (interval)\n\n";
    
    // Example 4: API Target with No Authentication (Public API)
    echo "4. Creating API Target with No Authentication...\n";
    $publicTarget = ApiTarget::create([
        'name' => 'Public Webhook',
        'description' => 'Webhook publik untuk notifikasi tanpa authentication',
        'base_url' => 'https://webhook.site',
        'endpoint' => '/your-unique-webhook-url',
        'method' => 'POST',
        'source_table' => 'notifications',
        'auth_type' => 'none',
        'auth_token' => null,
        'auth_username' => null,
        'auth_password' => null,
        'schedule_type' => 'interval',
        'schedule_frequency' => 10080, // Every week (10080 minutes)
        'is_active' => true,
        'created_by' => $user->id
    ]);
    
    echo "✓ Public API Target created successfully!\n";
    echo "  - ID: {$publicTarget->id}\n";
    echo "  - Name: {$publicTarget->name}\n";
    echo "  - URL: {$publicTarget->url}\n";
    echo "  - Auth: None (Public)\n";
    echo "  - Schedule: Weekly (interval)\n\n";
    
    // Example 5: API Target for Testing (using httpbin.org)
    echo "5. Creating Test API Target...\n";
    $testTarget = ApiTarget::create([
        'name' => 'Test API Target',
        'description' => 'API target untuk testing menggunakan httpbin.org',
        'base_url' => 'https://httpbin.org',
        'endpoint' => '/post',
        'method' => 'POST',
        'source_table' => 'users',
        'auth_type' => 'bearer_token',
        'auth_token' => 'test_token_12345',
        'auth_username' => null,
        'auth_password' => null,
        'schedule_type' => 'manual',
        'schedule_frequency' => null,
        'is_active' => true,
        'created_by' => $user->id
    ]);
    
    echo "✓ Test API Target created successfully!\n";
    echo "  - ID: {$testTarget->id}\n";
    echo "  - Name: {$testTarget->name}\n";
    echo "  - URL: {$testTarget->url}\n";
    echo "  - Auth: Bearer Token (test)\n";
    echo "  - Schedule: Manual\n\n";
    
    // Display summary
    echo "=== SUMMARY ===\n";
    echo "Total API Targets created: 5\n";
    echo "Active targets: " . ApiTarget::where('is_active', true)->count() . "\n";
    echo "Manual targets: " . ApiTarget::where('schedule_type', 'manual')->count() . "\n";
    echo "Scheduled targets: " . ApiTarget::where('schedule_type', '!=', 'manual')->count() . "\n\n";
    
    // Show all created targets
    echo "=== ALL CREATED TARGETS ===\n";
    $allTargets = ApiTarget::where('created_by', $user->id)->latest()->get();
    foreach ($allTargets as $target) {
        echo "ID: {$target->id} | {$target->name} | {$target->auth_type} | {$target->schedule_type}\n";
        echo "  URL: {$target->url}\n";
        echo "  Status: " . ($target->is_active ? 'Active' : 'Inactive') . "\n";
        echo "  Created: {$target->created_at->format('d M Y H:i:s')}\n";
        echo "  ---\n";
    }
    
    echo "\n=== NEXT STEPS ===\n";
    echo "1. Visit: http://127.0.0.1:8000/api-targets\n";
    echo "2. Test connection for each target\n";
    echo "3. Send data manually to test targets\n";
    echo "4. Check logs for successful/failed requests\n";
    echo "5. Monitor scheduled targets\n\n";
    
    echo "=== SUCCESS! ===\n";
    echo "All API targets created successfully with proper authentication!\n";
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
} 