<?php

namespace Database\Factories;

use App\Models\ApiSendLog;
use App\Models\ApiTarget;
use Illuminate\Database\Eloquent\Factories\Factory;

class ApiSendLogFactory extends Factory
{
    protected $model = ApiSendLog::class;

    public function definition()
    {
        return [
            'api_target_id' => ApiTarget::factory(),
            'success' => $this->faker->boolean(80), // 80% success rate
            'status_code' => $this->faker->randomElement([200, 201, 400, 401, 403, 404, 500]),
            'response_time' => $this->faker->randomFloat(3, 0.1, 5.0),
            'payload' => [
                'id' => $this->faker->numberBetween(1, 1000),
                'name' => $this->faker->name(),
                'email' => $this->faker->email(),
                'created_at' => $this->faker->dateTimeThisYear()->format('Y-m-d H:i:s')
            ],
            'response_body' => $this->faker->optional()->json(),
            'error_message' => $this->faker->optional()->sentence(),
            'records_sent' => $this->faker->numberBetween(1, 100),
            'retry_count' => $this->faker->numberBetween(0, 3),
            'sent_at' => $this->faker->dateTimeThisYear(),
            'completed_at' => $this->faker->optional()->dateTimeThisYear(),
            'batch_id' => $this->faker->optional()->uuid(),
        ];
    }

    /**
     * Indicate that the log entry was successful.
     */
    public function successful()
    {
        return $this->state(function (array $attributes) {
            return [
                'success' => true,
                'status_code' => $this->faker->randomElement([200, 201]),
                'error_message' => null,
            ];
        });
    }

    /**
     * Indicate that the log entry failed.
     */
    public function failed()
    {
        return $this->state(function (array $attributes) {
            return [
                'success' => false,
                'status_code' => $this->faker->randomElement([400, 401, 403, 404, 500]),
                'error_message' => $this->faker->sentence(),
            ];
        });
    }

    /**
     * Indicate that the log entry is pending.
     */
    public function pending()
    {
        return $this->state(function (array $attributes) {
            return [
                'success' => false,
                'status_code' => null,
                'error_message' => null,
            ];
        });
    }

    /**
     * Indicate that the log entry has a fast response time.
     */
    public function fast()
    {
        return $this->state(function (array $attributes) {
            return [
                'response_time' => $this->faker->randomFloat(3, 0.1, 1.0),
            ];
        });
    }

    /**
     * Indicate that the log entry has a slow response time.
     */
    public function slow()
    {
        return $this->state(function (array $attributes) {
            return [
                'response_time' => $this->faker->randomFloat(3, 2.0, 10.0),
            ];
        });
    }

    /**
     * Indicate that the log entry sent many records.
     */
    public function manyRecords()
    {
        return $this->state(function (array $attributes) {
            return [
                'records_sent' => $this->faker->numberBetween(100, 1000),
            ];
        });
    }

    /**
     * Indicate that the log entry sent few records.
     */
    public function fewRecords()
    {
        return $this->state(function (array $attributes) {
            return [
                'records_sent' => $this->faker->numberBetween(1, 10),
            ];
        });
    }
} 