<?php

namespace Database\Factories;

use App\Models\ApiTarget;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\Factory;

class ApiTargetFactory extends Factory
{
    protected $model = ApiTarget::class;

    public function definition()
    {
        return [
            'name' => $this->faker->company . ' API Target',
            'description' => $this->faker->sentence(),
            'base_url' => $this->faker->url(),
            'endpoint' => '/' . $this->faker->slug() . '/webhook',
            'method' => $this->faker->randomElement(['GET', 'POST', 'PUT', 'DELETE']),
            'source_table' => $this->faker->randomElement(['users', 'api_sources', 'api_targets']),
            'auth_type' => $this->faker->randomElement(['none', 'api_key', 'bearer_token', 'basic_auth']),
            'auth_token' => $this->faker->optional()->uuid(),
            'auth_username' => $this->faker->optional()->userName(),
            'auth_password' => $this->faker->optional()->password(),
            'headers' => null,
            'field_mapping' => null,
            'timeout' => $this->faker->numberBetween(10, 60),
            'is_active' => $this->faker->boolean(80),
            'callback_url' => $this->faker->optional()->url(),
            'callback_method' => $this->faker->randomElement(['POST', 'PUT', 'PATCH']),
            'success_response_template' => null,
            'error_response_template' => null,
            'schedule_type' => $this->faker->randomElement(['manual', 'hourly', 'daily', 'weekly']),
            'schedule_frequency' => $this->faker->optional()->numberBetween(1, 24),
            'last_sent_at' => $this->faker->optional()->dateTimeThisYear(),
            'total_forwarded' => $this->faker->numberBetween(0, 1000),
            'created_by' => User::factory(),
        ];
    }

    /**
     * Indicate that the API target is active.
     */
    public function active()
    {
        return $this->state(function (array $attributes) {
            return [
                'is_active' => true,
            ];
        });
    }

    /**
     * Indicate that the API target is inactive.
     */
    public function inactive()
    {
        return $this->state(function (array $attributes) {
            return [
                'is_active' => false,
            ];
        });
    }

    /**
     * Indicate that the API target uses basic authentication.
     */
    public function withBasicAuth()
    {
        return $this->state(function (array $attributes) {
            return [
                'auth_type' => 'basic_auth',
                'auth_username' => $this->faker->userName(),
                'auth_password' => $this->faker->password(),
            ];
        });
    }

    /**
     * Indicate that the API target uses bearer token authentication.
     */
    public function withBearerToken()
    {
        return $this->state(function (array $attributes) {
            return [
                'auth_type' => 'bearer_token',
                'auth_token' => $this->faker->uuid(),
            ];
        });
    }

    /**
     * Indicate that the API target uses API key authentication.
     */
    public function withApiKey()
    {
        return $this->state(function (array $attributes) {
            return [
                'auth_type' => 'api_key',
                'auth_token' => $this->faker->uuid(),
            ];
        });
    }

    /**
     * Indicate that the API target has no authentication.
     */
    public function withoutAuth()
    {
        return $this->state(function (array $attributes) {
            return [
                'auth_type' => 'none',
                'auth_token' => null,
                'auth_username' => null,
                'auth_password' => null,
            ];
        });
    }

    /**
     * Indicate that the API target is scheduled for hourly execution.
     */
    public function hourly()
    {
        return $this->state(function (array $attributes) {
            return [
                'schedule_type' => 'hourly',
                'schedule_frequency' => $this->faker->numberBetween(1, 24),
            ];
        });
    }

    /**
     * Indicate that the API target is scheduled for daily execution.
     */
    public function daily()
    {
        return $this->state(function (array $attributes) {
            return [
                'schedule_type' => 'daily',
                'schedule_frequency' => $this->faker->numberBetween(1, 7),
            ];
        });
    }

    /**
     * Indicate that the API target is scheduled for weekly execution.
     */
    public function weekly()
    {
        return $this->state(function (array $attributes) {
            return [
                'schedule_type' => 'weekly',
                'schedule_frequency' => $this->faker->numberBetween(1, 4),
            ];
        });
    }

    /**
     * Indicate that the API target is manual execution only.
     */
    public function manual()
    {
        return $this->state(function (array $attributes) {
            return [
                'schedule_type' => 'manual',
                'schedule_frequency' => null,
            ];
        });
    }
} 