<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Log::info('Starting comprehensive database cleanup...');

        // Fix api_sources table structure
        $this->fixApiSourcesTable();
        
        // Fix other tables
        $this->fixOtherTables();
        
        // Clean up tenant references
        $this->cleanupTenantReferences();
        
        Log::info('Database cleanup completed successfully');
    }

    /**
     * Fix api_sources table structure
     */
    private function fixApiSourcesTable(): void
    {
        if (!Schema::hasTable('api_sources')) {
            Log::warning('api_sources table does not exist, skipping...');
            return;
        }

        Schema::table('api_sources', function (Blueprint $table) {
            // Add missing columns if they don't exist
            if (!Schema::hasColumn('api_sources', 'auth_type')) {
                $table->string('auth_type')->default('none')->after('url');
                Log::info('Added auth_type column to api_sources');
            }
            
            if (!Schema::hasColumn('api_sources', 'auth_token')) {
                $table->text('auth_token')->nullable()->after('auth_type');
                Log::info('Added auth_token column to api_sources');
            }
            
            if (!Schema::hasColumn('api_sources', 'description')) {
                $table->text('description')->nullable()->after('last_fetched_at');
                Log::info('Added description column to api_sources');
            }
        });

        // Drop tenant_id if it exists
        if (Schema::hasColumn('api_sources', 'tenant_id')) {
            Schema::table('api_sources', function (Blueprint $table) {
                // Drop foreign key constraints first
                $this->dropForeignKeyIfExists('api_sources', 'api_sources_tenant_id_foreign');
                $table->dropColumn('tenant_id');
                Log::info('Removed tenant_id from api_sources');
            });
        }

        // Add indexes if they don't exist
        try {
            Schema::table('api_sources', function (Blueprint $table) {
                $table->index(['status', 'schedule_type']);
                $table->index('auth_type');
            });
        } catch (\Exception $e) {
            Log::info('Indexes might already exist: ' . $e->getMessage());
        }
    }

    /**
     * Fix other tables
     */
    private function fixOtherTables(): void
    {
        $tables = [
            'api_targets', 'api_receivers', 'api_clients', 'api_tokens', 
            'api_transceivers', 'api_logs', 'api_receive_logs', 
            'api_send_logs', 'api_access_logs', 'api_transceiver_logs'
        ];

        foreach ($tables as $tableName) {
            if (Schema::hasTable($tableName)) {
                $this->removeColumnIfExists($tableName, 'tenant_id');
            }
        }
    }

    /**
     * Clean up tenant references
     */
    private function cleanupTenantReferences(): void
    {
        // Remove any orphaned foreign key constraints
        $tables = [
            'api_sources', 'api_targets', 'api_receivers', 'api_clients', 
            'api_tokens', 'api_transceivers', 'api_logs'
        ];

        foreach ($tables as $table) {
            if (Schema::hasTable($table)) {
                $foreignKeys = [
                    $table . '_tenant_id_foreign',
                    'tenant_id_foreign',
                    $table . '_tenants_id_foreign'
                ];

                foreach ($foreignKeys as $foreignKey) {
                    $this->dropForeignKeyIfExists($table, $foreignKey);
                }
            }
        }
    }

    /**
     * Remove column if it exists
     */
    private function removeColumnIfExists(string $table, string $column): void
    {
        if (Schema::hasColumn($table, $column)) {
            Schema::table($table, function (Blueprint $table) use ($column) {
                $table->dropColumn($column);
            });
            Log::info("Removed {$column} from {$table}");
        }
    }

    /**
     * Drop foreign key if it exists
     */
    private function dropForeignKeyIfExists(string $table, string $foreignKey): void
    {
        try {
            DB::statement("ALTER TABLE {$table} DROP FOREIGN KEY {$foreignKey}");
            Log::info("Dropped foreign key {$foreignKey} from {$table}");
        } catch (\Exception $e) {
            // Foreign key doesn't exist, ignore
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // This is a cleanup migration, rollback is not recommended
        Log::warning('Rollback of cleanup migration is not recommended');
    }
};
