<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('api_callback_logs', function (Blueprint $table) {
            $table->id();
            
            // Relationship
            $table->foreignId('api_receiver_id')->constrained()->cascadeOnDelete();
            $table->foreignId('api_receive_log_id')->nullable()->constrained()->nullOnDelete();
            $table->foreignId('api_client_id')->nullable()->constrained()->nullOnDelete();
            
            // Callback details
            $table->string('callback_url', 500)->comment('Target callback URL');
            $table->string('method', 10)->default('POST')->comment('HTTP method used');
            $table->json('headers_sent')->nullable()->comment('Headers sent with callback');
            $table->longText('payload_sent')->nullable()->comment('Payload sent to callback URL');
            $table->integer('payload_size')->nullable()->comment('Size of payload in bytes');
            
            // Response details
            $table->integer('response_code')->nullable()->comment('HTTP response code received');
            $table->longText('response_body')->nullable()->comment('Response body received');
            $table->json('response_headers')->nullable()->comment('Response headers received');
            $table->float('response_time')->nullable()->comment('Response time in milliseconds');
            
            // Status and retry
            $table->enum('status', ['pending', 'success', 'failed', 'retrying'])->default('pending');
            $table->integer('retry_count')->default(0)->comment('Number of retry attempts');
            $table->integer('max_retries')->default(3)->comment('Maximum retry attempts');
            $table->timestamp('next_retry_at')->nullable()->comment('Next retry scheduled time');
            
            // Error handling
            $table->text('error_message')->nullable()->comment('Error message if failed');
            $table->text('error_details')->nullable()->comment('Detailed error information');
            
            // Timing
            $table->timestamp('sent_at')->nullable()->comment('When callback was sent');
            $table->timestamp('completed_at')->nullable()->comment('When callback completed');
            $table->timestamps();
            
            // Indexes for performance
            $table->index(['api_receiver_id', 'status', 'created_at']);
            $table->index(['status', 'next_retry_at']);
            $table->index(['api_client_id', 'created_at']);
            $table->index(['sent_at', 'completed_at']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('api_callback_logs');
    }
};
