<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\ApiReceiver;

class ApiReceiverSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Sample API Receivers for testing
        $receivers = [
            [
                'name' => 'Crypto Price Updates',
                'endpoint' => 'crypto-harga',
                'target_table' => 'crypto_prices',
                'auth_type' => 'bearer_token',
                'auth_token' => 'bt_' . str_random(40),
                'auth_secret' => null,
                'allowed_ips' => null,
                'allowed_headers' => ['Content-Type', 'Authorization'],
                'field_mapping' => [
                    'symbol' => 'coin_symbol',
                    'price' => 'current_price',
                    'timestamp' => 'updated_at'
                ],
                'validation_rules' => [
                    'symbol' => ['required' => true, 'type' => 'string', 'max_length' => 10],
                    'price' => ['required' => true, 'type' => 'float'],
                    'timestamp' => ['required' => true, 'type' => 'string']
                ],
                'auto_create_table' => true,
                'is_active' => true,
                'rate_limit' => 1000,
                'allowed_methods' => 'POST,PUT',
                'description' => 'Endpoint untuk menerima update harga cryptocurrency dari exchange',
                'sample_data' => [
                    'symbol' => 'BTCUSDT',
                    'price' => 67680.50,
                    'timestamp' => '2025-06-14T16:45:00Z',
                    'volume' => 1234.56
                ]
            ],
            [
                'name' => 'IoT Sensor Data',
                'endpoint' => 'iot-sensor',
                'target_table' => 'sensor_data',
                'auth_type' => 'api_key',
                'auth_token' => 'ak_' . str_random(32),
                'auth_secret' => null,
                'allowed_ips' => ['192.168.1.100', '192.168.1.101'],
                'allowed_headers' => ['Content-Type', 'X-API-Key', 'X-Device-ID'],
                'field_mapping' => null,
                'validation_rules' => [
                    'device_id' => ['required' => true, 'type' => 'string'],
                    'temperature' => ['required' => true, 'type' => 'float'],
                    'humidity' => ['required' => true, 'type' => 'float'],
                    'timestamp' => ['required' => true, 'type' => 'string']
                ],
                'auto_create_table' => true,
                'is_active' => true,
                'rate_limit' => 5000,
                'allowed_methods' => 'POST',
                'description' => 'Endpoint untuk menerima data sensor IoT (suhu, kelembaban, dll)',
                'sample_data' => [
                    'device_id' => 'SENSOR_001',
                    'temperature' => 25.5,
                    'humidity' => 60.2,
                    'battery' => 85,
                    'timestamp' => '2025-06-14T16:45:00Z'
                ]
            ],
            [
                'name' => 'User Registration Webhook',
                'endpoint' => 'user-registration',
                'target_table' => 'external_users',
                'auth_type' => 'hmac_signature',
                'auth_token' => null,
                'auth_secret' => str_random(64),
                'allowed_ips' => null,
                'allowed_headers' => ['Content-Type', 'X-Signature', 'X-Timestamp'],
                'field_mapping' => [
                    'user_email' => 'email',
                    'user_name' => 'name',
                    'registration_date' => 'created_at'
                ],
                'validation_rules' => [
                    'user_email' => ['required' => true, 'type' => 'email'],
                    'user_name' => ['required' => true, 'type' => 'string', 'max_length' => 100],
                    'registration_date' => ['required' => true, 'type' => 'string']
                ],
                'auto_create_table' => true,
                'is_active' => true,
                'rate_limit' => 500,
                'allowed_methods' => 'POST',
                'description' => 'Webhook untuk menerima notifikasi pendaftaran user baru dari sistem eksternal',
                'sample_data' => [
                    'user_email' => 'john.doe@example.com',
                    'user_name' => 'John Doe',
                    'registration_date' => '2025-06-14T16:45:00Z',
                    'source' => 'mobile_app'
                ]
            ],
            [
                'name' => 'Payment Notifications',
                'endpoint' => 'payment-notification',
                'target_table' => 'payment_logs',
                'auth_type' => 'bearer_token',
                'auth_token' => 'bt_' . str_random(40),
                'auth_secret' => null,
                'allowed_ips' => ['203.194.112.1', '203.194.112.2'],
                'allowed_headers' => ['Content-Type', 'Authorization', 'X-Gateway-ID'],
                'field_mapping' => [
                    'transaction_id' => 'trx_id',
                    'amount' => 'payment_amount',
                    'status' => 'payment_status'
                ],
                'validation_rules' => [
                    'transaction_id' => ['required' => true, 'type' => 'string'],
                    'amount' => ['required' => true, 'type' => 'float'],
                    'status' => ['required' => true, 'type' => 'string'],
                    'currency' => ['required' => true, 'type' => 'string']
                ],
                'auto_create_table' => true,
                'is_active' => true,
                'rate_limit' => 2000,
                'allowed_methods' => 'POST,PUT',
                'description' => 'Endpoint untuk menerima notifikasi pembayaran dari payment gateway',
                'sample_data' => [
                    'transaction_id' => 'TRX_12345678',
                    'amount' => 100000.00,
                    'currency' => 'IDR',
                    'status' => 'success',
                    'gateway' => 'midtrans',
                    'timestamp' => '2025-06-14T16:45:00Z'
                ]
            ],
            [
                'name' => 'Public Data Endpoint',
                'endpoint' => 'public-data',
                'target_table' => 'public_submissions',
                'auth_type' => 'none',
                'auth_token' => null,
                'auth_secret' => null,
                'allowed_ips' => null,
                'allowed_headers' => ['Content-Type'],
                'field_mapping' => null,
                'validation_rules' => [
                    'name' => ['required' => true, 'type' => 'string', 'max_length' => 100],
                    'message' => ['required' => true, 'type' => 'string', 'max_length' => 500]
                ],
                'auto_create_table' => true,
                'is_active' => true,
                'rate_limit' => 100,
                'allowed_methods' => 'POST',
                'description' => 'Endpoint publik untuk menerima feedback atau kontak dari website',
                'sample_data' => [
                    'name' => 'Jane Smith',
                    'email' => 'jane@example.com',
                    'message' => 'This is a test message from the contact form',
                    'ip_address' => '127.0.0.1',
                    'timestamp' => '2025-06-14T16:45:00Z'
                ]
            ]
        ];

        foreach ($receivers as $receiverData) {
            ApiReceiver::create($receiverData);
        }

        $this->command->info('Created ' . count($receivers) . ' sample API receivers');
    }
}

// Helper function for generating random strings
if (!function_exists('str_random')) {
    function str_random($length = 16) {
        $pool = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        return substr(str_shuffle(str_repeat($pool, 5)), 0, $length);
    }
}
