<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\PostOnlyReceiver;
use App\Models\ApiClient;

class PostOnlyReceiverSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get or create an API client
        $apiClient = ApiClient::first();
        
        if (!$apiClient) {
            $apiClient = ApiClient::create([
                'name' => 'Default Client for POST Receivers',
                'description' => 'Default API client for POST-only receivers',
                'client_id' => 'post_client_' . str_random(8),
                'client_secret' => str_random(64),
                'usertoken' => 'post_token_' . str_random(16),
                'passtoken' => str_random(32),
                'is_active' => true,
                'rate_limit' => 1000
            ]);
        }

        // Sample POST-Only Receivers
        $receivers = [
            [
                'name' => 'Webhook Data Collector',
                'description' => 'Endpoint untuk menerima data webhook dari sistem eksternal',
                'endpoint' => 'webhook-data',
                'target_table' => 'webhook_data',
                'api_client_id' => $apiClient->id,
                'require_token' => true,
                'field_mapping' => [
                    'event_type' => 'event',
                    'data' => 'payload',
                    'timestamp' => 'received_at'
                ],
                'validation_rules' => [
                    'event_type' => ['required' => true, 'type' => 'string', 'max_length' => 50],
                    'data' => ['required' => true, 'type' => 'array'],
                    'timestamp' => ['required' => true, 'type' => 'string']
                ],
                'auto_create_table' => true,
                'is_active' => true,
                'rate_limit' => 500,
                'sample_data' => [
                    'event_type' => 'user_created',
                    'data' => [
                        'user_id' => 123,
                        'email' => 'user@example.com',
                        'name' => 'John Doe'
                    ],
                    'timestamp' => '2025-01-23T10:30:00Z'
                ]
            ],
            [
                'name' => 'Payment Notification',
                'description' => 'Menerima notifikasi pembayaran dari payment gateway',
                'endpoint' => 'payment-notify',
                'target_table' => 'payment_notifications',
                'api_client_id' => $apiClient->id,
                'require_token' => true,
                'field_mapping' => [
                    'transaction_id' => 'trx_id',
                    'amount' => 'total_amount',
                    'status' => 'payment_status'
                ],
                'validation_rules' => [
                    'transaction_id' => ['required' => true, 'type' => 'string', 'max_length' => 100],
                    'amount' => ['required' => true, 'type' => 'numeric'],
                    'status' => ['required' => true, 'type' => 'string', 'max_length' => 20]
                ],
                'auto_create_table' => true,
                'is_active' => true,
                'rate_limit' => 1000,
                'sample_data' => [
                    'transaction_id' => 'TRX123456789',
                    'amount' => 150000,
                    'status' => 'completed',
                    'currency' => 'IDR',
                    'customer_email' => 'customer@example.com'
                ]
            ],
            [
                'name' => 'IoT Sensor Data',
                'description' => 'Endpoint untuk data sensor IoT',
                'endpoint' => 'iot-sensor',
                'target_table' => 'iot_sensor_data',
                'api_client_id' => $apiClient->id,
                'require_token' => true,
                'field_mapping' => [
                    'device_id' => 'sensor_id',
                    'temperature' => 'temp_celsius',
                    'humidity' => 'humidity_percent'
                ],
                'validation_rules' => [
                    'device_id' => ['required' => true, 'type' => 'string', 'max_length' => 50],
                    'temperature' => ['required' => true, 'type' => 'numeric'],
                    'humidity' => ['required' => true, 'type' => 'numeric']
                ],
                'auto_create_table' => true,
                'is_active' => true,
                'rate_limit' => 2000,
                'sample_data' => [
                    'device_id' => 'SENSOR_001',
                    'temperature' => 25.5,
                    'humidity' => 60.2,
                    'battery_level' => 85,
                    'location' => 'Room A'
                ]
            ],
            [
                'name' => 'Public Form Submission',
                'description' => 'Endpoint publik untuk form submission tanpa token',
                'endpoint' => 'form-submit',
                'target_table' => 'form_submissions',
                'api_client_id' => null,
                'require_token' => false,
                'field_mapping' => null,
                'validation_rules' => [
                    'name' => ['required' => true, 'type' => 'string', 'max_length' => 100],
                    'email' => ['required' => true, 'type' => 'email', 'max_length' => 150],
                    'message' => ['required' => true, 'type' => 'string', 'max_length' => 1000]
                ],
                'auto_create_table' => true,
                'is_active' => true,
                'rate_limit' => 100,
                'sample_data' => [
                    'name' => 'Jane Doe',
                    'email' => 'jane@example.com',
                    'message' => 'This is a test message from contact form',
                    'phone' => '+62812345678'
                ]
            ]
        ];

        foreach ($receivers as $receiverData) {
            PostOnlyReceiver::create($receiverData);
        }
    }
} 