<?php

/**
 * Debug script untuk mengidentifikasi masalah form GET-Only Receivers
 */

echo "=== DEBUG GET-ONLY RECEIVERS FORM ===\n";
echo "🔍 Analyzing form save issues...\n\n";

// 1. Test database table structure
echo "📋 Checking database table structure...\n";
$host = 'localhost';
$dbname = 'copg1493_api';
$username = 'copg1493_api';
$password = 'apipassword123';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Check if table exists
    $stmt = $pdo->query("SHOW TABLES LIKE 'get_only_receivers'");
    if ($stmt->rowCount() > 0) {
        echo "✅ get_only_receivers table exists\n";
        
        // Check table structure
        $stmt = $pdo->query("DESCRIBE get_only_receivers");
        $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo "✅ Table has " . count($columns) . " columns\n";
        
        // Check required columns
        $requiredColumns = [
            'name', 'endpoint', 'data_source', 'response_format', 
            'default_limit', 'max_limit', 'rate_limit_per_minute', 
            'rate_limit_per_hour', 'cache_duration'
        ];
        
        $existingColumns = array_column($columns, 'Field');
        $missingColumns = array_diff($requiredColumns, $existingColumns);
        
        if (empty($missingColumns)) {
            echo "✅ All required columns exist\n";
        } else {
            echo "❌ Missing columns: " . implode(', ', $missingColumns) . "\n";
        }
        
    } else {
        echo "❌ get_only_receivers table does not exist\n";
        echo "💡 Run the SQL file to create the table first\n";
        return;
    }
    
} catch (PDOException $e) {
    echo "❌ Database connection failed: " . $e->getMessage() . "\n";
    return;
}

// 2. Test form validation requirements
echo "\n📋 Checking form validation requirements...\n";

$validationRules = [
    'name' => 'required|string|max:255',
    'endpoint' => 'required|string|max:255|unique:get_only_receivers,endpoint|regex:/^[a-zA-Z0-9\-_]+$/',
    'data_source' => 'required|in:table,api,custom',
    'response_format' => 'required|in:json,xml,csv',
    'default_limit' => 'required|integer|min:1|max:1000',
    'max_limit' => 'required|integer|min:1|max:10000',
    'rate_limit_per_minute' => 'required|integer|min:1|max:1000',
    'rate_limit_per_hour' => 'required|integer|min:1|max:10000',
    'cache_duration' => 'required|integer|min:60|max:86400',
];

echo "✅ Found " . count($validationRules) . " validation rules\n";

// 3. Create test data
echo "\n📋 Creating test data for validation...\n";

$testData = [
    'name' => 'Test Receiver',
    'description' => 'Test description',
    'endpoint' => 'test-endpoint-' . time(),
    'data_source' => 'table',
    'source_table' => 'users',
    'response_format' => 'json',
    'default_limit' => 50,
    'max_limit' => 1000,
    'enable_pagination' => 1,
    'rate_limit_per_minute' => 60,
    'rate_limit_per_hour' => 1000,
    'enable_cors' => 1,
    'enable_cache' => 1,
    'cache_duration' => 300,
    'require_token' => 0,
    'is_active' => 1
];

echo "✅ Test data created with " . count($testData) . " fields\n";

// 4. Test data insertion
echo "\n📋 Testing data insertion...\n";

try {
    $columns = implode(', ', array_keys($testData));
    $placeholders = ':' . implode(', :', array_keys($testData));
    
    $sql = "INSERT INTO get_only_receivers ($columns) VALUES ($placeholders)";
    $stmt = $pdo->prepare($sql);
    
    foreach ($testData as $key => $value) {
        $stmt->bindValue(":$key", $value);
    }
    
    if ($stmt->execute()) {
        $insertId = $pdo->lastInsertId();
        echo "✅ Test record inserted successfully with ID: $insertId\n";
        
        // Clean up test record
        $pdo->exec("DELETE FROM get_only_receivers WHERE id = $insertId");
        echo "✅ Test record cleaned up\n";
    } else {
        echo "❌ Failed to insert test record\n";
        print_r($stmt->errorInfo());
    }
    
} catch (PDOException $e) {
    echo "❌ Database insertion error: " . $e->getMessage() . "\n";
}

// 5. Check common form issues
echo "\n📋 Checking common form issues...\n";

// Check if response_fields is handled correctly
echo "🔍 Checking response_fields handling...\n";
echo "💡 response_fields should be converted from string to array\n";

// Check if boolean fields are handled correctly
echo "🔍 Checking boolean fields...\n";
$booleanFields = ['enable_pagination', 'require_token', 'enable_cors', 'enable_cache', 'is_active'];
echo "💡 Boolean fields: " . implode(', ', $booleanFields) . "\n";
echo "💡 These should have value='1' and be unchecked by default or checked with old() values\n";

// Check if array fields are handled correctly
echo "🔍 Checking array fields...\n";
$arrayFields = ['allowed_tokens', 'response_fields', 'cors_origins'];
echo "💡 Array fields: " . implode(', ', $arrayFields) . "\n";
echo "💡 These should be converted from strings to arrays in controller\n";

// 6. Generate fix recommendations
echo "\n🔧 FIX RECOMMENDATIONS:\n";

echo "1. **Database Table**: ";
if ($stmt->rowCount() > 0) {
    echo "✅ Table exists\n";
} else {
    echo "❌ Create table using get-only-receivers-complete.sql\n";
}

echo "2. **Form Validation**: Check browser console for JavaScript errors\n";
echo "3. **Required Fields**: Ensure all required fields have values\n";
echo "4. **Boolean Fields**: Check checkbox states are sent correctly\n";
echo "5. **Array Fields**: Ensure comma-separated strings are converted to arrays\n";

// 7. Create simple test form
echo "\n📝 Creating simple test form...\n";

$testFormContent = '
<!DOCTYPE html>
<html>
<head>
    <title>Test GET-Only Receiver Form</title>
    <meta name="csrf-token" content="{{ csrf_token() }}">
</head>
<body>
    <h2>Simple Test Form</h2>
    <form action="/get-only-receivers" method="POST">
        <input type="hidden" name="_token" value="test-token">
        
        <label>Name *:</label>
        <input type="text" name="name" value="Test Receiver" required><br><br>
        
        <label>Endpoint *:</label>
        <input type="text" name="endpoint" value="test-' . time() . '" required><br><br>
        
        <label>Data Source *:</label>
        <select name="data_source" required>
            <option value="table">Database Table</option>
        </select><br><br>
        
        <label>Source Table:</label>
        <select name="source_table">
            <option value="users">users</option>
        </select><br><br>
        
        <label>Response Format *:</label>
        <select name="response_format" required>
            <option value="json">JSON</option>
        </select><br><br>
        
        <label>Default Limit *:</label>
        <input type="number" name="default_limit" value="50" required><br><br>
        
        <label>Max Limit *:</label>
        <input type="number" name="max_limit" value="1000" required><br><br>
        
        <label>Rate Limit Per Minute *:</label>
        <input type="number" name="rate_limit_per_minute" value="60" required><br><br>
        
        <label>Rate Limit Per Hour *:</label>
        <input type="number" name="rate_limit_per_hour" value="1000" required><br><br>
        
        <label>Cache Duration *:</label>
        <input type="number" name="cache_duration" value="300" required><br><br>
        
        <input type="hidden" name="enable_pagination" value="1">
        <input type="hidden" name="enable_cors" value="1">
        <input type="hidden" name="enable_cache" value="1">
        <input type="hidden" name="require_token" value="0">
        <input type="hidden" name="is_active" value="1">
        
        <button type="submit">Save Test Receiver</button>
    </form>
</body>
</html>';

file_put_contents('test-get-only-receiver-form.html', $testFormContent);
echo "✅ Simple test form created: test-get-only-receiver-form.html\n";

echo "\n🎯 NEXT STEPS:\n";
echo "1. Ensure get_only_receivers table exists (run SQL file)\n";
echo "2. Check browser console for JavaScript errors\n";
echo "3. Try the simple test form\n";
echo "4. Check Laravel logs for detailed error messages\n";

echo "\n🎉 Debug completed!\n"; 