<?php
/**
 * Deployment Script for POST-Only Receivers
 * Run this on production server to deploy the feature
 */

// Set time limit
set_time_limit(600);

// Get Laravel root directory
$laravelRoot = __DIR__;
chdir($laravelRoot);

echo "<h1>🚀 POST-Only Receivers Deployment</h1>";
echo "<p><strong>Laravel Root:</strong> " . $laravelRoot . "</p>";
echo "<p><strong>Time:</strong> " . date('Y-m-d H:i:s') . "</p>";
echo "<hr>";

$steps = [
    'migrate' => 'Run Database Migrations',
    'route:clear' => 'Clear Route Cache',
    'config:clear' => 'Clear Config Cache',
    'view:clear' => 'Clear View Cache',
    'cache:clear' => 'Clear Application Cache',
    'optimize:clear' => 'Clear Optimized Files',
    'dump-autoload' => 'Regenerate Autoload Files',
    'seed' => 'Run Database Seeders'
];

$results = [];

foreach ($steps as $command => $description) {
    echo "<h3>📋 Step: {$description}</h3>";
    
    try {
        $output = [];
        $returnCode = 0;
        
        // Special handling for different commands
        if ($command === 'migrate') {
            echo "<p>🔄 Running migrations...</p>";
            exec("php artisan migrate --force 2>&1", $output, $returnCode);
        } elseif ($command === 'dump-autoload') {
            echo "<p>🔄 Regenerating autoload files...</p>";
            exec("composer dump-autoload --optimize 2>&1", $output, $returnCode);
        } elseif ($command === 'seed') {
            echo "<p>🔄 Running seeders...</p>";
            exec("php artisan db:seed --class=PostOnlyReceiverSeeder --force 2>&1", $output, $returnCode);
        } else {
            echo "<p>🔄 Running php artisan {$command}...</p>";
            exec("php artisan {$command} 2>&1", $output, $returnCode);
        }
        
        if ($returnCode === 0) {
            echo "<p style='color: green;'>✅ {$description} completed successfully</p>";
            $results[$command] = 'success';
            
            // Show relevant output
            if (!empty($output)) {
                $relevantOutput = array_slice($output, -5); // Last 5 lines
                echo "<details><summary>Show output</summary>";
                echo "<pre style='background: #f8f9fa; padding: 10px; border-radius: 5px;'>";
                echo htmlspecialchars(implode("\n", $relevantOutput));
                echo "</pre></details>";
            }
        } else {
            echo "<p style='color: red;'>❌ {$description} failed</p>";
            $results[$command] = 'failed';
            
            if (!empty($output)) {
                echo "<pre style='background: #ffe6e6; padding: 10px; border-radius: 5px;'>";
                echo htmlspecialchars(implode("\n", $output));
                echo "</pre>";
            }
        }
        
    } catch (Exception $e) {
        echo "<p style='color: red;'>❌ Error in {$description}: " . $e->getMessage() . "</p>";
        $results[$command] = 'error';
    }
    
    echo "<hr>";
}

// Verification Steps
echo "<h2>🔍 Verification</h2>";

// Check if routes are available
echo "<h3>📍 Checking Routes</h3>";
try {
    $output = [];
    exec("php artisan route:list --name=post-only 2>&1", $output);
    
    if (!empty($output)) {
        echo "<p style='color: green;'>✅ POST-Only Receiver routes found:</p>";
        echo "<pre style='background: #f8f9fa; padding: 10px; border-radius: 5px;'>";
        echo htmlspecialchars(implode("\n", $output));
        echo "</pre>";
    } else {
        echo "<p style='color: orange;'>⚠️ No POST-Only Receiver routes found</p>";
    }
} catch (Exception $e) {
    echo "<p style='color: red;'>❌ Error checking routes: " . $e->getMessage() . "</p>";
}

// Check if tables exist
echo "<h3>📊 Checking Database Tables</h3>";
try {
    $output = [];
    exec("php artisan tinker --execute=\"echo 'post_only_receivers: ' . \\App\\Models\\PostOnlyReceiver::count() . PHP_EOL; echo 'post_only_receive_logs: ' . \\App\\Models\\PostOnlyReceiveLog::count() . PHP_EOL;\" 2>&1", $output);
    
    if (!empty($output)) {
        echo "<p style='color: green;'>✅ Database tables check:</p>";
        echo "<pre style='background: #f8f9fa; padding: 10px; border-radius: 5px;'>";
        echo htmlspecialchars(implode("\n", $output));
        echo "</pre>";
    }
} catch (Exception $e) {
    echo "<p style='color: red;'>❌ Error checking database: " . $e->getMessage() . "</p>";
}

// Check if files exist
echo "<h3>📁 Checking Files</h3>";
$requiredFiles = [
    'app/Http/Controllers/PostOnlyReceiverController.php',
    'app/Models/PostOnlyReceiver.php',
    'app/Models/PostOnlyReceiveLog.php',
    'resources/views/post-only-receivers/index.blade.php',
    'resources/views/post-only-receivers/create.blade.php',
    'resources/views/post-only-receivers/show.blade.php',
    'database/migrations/2025_01_23_000001_create_post_only_receivers_table.php',
    'database/migrations/2025_01_23_000002_create_post_only_receive_logs_table.php'
];

foreach ($requiredFiles as $file) {
    if (file_exists($file)) {
        echo "<p style='color: green;'>✅ {$file}</p>";
    } else {
        echo "<p style='color: red;'>❌ Missing: {$file}</p>";
    }
}

// Summary
echo "<hr>";
echo "<h2>📊 Deployment Summary</h2>";

$successCount = count(array_filter($results, function($result) { return $result === 'success'; }));
$totalCount = count($results);

if ($successCount === $totalCount) {
    echo "<div style='background: #d4edda; color: #155724; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
    echo "<h3>✅ Deployment Successful!</h3>";
    echo "<p>All {$totalCount} deployment steps completed successfully.</p>";
    echo "<p><strong>POST-Only Receivers feature is now ready to use.</strong></p>";
    echo "</div>";
} else {
    echo "<div style='background: #fff3cd; color: #856404; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
    echo "<h3>⚠️ Deployment Completed with Issues</h3>";
    echo "<p>Success: {$successCount}/{$totalCount} steps</p>";
    echo "<p>Please review the failed steps above and fix any issues.</p>";
    echo "</div>";
}

echo "<h3>🔗 Quick Access Links</h3>";
echo "<ul>";
echo "<li><a href='/post-only-receivers' target='_blank' style='color: #007bff;'>📧 POST-Only Receivers Dashboard</a></li>";
echo "<li><a href='/post-only-receivers/create' target='_blank' style='color: #28a745;'>➕ Create New Receiver</a></li>";
echo "<li><a href='/api-clients' target='_blank' style='color: #6c757d;'>👥 API Clients</a></li>";
echo "<li><a href='/api-tokens' target='_blank' style='color: #ffc107;'>🔑 API Tokens</a></li>";
echo "<li><a href='/test-post-receiver.html' target='_blank' style='color: #17a2b8;'>🧪 Test Interface</a></li>";
echo "</ul>";

echo "<h3>📖 Documentation</h3>";
echo "<ul>";
echo "<li><a href='/docs/POST_ONLY_RECEIVER_GUIDE.md' target='_blank' style='color: #6f42c1;'>📚 Complete Guide</a></li>";
echo "<li><a href='/public/clear-cache-post-receivers.php' target='_blank' style='color: #dc3545;'>🧹 Cache Cleaner</a></li>";
echo "</ul>";

echo "<h3>🛠️ Manual Commands (if needed)</h3>";
echo "<div style='background: #f8f9fa; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
echo "<p><strong>If you still experience issues, run these commands manually:</strong></p>";
echo "<code>cd " . $laravelRoot . "</code><br>";
echo "<code>php artisan migrate --force</code><br>";
echo "<code>php artisan route:clear</code><br>";
echo "<code>php artisan config:clear</code><br>";
echo "<code>php artisan view:clear</code><br>";
echo "<code>php artisan cache:clear</code><br>";
echo "<code>composer dump-autoload --optimize</code><br>";
echo "</div>";

echo "<hr>";
echo "<p><small>Deployment completed at: " . date('Y-m-d H:i:s') . "</small></p>";
echo "<p><small>Script: deploy-post-receivers.php</small></p>";
?> 