<?php

/**
 * Diagnose Button Issue - Advanced troubleshooting
 */

echo "=== DIAGNOSE BUTTON ISSUE ===\n";
echo "🔍 Advanced troubleshooting for button functionality...\n\n";

// 1. Check if the form is actually receiving the updated JavaScript
echo "📋 Checking potential issues...\n";

$possibleIssues = [
    "1. Browser cache - JavaScript lama masih ter-cache",
    "2. Laravel view cache - Blade template tidak terupdate",
    "3. JavaScript error - Preventing event handlers from loading",
    "4. Missing jQuery/Bootstrap dependencies",
    "5. CSRF token issues",
    "6. Form submission blocked by server",
    "7. Database table masih belum ada",
    "8. Laravel route issues"
];

foreach ($possibleIssues as $issue) {
    echo "❌ $issue\n";
}

// 2. Create emergency JavaScript fix
echo "\n📋 Creating emergency JavaScript fix...\n";

$emergencyJS = '
<!-- EMERGENCY BUTTON FIX - Add this to the bottom of create.blade.php -->
<script>
// Emergency fix - bypass all existing JavaScript
(function() {
    "use strict";
    
    console.log("🚨 Emergency button fix loaded");
    
    // Wait for DOM to be ready
    function ready(fn) {
        if (document.readyState !== "loading") {
            fn();
        } else {
            document.addEventListener("DOMContentLoaded", fn);
        }
    }
    
    ready(function() {
        console.log("🚨 Emergency fix: DOM ready");
        
        // Get form
        const form = document.querySelector("#createForm");
        if (!form) {
            console.error("🚨 Emergency fix: Form not found");
            return;
        }
        console.log("✅ Emergency fix: Form found");
        
        // Remove existing event listeners and add new ones
        const buttons = [
            { id: "saveBtn", type: "submit" },
            { id: "altSubmitBtn", type: "submit" },
            { id: "simpleSubmitBtn", type: "submit" },
            { id: "testEndpointBtn", type: "test" }
        ];
        
        buttons.forEach(function(btnConfig) {
            const btn = document.getElementById(btnConfig.id);
            if (btn) {
                // Clone button to remove all event listeners
                const newBtn = btn.cloneNode(true);
                btn.parentNode.replaceChild(newBtn, btn);
                
                // Add new event listener
                newBtn.addEventListener("click", function(e) {
                    console.log("🚨 Emergency fix: Button clicked - " + btnConfig.id);
                    e.preventDefault();
                    
                    if (btnConfig.type === "submit") {
                        // Basic validation
                        const name = document.getElementById("name");
                        const endpoint = document.getElementById("endpoint");
                        const dataSource = document.getElementById("data_source");
                        const responseFormat = document.getElementById("response_format");
                        
                        const errors = [];
                        if (!name || !name.value.trim()) errors.push("Name is required");
                        if (!endpoint || !endpoint.value.trim()) errors.push("Endpoint is required");
                        if (!dataSource || !dataSource.value) errors.push("Data Source is required");
                        if (!responseFormat || !responseFormat.value) errors.push("Response Format is required");
                        
                        // Check conditional fields
                        if (dataSource && dataSource.value === "table") {
                            const sourceTable = document.getElementById("source_table");
                            if (!sourceTable || !sourceTable.value) {
                                errors.push("Source Table is required when Data Source is Database Table");
                            }
                        }
                        
                        if (errors.length > 0) {
                            console.error("🚨 Emergency fix: Validation errors", errors);
                            alert("Please fix these errors:\\n" + errors.join("\\n"));
                            return;
                        }
                        
                        // Show loading
                        newBtn.disabled = true;
                        newBtn.innerHTML = "<i class=\"fas fa-spinner fa-spin\"></i> Saving...";
                        
                        console.log("🚨 Emergency fix: Submitting form");
                        
                        // Submit form
                        setTimeout(function() {
                            form.submit();
                        }, 100);
                        
                    } else if (btnConfig.type === "test") {
                        const endpoint = document.getElementById("endpoint");
                        if (!endpoint || !endpoint.value.trim()) {
                            alert("Please enter an endpoint first");
                            return;
                        }
                        
                        const testUrl = "/api/get/" + endpoint.value + "?test=1";
                        console.log("🚨 Emergency fix: Opening test URL", testUrl);
                        window.open(testUrl, "_blank");
                    }
                });
                
                console.log("✅ Emergency fix: Fixed button " + btnConfig.id);
            } else {
                console.warn("🚨 Emergency fix: Button not found - " + btnConfig.id);
            }
        });
        
        // Fix data source toggle
        const dataSourceSelect = document.getElementById("data_source");
        if (dataSourceSelect) {
            dataSourceSelect.addEventListener("change", function() {
                console.log("🚨 Emergency fix: Data source changed to", this.value);
                
                // Hide all configs
                const configs = document.querySelectorAll(".data-source-config");
                configs.forEach(function(config) {
                    config.style.display = "none";
                });
                
                // Show selected config
                if (this.value) {
                    const targetConfig = document.getElementById(this.value + "_source");
                    if (targetConfig) {
                        targetConfig.style.display = "block";
                    }
                }
            });
            
            // Trigger change
            dataSourceSelect.dispatchEvent(new Event("change"));
        }
        
        // Auto-generate endpoint
        const nameInput = document.getElementById("name");
        const endpointInput = document.getElementById("endpoint");
        if (nameInput && endpointInput) {
            nameInput.addEventListener("input", function() {
                const name = this.value;
                const endpoint = name.toLowerCase()
                    .replace(/[^a-z0-9\\s\\-_]/g, "")
                    .replace(/\\s+/g, "-")
                    .replace(/\\-+/g, "-")
                    .replace(/^-+|-+$/g, "");
                
                if (!endpointInput.value || endpointInput.dataset.autoGenerated === "true") {
                    endpointInput.value = endpoint;
                    endpointInput.dataset.autoGenerated = "true";
                }
            });
        }
        
        console.log("✅ Emergency fix: All fixes applied");
    });
})();
</script>
';

file_put_contents('emergency-button-fix.html', $emergencyJS);
echo "✅ Emergency JavaScript fix saved to: emergency-button-fix.html\n";

// 3. Create cache clearing commands
echo "\n📋 Creating cache clearing commands...\n";

$cacheCommands = [
    "php artisan view:clear",
    "php artisan cache:clear", 
    "php artisan config:clear",
    "php artisan route:clear"
];

$cacheScript = "#!/bin/bash\n";
$cacheScript .= "echo 'Clearing Laravel caches...'\n";
foreach ($cacheCommands as $cmd) {
    $cacheScript .= "echo 'Running: $cmd'\n";
    $cacheScript .= "$cmd\n";
}
$cacheScript .= "echo 'Cache clearing completed!'\n";

file_put_contents('clear-caches.sh', $cacheScript);
echo "✅ Cache clearing script saved to: clear-caches.sh\n";

// 4. Create minimal working form
echo "\n📋 Creating minimal working form...\n";

$minimalForm = '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Minimal Working Form</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <div class="container mt-4">
        <div class="alert alert-info">
            <h4>🧪 Minimal Working Form Test</h4>
            <p>This form tests the absolute minimum required to submit a GET-Only Receiver</p>
        </div>
        
        <form action="https://copilotproduction.site/get-only-receivers" method="POST" id="minimalForm">
            <input type="hidden" name="_token" value="REPLACE_WITH_REAL_TOKEN">
            
            <div class="mb-3">
                <label for="name" class="form-label">Name *</label>
                <input type="text" class="form-control" id="name" name="name" value="Test Receiver" required>
            </div>
            
            <div class="mb-3">
                <label for="endpoint" class="form-label">Endpoint *</label>
                <input type="text" class="form-control" id="endpoint" name="endpoint" value="test-endpoint" required>
            </div>
            
            <div class="mb-3">
                <label for="data_source" class="form-label">Data Source *</label>
                <select class="form-control" id="data_source" name="data_source" required>
                    <option value="table">Database Table</option>
                </select>
            </div>
            
            <div class="mb-3">
                <label for="source_table" class="form-label">Source Table *</label>
                <select class="form-control" id="source_table" name="source_table" required>
                    <option value="users">users</option>
                </select>
            </div>
            
            <div class="mb-3">
                <label for="response_format" class="form-label">Response Format *</label>
                <select class="form-control" id="response_format" name="response_format" required>
                    <option value="json">JSON</option>
                </select>
            </div>
            
            <div class="mb-3">
                <label for="default_limit" class="form-label">Default Limit *</label>
                <input type="number" class="form-control" id="default_limit" name="default_limit" value="50" required>
            </div>
            
            <div class="mb-3">
                <label for="max_limit" class="form-label">Max Limit *</label>
                <input type="number" class="form-control" id="max_limit" name="max_limit" value="1000" required>
            </div>
            
            <div class="mb-3">
                <label for="rate_limit_per_minute" class="form-label">Rate Limit Per Minute *</label>
                <input type="number" class="form-control" id="rate_limit_per_minute" name="rate_limit_per_minute" value="60" required>
            </div>
            
            <div class="mb-3">
                <label for="rate_limit_per_hour" class="form-label">Rate Limit Per Hour *</label>
                <input type="number" class="form-control" id="rate_limit_per_hour" name="rate_limit_per_hour" value="1000" required>
            </div>
            
            <div class="mb-3">
                <label for="cache_duration" class="form-label">Cache Duration *</label>
                <input type="number" class="form-control" id="cache_duration" name="cache_duration" value="300" required>
            </div>
            
            <!-- Hidden fields with default values -->
            <input type="hidden" name="enable_pagination" value="1">
            <input type="hidden" name="require_token" value="0">
            <input type="hidden" name="enable_cors" value="1">
            <input type="hidden" name="enable_cache" value="1">
            <input type="hidden" name="is_active" value="1">
            
            <button type="submit" class="btn btn-success" id="submitBtn">
                <i class="fas fa-save"></i> Submit Minimal Form
            </button>
        </form>
        
        <div class="mt-4 p-3 bg-light">
            <h5>Debug Info:</h5>
            <div id="debugInfo"></div>
        </div>
    </div>
    
    <script>
    document.addEventListener("DOMContentLoaded", function() {
        const form = document.getElementById("minimalForm");
        const submitBtn = document.getElementById("submitBtn");
        const debugInfo = document.getElementById("debugInfo");
        
        function log(message) {
            debugInfo.innerHTML += new Date().toLocaleTimeString() + ": " + message + "<br>";
            console.log(message);
        }
        
        log("✅ Minimal form loaded");
        
        submitBtn.addEventListener("click", function(e) {
            e.preventDefault();
            log("🔴 Submit button clicked");
            
            // Basic validation
            const required = ["name", "endpoint", "data_source", "source_table", "response_format"];
            const errors = [];
            
            required.forEach(function(fieldName) {
                const field = document.getElementById(fieldName);
                if (!field || !field.value.trim()) {
                    errors.push(fieldName + " is required");
                }
            });
            
            if (errors.length > 0) {
                log("❌ Validation errors: " + errors.join(", "));
                alert("Errors: " + errors.join(", "));
                return;
            }
            
            log("✅ Validation passed");
            
            // Show loading
            submitBtn.disabled = true;
            submitBtn.innerHTML = "<i class=\"fas fa-spinner fa-spin\"></i> Submitting...";
            
            log("🚀 Submitting form to server...");
            
            // Submit form
            setTimeout(function() {
                form.submit();
            }, 500);
        });
        
        log("✅ Event handlers attached");
    });
    </script>
</body>
</html>';

file_put_contents('minimal-working-form.html', $minimalForm);
echo "✅ Minimal working form saved to: minimal-working-form.html\n";

// 5. Create step-by-step troubleshooting guide
echo "\n📋 Creating step-by-step troubleshooting guide...\n";

$troubleshootingSteps = '# Step-by-Step Troubleshooting Guide

## 🚨 IMMEDIATE ACTIONS NEEDED

### Step 1: Clear Browser Cache
1. Open browser (Chrome/Firefox/Safari)
2. Press Ctrl+Shift+Delete (Windows) or Cmd+Shift+Delete (Mac)
3. Select "All time" and clear cache
4. Refresh the page

### Step 2: Clear Laravel Caches
Run these commands in terminal:
```bash
php artisan view:clear
php artisan cache:clear
php artisan config:clear
php artisan route:clear
```

### Step 3: Check Browser Console
1. Open https://copilotproduction.site/get-only-receivers/create
2. Press F12 to open Developer Tools
3. Go to Console tab
4. Look for JavaScript errors (red text)
5. Take screenshot of any errors

### Step 4: Apply Emergency Fix
1. Open `resources/views/get-only-receivers/create.blade.php`
2. Add the content from `emergency-button-fix.html` before `@endsection`
3. Save the file
4. Test the form again

### Step 5: Test with Minimal Form
1. Open `minimal-working-form.html` in browser
2. Replace `REPLACE_WITH_REAL_TOKEN` with actual CSRF token
3. Test if basic submission works

## 🔍 DEBUGGING CHECKLIST

- [ ] Browser cache cleared
- [ ] Laravel caches cleared  
- [ ] JavaScript console checked for errors
- [ ] CSRF token present in form
- [ ] Database table `get_only_receivers` exists
- [ ] Form action URL is correct
- [ ] All required fields have default values
- [ ] Emergency fix applied

## 📊 EXPECTED RESULTS

After applying fixes:
1. Console should show: "🚨 Emergency button fix loaded"
2. Button click should show: "🚨 Emergency fix: Button clicked"
3. Form should submit without "Loading..." hanging
4. Should redirect to index page or show validation errors

## 🎯 ROOT CAUSE ANALYSIS

The "Loading..." text suggests:
1. JavaScript is hanging or has errors
2. Form submission is blocked
3. Server is not responding
4. CSRF token mismatch
5. Database connection issues

## 🚀 NEXT STEPS

1. **Apply emergency fix immediately**
2. **Clear all caches**
3. **Test with minimal form**
4. **Check browser console for errors**
5. **Verify database table exists**

If still not working, the issue is likely:
- Database table missing
- CSRF token problems
- Server-side errors
- Network connectivity issues
';

file_put_contents('step-by-step-troubleshooting.md', $troubleshootingSteps);
echo "✅ Step-by-step troubleshooting guide saved to: step-by-step-troubleshooting.md\n";

echo "\n🎯 IMMEDIATE ACTION REQUIRED:\n";
echo "1. ⚡ Apply emergency fix from emergency-button-fix.html\n";
echo "2. 🧹 Clear browser cache and Laravel caches\n";
echo "3. 🔍 Check browser console for JavaScript errors\n";
echo "4. 🧪 Test with minimal-working-form.html\n";
echo "5. 📊 Follow step-by-step-troubleshooting.md\n";

echo "\n🚨 The 'Loading...' text indicates JavaScript is hanging!\n";
echo "Emergency fix will bypass existing JavaScript and create new event handlers.\n";

echo "\n🎉 Diagnostic completed! Apply emergency fix immediately.\n"; 