# Panduan Integrasi API untuk Client

## 1. Mendapatkan Akses API

### 1.1 Registrasi API Client
1. Akses halaman: `http://127.0.0.1:8000/api-clients/create`
2. Isi form registrasi dengan informasi:
   - Nama Client
   - Deskripsi
   - Email kontak
3. Setelah submit, Anda akan mendapatkan:
   - Client ID
   - API Token

### 1.2 Mendapatkan API Token
1. Login ke dashboard
2. Buka menu API Tokens
3. Generate token baru dengan mengklik "Create New Token"
4. Simpan token yang diberikan (token hanya ditampilkan sekali)

## 2. Menerima Data (Webhook)

### 2.1 Setup Endpoint Penerima
1. Siapkan endpoint di sistem Anda untuk menerima data
   Contoh (PHP):
   ```php
   // webhook.php
   <?php
   $data = file_get_contents('php://input');
   $payload = json_decode($data, true);
   
   // Proses data
   file_put_contents('webhook.log', date('Y-m-d H:i:s') . " Received: " . $data . "\n", FILE_APPEND);
   
   // Kirim response
   header('Content-Type: application/json');
   echo json_encode(['success' => true]);
   ```

### 2.2 Konfigurasi Webhook di API
1. Buka `http://127.0.0.1:8000/api-transceivers/[ID]/edit`
2. Isi konfigurasi webhook:
   - Callback URL: URL endpoint Anda
   - Method: POST/PUT
   - Format Response: JSON
   - Retry Settings: Sesuai kebutuhan

## 3. Mengirim Data ke API

### 3.1 Autentikasi
Sertakan API Token di setiap request:
```bash
curl -X POST http://127.0.0.1:8000/api/in/[endpoint] \
  -H "Authorization: Bearer YOUR_API_TOKEN" \
  -H "Content-Type: application/json"
```

### 3.2 Format Request
```json
{
    "data": {
        "field1": "value1",
        "field2": "value2"
    },
    "metadata": {
        "source": "client_system",
        "timestamp": "2024-01-01T00:00:00Z"
    }
}
```

### 3.3 Contoh Kode Integrasi

#### PHP
```php
<?php
$apiToken = 'YOUR_API_TOKEN';
$endpoint = 'your-endpoint';
$data = [
    'data' => [
        'field1' => 'value1',
        'field2' => 'value2'
    ]
];

$ch = curl_init("http://127.0.0.1:8000/api/in/{$endpoint}");
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Authorization: Bearer ' . $apiToken,
    'Content-Type: application/json'
]);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

echo "Response: " . $response;
```

#### Python
```python
import requests
import json

api_token = 'YOUR_API_TOKEN'
endpoint = 'your-endpoint'
data = {
    'data': {
        'field1': 'value1',
        'field2': 'value2'
    }
}

headers = {
    'Authorization': f'Bearer {api_token}',
    'Content-Type': 'application/json'
}

response = requests.post(
    f'http://127.0.0.1:8000/api/in/{endpoint}',
    headers=headers,
    json=data
)

print(f"Status: {response.status_code}")
print(f"Response: {response.json()}")
```

#### JavaScript/Node.js
```javascript
const axios = require('axios');

const apiToken = 'YOUR_API_TOKEN';
const endpoint = 'your-endpoint';
const data = {
    data: {
        field1: 'value1',
        field2: 'value2'
    }
};

axios.post(`http://127.0.0.1:8000/api/in/${endpoint}`, data, {
    headers: {
        'Authorization': `Bearer ${apiToken}`,
        'Content-Type': 'application/json'
    }
})
.then(response => {
    console.log('Success:', response.data);
})
.catch(error => {
    console.error('Error:', error.response?.data || error.message);
});
```

## 4. Monitoring dan Troubleshooting

### 4.1 Melihat Log
1. Akses dashboard: `http://127.0.0.1:8000/api-transceivers`
2. Pilih transceiver yang ingin dilihat
3. Klik tab "Logs"

### 4.2 Status Codes
- 200: Sukses
- 401: Token tidak valid
- 403: Tidak memiliki akses
- 422: Data tidak valid
- 500: Error server

### 4.3 Penanganan Error
1. Selalu cek response status
2. Simpan log error di sisi client
3. Implementasi retry mechanism untuk error temporary
4. Hubungi support jika error persisten

## 5. Best Practices

1. **Keamanan**
   - Jangan share API Token
   - Gunakan HTTPS
   - Validasi data sebelum kirim

2. **Performance**
   - Batasi ukuran payload
   - Implementasi rate limiting
   - Gunakan connection pooling

3. **Monitoring**
   - Log semua request/response
   - Set up alert untuk error
   - Monitor response time

4. **Error Handling**
   - Implement circuit breaker
   - Graceful degradation
   - Retry dengan exponential backoff

## 6. Support

Jika mengalami masalah atau butuh bantuan:
1. Check dokumentasi di `http://127.0.0.1:8000/docs`
2. Lihat log di dashboard
3. Kontak support: support@example.com 