<?php

/**
 * Fix button functionality untuk GET-Only Receivers form
 */

echo "=== FIX BUTTON FUNCTIONALITY ===\n";
echo "🔧 Analyzing and fixing button issues...\n\n";

// 1. Analyze current issues
echo "📋 Current Issues Identified:\n";
echo "❌ JavaScript may not be loading properly\n";
echo "❌ Form validation might be preventing submission\n";
echo "❌ CSRF token issues\n";
echo "❌ Missing jQuery or Bootstrap dependencies\n";
echo "❌ Button event handlers not working\n";

// 2. Create improved JavaScript
echo "\n📋 Creating improved JavaScript...\n";

$improvedJS = "
<script>
// Ensure DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    console.log('DOM loaded, initializing form...');
    
    // Get form element
    const form = document.getElementById('createForm');
    if (!form) {
        console.error('Form with ID createForm not found!');
        return;
    }
    
    // Get all buttons
    const saveBtn = document.getElementById('saveBtn');
    const altSubmitBtn = document.getElementById('altSubmitBtn');
    const simpleSubmitBtn = document.getElementById('simpleSubmitBtn');
    const testEndpointBtn = document.getElementById('testEndpointBtn');
    
    console.log('Buttons found:', {
        saveBtn: !!saveBtn,
        altSubmitBtn: !!altSubmitBtn,
        simpleSubmitBtn: !!simpleSubmitBtn,
        testEndpointBtn: !!testEndpointBtn
    });
    
    // Handle data source change
    const dataSourceSelect = document.getElementById('data_source');
    if (dataSourceSelect) {
        dataSourceSelect.addEventListener('change', function() {
            console.log('Data source changed to:', this.value);
            
            // Hide all data source configs
            const configs = document.querySelectorAll('.data-source-config');
            configs.forEach(config => config.style.display = 'none');
            
            // Show selected config
            const selectedConfig = document.getElementById(this.value + '_source');
            if (selectedConfig) {
                selectedConfig.style.display = 'block';
            }
        });
        
        // Trigger change on load
        dataSourceSelect.dispatchEvent(new Event('change'));
    }
    
    // Auto-generate endpoint from name
    const nameInput = document.getElementById('name');
    const endpointInput = document.getElementById('endpoint');
    
    if (nameInput && endpointInput) {
        nameInput.addEventListener('input', function() {
            const name = this.value;
            const endpoint = name.toLowerCase()
                .replace(/[^a-z0-9\\s\\-_]/g, '')
                .replace(/\\s+/g, '-')
                .replace(/\\-+/g, '-')
                .replace(/^-+|-+$/g, '');
            
            if (endpointInput.value === '' || endpointInput.dataset.autoGenerated === 'true') {
                endpointInput.value = endpoint;
                endpointInput.dataset.autoGenerated = 'true';
            }
        });
        
        endpointInput.addEventListener('input', function() {
            this.dataset.autoGenerated = 'false';
        });
    }
    
    // Form validation function
    function validateForm() {
        console.log('Validating form...');
        
        const requiredFields = ['name', 'endpoint', 'data_source', 'response_format'];
        const errors = [];
        
        requiredFields.forEach(fieldName => {
            const field = document.getElementById(fieldName);
            if (!field || !field.value.trim()) {
                errors.push(fieldName + ' is required');
            }
        });
        
        // Check conditional requirements
        const dataSource = document.getElementById('data_source').value;
        if (dataSource === 'table') {
            const sourceTable = document.getElementById('source_table');
            if (!sourceTable || !sourceTable.value) {
                errors.push('Source table is required when data source is table');
            }
        } else if (dataSource === 'api') {
            const sourceApiUrl = document.getElementById('source_api_url');
            if (!sourceApiUrl || !sourceApiUrl.value) {
                errors.push('API URL is required when data source is API');
            }
        } else if (dataSource === 'custom') {
            const customQuery = document.getElementById('custom_query');
            if (!customQuery || !customQuery.value) {
                errors.push('Custom query is required when data source is custom');
            }
        }
        
        if (errors.length > 0) {
            console.error('Validation errors:', errors);
            alert('Please fix the following errors:\\n' + errors.join('\\n'));
            return false;
        }
        
        console.log('Form validation passed');
        return true;
    }
    
    // Button event handlers
    if (saveBtn) {
        saveBtn.addEventListener('click', function(e) {
            console.log('Save button clicked');
            e.preventDefault();
            
            if (validateForm()) {
                console.log('Submitting form...');
                form.submit();
            }
        });
    }
    
    if (altSubmitBtn) {
        altSubmitBtn.addEventListener('click', function(e) {
            console.log('Alt submit button clicked');
            e.preventDefault();
            
            if (validateForm()) {
                console.log('Submitting form...');
                form.submit();
            }
        });
    }
    
    if (simpleSubmitBtn) {
        simpleSubmitBtn.addEventListener('click', function(e) {
            console.log('Simple submit button clicked');
            e.preventDefault();
            
            // Simple submit with minimal validation
            const name = document.getElementById('name').value;
            const endpoint = document.getElementById('endpoint').value;
            const dataSource = document.getElementById('data_source').value;
            
            if (!name || !endpoint || !dataSource) {
                alert('Please fill in Name, Endpoint, and Data Source');
                return;
            }
            
            console.log('Simple submitting form...');
            form.submit();
        });
    }
    
    if (testEndpointBtn) {
        testEndpointBtn.addEventListener('click', function(e) {
            console.log('Test endpoint button clicked');
            e.preventDefault();
            
            const endpoint = document.getElementById('endpoint').value;
            if (!endpoint) {
                alert('Please enter an endpoint first');
                return;
            }
            
            const testUrl = '/api/get/' + endpoint + '?test=1';
            console.log('Opening test URL:', testUrl);
            window.open(testUrl, '_blank');
        });
    }
    
    // Form submit handler
    form.addEventListener('submit', function(e) {
        console.log('Form submit event triggered');
        
        // Add loading state
        const submitButtons = form.querySelectorAll('button[type=submit], button[id*=Btn]');
        submitButtons.forEach(btn => {
            btn.disabled = true;
            btn.innerHTML = '<i class=\"fas fa-spinner fa-spin\"></i> Saving...';
        });
        
        // Let the form submit naturally
        return true;
    });
    
    console.log('Form initialization complete');
});

// Fallback for older browsers
if (typeof jQuery !== 'undefined') {
    $(document).ready(function() {
        console.log('jQuery fallback initialized');
        
        // Ensure buttons work with jQuery as well
        $('#saveBtn, #altSubmitBtn, #simpleSubmitBtn').on('click', function(e) {
            console.log('jQuery button clicked:', this.id);
            e.preventDefault();
            
            const form = $('#createForm');
            if (form.length) {
                form.submit();
            }
        });
    });
}
</script>
";

file_put_contents('improved-form-js.html', $improvedJS);
echo "✅ Improved JavaScript saved to: improved-form-js.html\n";

// 3. Create debug form
echo "\n📋 Creating debug form...\n";

$debugForm = '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Debug GET-Only Receiver Form</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <div class="container mt-4">
        <h2>Debug GET-Only Receiver Form</h2>
        <p class="text-muted">This form tests the basic functionality without Laravel dependencies</p>
        
        <form id="debugForm" action="https://copilotproduction.site/get-only-receivers" method="POST">
            <input type="hidden" name="_token" value="DEBUG_TOKEN">
            
            <div class="row">
                <div class="col-md-6">
                    <div class="mb-3">
                        <label for="name" class="form-label">Name *</label>
                        <input type="text" class="form-control" id="name" name="name" required>
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="mb-3">
                        <label for="endpoint" class="form-label">Endpoint *</label>
                        <input type="text" class="form-control" id="endpoint" name="endpoint" required>
                    </div>
                </div>
            </div>
            
            <div class="mb-3">
                <label for="data_source" class="form-label">Data Source *</label>
                <select class="form-control" id="data_source" name="data_source" required>
                    <option value="">Select Data Source</option>
                    <option value="table">Database Table</option>
                    <option value="api">External API</option>
                    <option value="custom">Custom Query</option>
                </select>
            </div>
            
            <div id="table_source" class="mb-3" style="display:none;">
                <label for="source_table" class="form-label">Source Table</label>
                <select class="form-control" id="source_table" name="source_table">
                    <option value="">Select Table</option>
                    <option value="users">users</option>
                    <option value="api_logs">api_logs</option>
                </select>
            </div>
            
            <div id="api_source" class="mb-3" style="display:none;">
                <label for="source_api_url" class="form-label">API URL</label>
                <input type="url" class="form-control" id="source_api_url" name="source_api_url">
            </div>
            
            <div id="custom_source" class="mb-3" style="display:none;">
                <label for="custom_query" class="form-label">Custom Query</label>
                <textarea class="form-control" id="custom_query" name="custom_query"></textarea>
            </div>
            
            <div class="row">
                <div class="col-md-6">
                    <div class="mb-3">
                        <label for="response_format" class="form-label">Response Format *</label>
                        <select class="form-control" id="response_format" name="response_format" required>
                            <option value="json">JSON</option>
                            <option value="xml">XML</option>
                            <option value="csv">CSV</option>
                        </select>
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="mb-3">
                        <label for="default_limit" class="form-label">Default Limit *</label>
                        <input type="number" class="form-control" id="default_limit" name="default_limit" value="50" required>
                    </div>
                </div>
            </div>
            
            <div class="row">
                <div class="col-md-6">
                    <div class="mb-3">
                        <label for="max_limit" class="form-label">Max Limit *</label>
                        <input type="number" class="form-control" id="max_limit" name="max_limit" value="1000" required>
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="mb-3">
                        <label for="cache_duration" class="form-label">Cache Duration *</label>
                        <input type="number" class="form-control" id="cache_duration" name="cache_duration" value="300" required>
                    </div>
                </div>
            </div>
            
            <div class="row">
                <div class="col-md-6">
                    <div class="mb-3">
                        <label for="rate_limit_per_minute" class="form-label">Rate Limit Per Minute *</label>
                        <input type="number" class="form-control" id="rate_limit_per_minute" name="rate_limit_per_minute" value="60" required>
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="mb-3">
                        <label for="rate_limit_per_hour" class="form-label">Rate Limit Per Hour *</label>
                        <input type="number" class="form-control" id="rate_limit_per_hour" name="rate_limit_per_hour" value="1000" required>
                    </div>
                </div>
            </div>
            
            <!-- Hidden fields for boolean values -->
            <input type="hidden" name="enable_pagination" value="1">
            <input type="hidden" name="require_token" value="0">
            <input type="hidden" name="enable_cors" value="1">
            <input type="hidden" name="enable_cache" value="1">
            <input type="hidden" name="is_active" value="1">
            
            <div class="mb-4">
                <button type="submit" class="btn btn-success me-2" id="saveBtn">
                    <i class="fas fa-save me-2"></i>SAVE GET-ONLY RECEIVER
                </button>
                <button type="button" class="btn btn-primary me-2" id="altSubmitBtn">
                    <i class="fas fa-rocket me-2"></i>SUBMIT NOW
                </button>
                <button type="button" class="btn btn-info me-2" id="simpleSubmitBtn">
                    <i class="fas fa-paper-plane me-2"></i>SIMPLE SUBMIT
                </button>
                <button type="button" class="btn btn-warning me-2" id="testEndpointBtn">
                    <i class="fas fa-vial me-2"></i>Test Endpoint
                </button>
            </div>
        </form>
        
        <div id="debugOutput" class="mt-4">
            <h4>Debug Output:</h4>
            <div id="debugLog" class="bg-light p-3" style="height: 200px; overflow-y: auto; font-family: monospace;"></div>
        </div>
    </div>
    
    <script>
    // Debug logging
    function debugLog(message) {
        const log = document.getElementById("debugLog");
        const timestamp = new Date().toLocaleTimeString();
        log.innerHTML += `[${timestamp}] ${message}<br>`;
        log.scrollTop = log.scrollHeight;
        console.log(message);
    }
    
    document.addEventListener("DOMContentLoaded", function() {
        debugLog("DOM loaded, initializing debug form...");
        
        const form = document.getElementById("debugForm");
        const dataSourceSelect = document.getElementById("data_source");
        
        // Data source change handler
        dataSourceSelect.addEventListener("change", function() {
            debugLog("Data source changed to: " + this.value);
            
            // Hide all source configs
            document.getElementById("table_source").style.display = "none";
            document.getElementById("api_source").style.display = "none";
            document.getElementById("custom_source").style.display = "none";
            
            // Show selected config
            if (this.value) {
                const targetDiv = document.getElementById(this.value + "_source");
                if (targetDiv) {
                    targetDiv.style.display = "block";
                }
            }
        });
        
        // Auto-generate endpoint
        document.getElementById("name").addEventListener("input", function() {
            const endpoint = this.value.toLowerCase()
                .replace(/[^a-z0-9\\s\\-_]/g, "")
                .replace(/\\s+/g, "-")
                .replace(/\\-+/g, "-")
                .replace(/^-+|-+$/g, "");
            
            const endpointInput = document.getElementById("endpoint");
            if (endpointInput.value === "" || endpointInput.dataset.autoGenerated === "true") {
                endpointInput.value = endpoint;
                endpointInput.dataset.autoGenerated = "true";
            }
        });
        
        // Button handlers
        document.getElementById("saveBtn").addEventListener("click", function(e) {
            debugLog("Save button clicked");
            e.preventDefault();
            if (validateAndSubmit()) {
                debugLog("Form submitted via save button");
            }
        });
        
        document.getElementById("altSubmitBtn").addEventListener("click", function(e) {
            debugLog("Alt submit button clicked");
            e.preventDefault();
            if (validateAndSubmit()) {
                debugLog("Form submitted via alt button");
            }
        });
        
        document.getElementById("simpleSubmitBtn").addEventListener("click", function(e) {
            debugLog("Simple submit button clicked");
            e.preventDefault();
            if (validateAndSubmit()) {
                debugLog("Form submitted via simple button");
            }
        });
        
        document.getElementById("testEndpointBtn").addEventListener("click", function(e) {
            debugLog("Test endpoint button clicked");
            e.preventDefault();
            
            const endpoint = document.getElementById("endpoint").value;
            if (!endpoint) {
                alert("Please enter an endpoint first");
                return;
            }
            
            const testUrl = "https://copilotproduction.site/api/get/" + endpoint + "?test=1";
            debugLog("Opening test URL: " + testUrl);
            window.open(testUrl, "_blank");
        });
        
        function validateAndSubmit() {
            debugLog("Validating form...");
            
            const requiredFields = ["name", "endpoint", "data_source", "response_format"];
            const errors = [];
            
            requiredFields.forEach(fieldName => {
                const field = document.getElementById(fieldName);
                if (!field || !field.value.trim()) {
                    errors.push(fieldName + " is required");
                }
            });
            
            if (errors.length > 0) {
                debugLog("Validation errors: " + errors.join(", "));
                alert("Please fix the following errors:\\n" + errors.join("\\n"));
                return false;
            }
            
            debugLog("Validation passed, submitting form...");
            form.submit();
            return true;
        }
        
        debugLog("Debug form initialization complete");
    });
    </script>
</body>
</html>';

file_put_contents('debug-get-only-receiver-form.html', $debugForm);
echo "✅ Debug form saved to: debug-get-only-receiver-form.html\n";

// 4. Create button fix patch
echo "\n📋 Creating button fix patch...\n";

$buttonFixPatch = '
<!-- Add this to the bottom of create.blade.php before @endsection -->

<script>
// Emergency button fix - ensure buttons work
document.addEventListener("DOMContentLoaded", function() {
    console.log("Emergency button fix loaded");
    
    // Get form
    const form = document.getElementById("createForm");
    if (!form) {
        console.error("Form not found!");
        return;
    }
    
    // Fix all buttons
    const buttons = [
        { id: "saveBtn", action: "submit" },
        { id: "altSubmitBtn", action: "submit" },
        { id: "simpleSubmitBtn", action: "submit" },
        { id: "testEndpointBtn", action: "test" }
    ];
    
    buttons.forEach(buttonConfig => {
        const button = document.getElementById(buttonConfig.id);
        if (button) {
            // Remove existing event listeners
            button.removeEventListener("click", arguments.callee);
            
            // Add new event listener
            button.addEventListener("click", function(e) {
                console.log("Button clicked:", buttonConfig.id);
                e.preventDefault();
                
                if (buttonConfig.action === "submit") {
                    // Basic validation
                    const name = document.getElementById("name").value;
                    const endpoint = document.getElementById("endpoint").value;
                    const dataSource = document.getElementById("data_source").value;
                    
                    if (!name || !endpoint || !dataSource) {
                        alert("Please fill in required fields: Name, Endpoint, and Data Source");
                        return;
                    }
                    
                    // Submit form
                    console.log("Submitting form...");
                    form.submit();
                    
                } else if (buttonConfig.action === "test") {
                    const endpoint = document.getElementById("endpoint").value;
                    if (!endpoint) {
                        alert("Please enter an endpoint first");
                        return;
                    }
                    
                    const testUrl = "/api/get/" + endpoint + "?test=1";
                    console.log("Opening test URL:", testUrl);
                    window.open(testUrl, "_blank");
                }
            });
            
            console.log("Fixed button:", buttonConfig.id);
        } else {
            console.warn("Button not found:", buttonConfig.id);
        }
    });
    
    console.log("Emergency button fix complete");
});
</script>
';

file_put_contents('button-fix-patch.html', $buttonFixPatch);
echo "✅ Button fix patch saved to: button-fix-patch.html\n";

// 5. Create comprehensive troubleshooting guide
echo "\n📋 Creating troubleshooting guide...\n";

$troubleshootingGuide = '# Button Functionality Troubleshooting Guide

## 🚨 BUTTON ISSUES IDENTIFIED

### Common Problems:
1. **JavaScript not loading** - Missing jQuery or Bootstrap dependencies
2. **Form validation errors** - Required fields preventing submission
3. **CSRF token missing** - Laravel security preventing form submission
4. **Database table missing** - Backend errors preventing save
5. **Event handlers not attached** - JavaScript timing issues

## 🔧 IMMEDIATE FIXES

### 1. Test with Debug Form
Open `debug-get-only-receiver-form.html` in your browser:
- Tests basic button functionality
- Shows debug output in console
- Bypasses Laravel dependencies

### 2. Check Browser Console
1. Open Developer Tools (F12)
2. Go to Console tab
3. Look for JavaScript errors
4. Check if jQuery is loaded

### 3. Verify Form Structure
- Form ID should be `createForm`
- All buttons should have correct IDs
- CSRF token should be present

### 4. Apply Emergency Fix
Add the content from `button-fix-patch.html` to your create.blade.php file before @endsection

## 🎯 STEP-BY-STEP DEBUGGING

### Step 1: Basic Button Test
```javascript
// Run in browser console
document.getElementById("saveBtn").click();
```

### Step 2: Check Form Submission
```javascript
// Run in browser console
document.getElementById("createForm").submit();
```

### Step 3: Manual Form Data
```javascript
// Run in browser console
const form = document.getElementById("createForm");
const formData = new FormData(form);
for (let [key, value] of formData.entries()) {
    console.log(key, value);
}
```

## 📋 COMMON SOLUTIONS

### Solution 1: Missing Database Table
```sql
-- Run in phpMyAdmin
-- Use content from create-table-manual.sql
```

### Solution 2: CSRF Token Issue
```html
<!-- Check if this exists in form -->
<input type="hidden" name="_token" value="{{ csrf_token() }}">
```

### Solution 3: JavaScript Dependencies
```html
<!-- Add to layout if missing -->
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
```

### Solution 4: Form Validation
```javascript
// Add basic validation
function validateForm() {
    const required = ["name", "endpoint", "data_source", "response_format"];
    for (let field of required) {
        const input = document.getElementById(field);
        if (!input || !input.value.trim()) {
            alert(field + " is required");
            return false;
        }
    }
    return true;
}
```

## 🔍 DEBUGGING CHECKLIST

- [ ] Database table `get_only_receivers` exists
- [ ] CSRF token is present in form
- [ ] JavaScript console shows no errors
- [ ] All button IDs are correct
- [ ] Form ID is `createForm`
- [ ] Required fields have values
- [ ] jQuery/Bootstrap are loaded
- [ ] Network tab shows form submission

## 🎉 EXPECTED BEHAVIOR

After fixing:
1. Click any submit button
2. Form validates required fields
3. Shows loading state
4. Submits to server
5. Redirects to index page
6. New receiver appears in list

## 📝 FILES TO USE

1. `debug-get-only-receiver-form.html` - Test basic functionality
2. `button-fix-patch.html` - Emergency JavaScript fix
3. `improved-form-js.html` - Complete JavaScript solution
4. `create-table-manual.sql` - Database table creation

## 🚀 QUICK FIX STEPS

1. **Create database table** (most important)
2. **Test with debug form** to verify functionality
3. **Apply button fix patch** if needed
4. **Check browser console** for errors
5. **Verify all required fields** are filled

The buttons should work after these fixes!
';

file_put_contents('button-troubleshooting-guide.md', $troubleshootingGuide);
echo "✅ Troubleshooting guide saved to: button-troubleshooting-guide.md\n";

echo "\n🎯 SUMMARY:\n";
echo "❌ **Button Issues**: JavaScript timing, validation, dependencies\n";
echo "✅ **Solutions Created**:\n";
echo "1. Debug form for testing\n";
echo "2. Emergency button fix patch\n";
echo "3. Improved JavaScript code\n";
echo "4. Complete troubleshooting guide\n";

echo "\n🚀 **NEXT STEPS**:\n";
echo "1. Test with debug-get-only-receiver-form.html\n";
echo "2. Apply button-fix-patch.html to your form\n";
echo "3. Create database table if not done yet\n";
echo "4. Check browser console for errors\n";

echo "\n🎉 Button fix preparation completed!\n"; 