<?php

/**
 * Fix untuk masalah form save GET-Only Receivers
 */

echo "=== FIX FORM SAVE ISSUE ===\n";
echo "🔧 Fixing GET-Only Receivers form save problems...\n\n";

// 1. Fix controller validation for array fields
echo "📋 Fixing controller validation...\n";

$controllerFix = '
// Add this to GetOnlyReceiverController store method before validation

// Process array fields from comma-separated strings
if ($request->has("response_fields") && is_string($request->response_fields)) {
    $fields = array_filter(array_map("trim", explode(",", $request->response_fields)));
    $request->merge(["response_fields" => $fields]);
}

if ($request->has("cors_origins") && is_string($request->cors_origins)) {
    $origins = array_filter(array_map("trim", explode(",", $request->cors_origins)));
    $request->merge(["cors_origins" => $origins]);
}

// Handle boolean fields properly
$booleanFields = ["enable_pagination", "require_token", "enable_cors", "enable_cache", "is_active"];
foreach ($booleanFields as $field) {
    if (!$request->has($field)) {
        $request->merge([$field => false]);
    }
}
';

echo "✅ Controller validation fix prepared\n";

// 2. Create manual SQL for table creation
echo "\n📋 Creating manual table creation SQL...\n";

$createTableSQL = "
-- Manual creation of get_only_receivers table
CREATE TABLE IF NOT EXISTS get_only_receivers (
    id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
    name varchar(255) NOT NULL,
    description text DEFAULT NULL,
    endpoint varchar(255) NOT NULL,
    data_source enum('table','api','custom') NOT NULL,
    source_table varchar(255) DEFAULT NULL,
    source_api_url varchar(500) DEFAULT NULL,
    source_api_headers json DEFAULT NULL,
    custom_query text DEFAULT NULL,
    api_token_id bigint(20) unsigned DEFAULT NULL,
    require_token tinyint(1) NOT NULL DEFAULT 0,
    allowed_tokens json DEFAULT NULL,
    response_format enum('json','xml','csv') NOT NULL DEFAULT 'json',
    response_fields json DEFAULT NULL,
    default_filters json DEFAULT NULL,
    allowed_filters json DEFAULT NULL,
    default_limit int(11) NOT NULL DEFAULT 50,
    max_limit int(11) NOT NULL DEFAULT 1000,
    enable_pagination tinyint(1) NOT NULL DEFAULT 1,
    rate_limit_per_minute int(11) NOT NULL DEFAULT 60,
    rate_limit_per_hour int(11) NOT NULL DEFAULT 1000,
    allowed_ips json DEFAULT NULL,
    enable_cors tinyint(1) NOT NULL DEFAULT 1,
    cors_origins json DEFAULT NULL,
    enable_cache tinyint(1) NOT NULL DEFAULT 1,
    cache_duration int(11) NOT NULL DEFAULT 300,
    cache_key_prefix varchar(100) DEFAULT NULL,
    validation_rules json DEFAULT NULL,
    required_parameters json DEFAULT NULL,
    optional_parameters json DEFAULT NULL,
    is_active tinyint(1) NOT NULL DEFAULT 1,
    total_requests int(11) NOT NULL DEFAULT 0,
    total_success int(11) NOT NULL DEFAULT 0,
    total_errors int(11) NOT NULL DEFAULT 0,
    last_accessed_at timestamp NULL DEFAULT NULL,
    avg_response_time decimal(8,3) DEFAULT NULL,
    metadata json DEFAULT NULL,
    created_by bigint(20) unsigned DEFAULT NULL,
    created_at timestamp NULL DEFAULT NULL,
    updated_at timestamp NULL DEFAULT NULL,
    PRIMARY KEY (id),
    UNIQUE KEY get_only_receivers_endpoint_unique (endpoint)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
";

file_put_contents('create-table-manual.sql', $createTableSQL);
echo "✅ Manual table creation SQL saved to: create-table-manual.sql\n";

// 3. Create controller patch
echo "\n📋 Creating controller patch...\n";

$controllerPatch = '<?php

// Patch for GetOnlyReceiverController.php store method
// Add this code before the validator in the store method

public function store(Request $request)
{
    // PRE-PROCESS REQUEST DATA
    
    // 1. Handle array fields from comma-separated strings
    if ($request->has("response_fields") && is_string($request->response_fields)) {
        $fields = array_filter(array_map("trim", explode(",", $request->response_fields)));
        $request->merge(["response_fields" => $fields]);
    }

    if ($request->has("cors_origins") && is_string($request->cors_origins)) {
        $origins = array_filter(array_map("trim", explode(",", $request->cors_origins)));
        $request->merge(["cors_origins" => $origins]);
    }

    // 2. Handle boolean fields properly (checkboxes that are unchecked dont send values)
    $booleanFields = ["enable_pagination", "require_token", "enable_cors", "enable_cache", "is_active"];
    foreach ($booleanFields as $field) {
        if (!$request->has($field)) {
            $request->merge([$field => false]);
        } else {
            $request->merge([$field => true]);
        }
    }

    // 3. Set default values for optional fields
    if (!$request->has("description")) {
        $request->merge(["description" => null]);
    }

    // 4. Generate endpoint from name if empty
    if (!$request->has("endpoint") || empty($request->endpoint)) {
        $endpoint = Str::slug($request->name) . "-" . Str::random(6);
        $request->merge(["endpoint" => $endpoint]);
    }

    // CONTINUE WITH EXISTING VALIDATION...
    $validator = Validator::make($request->all(), [
        // ... existing validation rules
    ]);
    
    // REST OF THE METHOD REMAINS THE SAME
}
';

file_put_contents('controller-patch.php', $controllerPatch);
echo "✅ Controller patch saved to: controller-patch.php\n";

// 4. Create simple working form
echo "\n📋 Creating simple working form...\n";

$simpleForm = '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Simple GET-Only Receiver Form</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .form-group { margin-bottom: 15px; }
        label { display: block; margin-bottom: 5px; font-weight: bold; }
        input, select, textarea { width: 100%; padding: 8px; margin-bottom: 5px; }
        button { background: #007bff; color: white; padding: 10px 20px; border: none; cursor: pointer; }
        .required { color: red; }
        .error { color: red; font-size: 12px; }
    </style>
</head>
<body>
    <h2>Create GET-Only Receiver (Simple Form)</h2>
    
    <form action="https://copilotproduction.site/get-only-receivers" method="POST">
        <input type="hidden" name="_token" value="REPLACE_WITH_CSRF_TOKEN">
        
        <div class="form-group">
            <label>Name <span class="required">*</span></label>
            <input type="text" name="name" required placeholder="My API Endpoint">
        </div>
        
        <div class="form-group">
            <label>Description</label>
            <textarea name="description" placeholder="Optional description"></textarea>
        </div>
        
        <div class="form-group">
            <label>Endpoint <span class="required">*</span></label>
            <input type="text" name="endpoint" required placeholder="my-endpoint" pattern="[a-zA-Z0-9\\-_]+">
            <div class="error">Only letters, numbers, dash, and underscore allowed</div>
        </div>
        
        <div class="form-group">
            <label>Data Source <span class="required">*</span></label>
            <select name="data_source" required onchange="toggleDataSource(this.value)">
                <option value="">Select Data Source</option>
                <option value="table">Database Table</option>
                <option value="api">External API</option>
                <option value="custom">Custom Query</option>
            </select>
        </div>
        
        <div class="form-group" id="table_source" style="display:none;">
            <label>Source Table</label>
            <select name="source_table">
                <option value="">Select Table</option>
                <option value="users">users</option>
                <option value="api_logs">api_logs</option>
                <option value="api_clients">api_clients</option>
            </select>
        </div>
        
        <div class="form-group" id="api_source" style="display:none;">
            <label>API URL</label>
            <input type="url" name="source_api_url" placeholder="https://api.example.com/data">
        </div>
        
        <div class="form-group" id="custom_source" style="display:none;">
            <label>Custom SQL Query</label>
            <textarea name="custom_query" placeholder="SELECT * FROM users WHERE active = 1"></textarea>
        </div>
        
        <div class="form-group">
            <label>Response Format <span class="required">*</span></label>
            <select name="response_format" required>
                <option value="json">JSON</option>
                <option value="xml">XML</option>
                <option value="csv">CSV</option>
            </select>
        </div>
        
        <div class="form-group">
            <label>Default Limit <span class="required">*</span></label>
            <input type="number" name="default_limit" value="50" min="1" max="1000" required>
        </div>
        
        <div class="form-group">
            <label>Maximum Limit <span class="required">*</span></label>
            <input type="number" name="max_limit" value="1000" min="1" max="10000" required>
        </div>
        
        <div class="form-group">
            <label>Rate Limit Per Minute <span class="required">*</span></label>
            <input type="number" name="rate_limit_per_minute" value="60" min="1" max="1000" required>
        </div>
        
        <div class="form-group">
            <label>Rate Limit Per Hour <span class="required">*</span></label>
            <input type="number" name="rate_limit_per_hour" value="1000" min="1" max="10000" required>
        </div>
        
        <div class="form-group">
            <label>Cache Duration (seconds) <span class="required">*</span></label>
            <input type="number" name="cache_duration" value="300" min="60" max="86400" required>
        </div>
        
        <div class="form-group">
            <label>Response Fields (comma-separated, optional)</label>
            <input type="text" name="response_fields" placeholder="id,name,email">
        </div>
        
        <div class="form-group">
            <label>CORS Origins (comma-separated, optional)</label>
            <input type="text" name="cors_origins" placeholder="https://example.com,https://app.example.com">
        </div>
        
        <!-- Boolean fields as hidden inputs with default values -->
        <input type="hidden" name="enable_pagination" value="1">
        <input type="hidden" name="require_token" value="0">
        <input type="hidden" name="enable_cors" value="1">
        <input type="hidden" name="enable_cache" value="1">
        <input type="hidden" name="is_active" value="1">
        
        <button type="submit">Save GET-Only Receiver</button>
    </form>
    
    <script>
    function toggleDataSource(source) {
        document.getElementById("table_source").style.display = source === "table" ? "block" : "none";
        document.getElementById("api_source").style.display = source === "api" ? "block" : "none";
        document.getElementById("custom_source").style.display = source === "custom" ? "block" : "none";
    }
    </script>
</body>
</html>';

file_put_contents('simple-get-only-receiver-form.html', $simpleForm);
echo "✅ Simple working form saved to: simple-get-only-receiver-form.html\n";

// 5. Create troubleshooting guide
echo "\n📋 Creating troubleshooting guide...\n";

$troubleshootingGuide = '# GET-Only Receivers Form Save Troubleshooting

## 🚨 MAIN ISSUE: Database Table Missing

The primary reason the form cannot save is that the `get_only_receivers` table does not exist in the database.

## 🔧 IMMEDIATE FIXES:

### 1. Create Database Table
Run this SQL in phpMyAdmin:
```sql
' . trim($createTableSQL) . '
```

### 2. Check Form Validation
- Ensure all required fields have values
- Check browser console for JavaScript errors
- Verify CSRF token is present

### 3. Controller Issues
The controller needs to handle:
- Array fields (response_fields, cors_origins) from comma-separated strings
- Boolean fields that are unchecked (they dont send values)
- Default values for optional fields

## 🎯 STEP-BY-STEP SOLUTION:

1. **Create Table**: Run the SQL file in phpMyAdmin
2. **Test Simple Form**: Use simple-get-only-receiver-form.html
3. **Check Logs**: Look at Laravel logs for detailed errors
4. **Apply Controller Patch**: Use the controller patch if needed

## 📝 FILES CREATED:
- create-table-manual.sql - Manual table creation
- controller-patch.php - Controller fixes
- simple-get-only-receiver-form.html - Working test form

## 🔍 DEBUGGING STEPS:
1. Check if table exists: `SHOW TABLES LIKE "get_only_receivers"`
2. Check Laravel logs: storage/logs/laravel.log
3. Check browser console for JavaScript errors
4. Test with simple form first

## ✅ EXPECTED RESULT:
After creating the table, the form should save successfully and redirect to the GET-Only Receivers index page.
';

file_put_contents('troubleshooting-guide.md', $troubleshootingGuide);
echo "✅ Troubleshooting guide saved to: troubleshooting-guide.md\n";

echo "\n🎯 SUMMARY OF ISSUES & FIXES:\n";
echo "❌ **Main Issue**: get_only_receivers table does not exist\n";
echo "❌ **Secondary Issues**: Array field handling, boolean field validation\n";
echo "\n✅ **Solutions Created**:\n";
echo "1. Manual table creation SQL\n";
echo "2. Controller patch for data processing\n";
echo "3. Simple working test form\n";
echo "4. Complete troubleshooting guide\n";

echo "\n🚀 **NEXT STEPS**:\n";
echo "1. Run create-table-manual.sql in phpMyAdmin\n";
echo "2. Test with simple-get-only-receiver-form.html\n";
echo "3. Apply controller patch if still having issues\n";
echo "4. Check troubleshooting guide for detailed steps\n";

echo "\n🎉 Fix preparation completed!\n";
echo "The main issue is the missing database table. Create it and the form should work.\n"; 