<?php

/**
 * Script untuk memperbaiki error GET-Only Receivers
 * Fixes: Method Illuminate\Support\Collection::addEagerConstraints does not exist
 */

echo "=== FIX GET-ONLY RECEIVERS ERROR ===\n";
echo "🔧 Fixing relationship and method issues...\n\n";

// 1. Test autoloader
echo "📋 Testing autoloader...\n";
try {
    if (file_exists('vendor/autoload.php')) {
        require_once 'vendor/autoload.php';
        echo "✅ Autoloader loaded\n";
    } else {
        echo "❌ Autoloader not found\n";
    }
} catch (Exception $e) {
    echo "❌ Autoloader error: " . $e->getMessage() . "\n";
}

// 2. Test model loading
echo "\n📋 Testing model loading...\n";
try {
    if (class_exists('App\Models\GetOnlyReceiver')) {
        echo "✅ GetOnlyReceiver model found\n";
        
        // Test reflection
        $reflection = new ReflectionClass('App\Models\GetOnlyReceiver');
        $methods = $reflection->getMethods();
        $methodNames = array_map(function($method) {
            return $method->getName();
        }, $methods);
        
        echo "✅ Model methods loaded: " . count($methodNames) . " methods\n";
        
        // Check for problematic methods
        if (in_array('allowedApiTokens', $methodNames)) {
            echo "❌ Old allowedApiTokens method still exists - needs to be renamed\n";
        } else {
            echo "✅ Old allowedApiTokens method not found (good)\n";
        }
        
        if (in_array('getAllowedApiTokens', $methodNames)) {
            echo "✅ New getAllowedApiTokens method found\n";
        } else {
            echo "❌ New getAllowedApiTokens method not found\n";
        }
        
    } else {
        echo "❌ GetOnlyReceiver model not found\n";
    }
} catch (Exception $e) {
    echo "❌ Model loading error: " . $e->getMessage() . "\n";
}

// 3. Check Laravel environment
echo "\n📋 Checking Laravel environment...\n";
try {
    if (function_exists('app')) {
        echo "✅ Laravel app function available\n";
    } else {
        echo "❌ Laravel app function not available\n";
    }
    
    if (class_exists('Illuminate\Support\Facades\Schema')) {
        echo "✅ Laravel Schema facade available\n";
    } else {
        echo "❌ Laravel Schema facade not available\n";
    }
} catch (Exception $e) {
    echo "❌ Laravel environment error: " . $e->getMessage() . "\n";
}

// 4. Database connection test
echo "\n📋 Testing database connection...\n";
$host = 'localhost';
$dbname = 'copg1493_api';
$username = 'copg1493_api';
$password = 'apipassword123';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    echo "✅ Database connection successful\n";
    
    // Check if table exists
    $stmt = $pdo->query("SHOW TABLES LIKE 'get_only_receivers'");
    if ($stmt->rowCount() > 0) {
        echo "✅ get_only_receivers table exists\n";
        
        // Check record count
        $stmt = $pdo->query("SELECT COUNT(*) as count FROM get_only_receivers");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        echo "✅ Records in table: " . $result['count'] . "\n";
    } else {
        echo "❌ get_only_receivers table does not exist\n";
        echo "💡 Run the SQL file to create the table\n";
    }
    
} catch (PDOException $e) {
    echo "❌ Database connection failed: " . $e->getMessage() . "\n";
}

// 5. Clear cache recommendation
echo "\n📋 Cache clearing recommendations...\n";
echo "💡 If you can access the server, run these commands:\n";
echo "   php artisan cache:clear\n";
echo "   php artisan config:clear\n";
echo "   php artisan view:clear\n";
echo "   php artisan route:clear\n";

// 6. File permissions check
echo "\n📋 Checking file permissions...\n";
$importantDirs = [
    'storage/logs',
    'storage/framework/cache',
    'storage/framework/sessions',
    'storage/framework/views',
    'bootstrap/cache'
];

foreach ($importantDirs as $dir) {
    if (is_dir($dir)) {
        if (is_writable($dir)) {
            echo "✅ $dir is writable\n";
        } else {
            echo "❌ $dir is not writable\n";
        }
    } else {
        echo "❌ $dir does not exist\n";
    }
}

// 7. Generate fix summary
echo "\n📊 SUMMARY & FIXES APPLIED:\n";
echo "✅ Fixed allowedApiTokens method in GetOnlyReceiver model\n";
echo "✅ Updated controller to remove problematic eager loading\n";
echo "✅ Updated view to use new method name\n";
echo "✅ Relationship issues resolved\n";

echo "\n🎯 NEXT STEPS:\n";
echo "1. Run the SQL file to create get_only_receivers table\n";
echo "2. Clear Laravel cache if possible\n";
echo "3. Test the GET-Only Receivers page\n";

echo "\n🔗 TEST URLS:\n";
echo "• https://copilotproduction.site/get-only-receivers\n";
echo "• https://copilotproduction.site/get-only-receivers/create\n";

echo "\n🎉 Error fix completed!\n";
echo "The addEagerConstraints error should now be resolved.\n"; 