<?php

/**
 * Fix GET-Only Receivers Save Error
 * Comprehensive solution to fix the form save issue
 */

echo "=== FIX GET-ONLY RECEIVERS SAVE ERROR ===\n";
echo "🔧 Diagnosing and fixing the save error...\n\n";

// Step 1: Check database connection
echo "📋 Step 1: Checking database connection...\n";

// Database configuration from Laravel .env
$dbConfig = [
    'host' => '127.0.0.1',
    'database' => 'copg1493_api',
    'username' => 'copg1493_api',
    'password' => 'UFXHmm5i8zfUqQP9'
];

try {
    $pdo = new PDO(
        "mysql:host={$dbConfig['host']};dbname={$dbConfig['database']};charset=utf8mb4", 
        $dbConfig['username'], 
        $dbConfig['password'],
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
        ]
    );
    echo "✅ Database connection successful\n";
} catch (PDOException $e) {
    echo "❌ Database connection failed: " . $e->getMessage() . "\n";
    echo "🔧 Please check your database configuration in .env file\n";
    echo "📝 Current config:\n";
    echo "   Host: {$dbConfig['host']}\n";
    echo "   Database: {$dbConfig['database']}\n";
    echo "   Username: {$dbConfig['username']}\n";
    echo "\n⚠️  Cannot proceed without database connection\n";
    exit(1);
}

// Step 2: Check if table exists
echo "\n📋 Step 2: Checking if get_only_receivers table exists...\n";

try {
    $stmt = $pdo->query("SHOW TABLES LIKE 'get_only_receivers'");
    $tableExists = $stmt->rowCount() > 0;
    
    if ($tableExists) {
        echo "✅ Table 'get_only_receivers' exists\n";
        
        // Check table structure
        $stmt = $pdo->query("DESCRIBE get_only_receivers");
        $columns = $stmt->fetchAll();
        echo "📊 Table has " . count($columns) . " columns\n";
        
        // Check for required columns
        $requiredColumns = ['name', 'endpoint', 'data_source', 'response_format', 'default_limit', 'max_limit', 'cache_duration'];
        $existingColumns = array_column($columns, 'Field');
        $missingColumns = array_diff($requiredColumns, $existingColumns);
        
        if (!empty($missingColumns)) {
            echo "⚠️  Missing columns: " . implode(', ', $missingColumns) . "\n";
            echo "🔧 Will recreate table with correct structure\n";
            $tableExists = false;
        }
    } else {
        echo "❌ Table 'get_only_receivers' does not exist\n";
    }
} catch (PDOException $e) {
    echo "❌ Error checking table: " . $e->getMessage() . "\n";
    $tableExists = false;
}

// Step 3: Create or recreate table
if (!$tableExists) {
    echo "\n📋 Step 3: Creating get_only_receivers table...\n";
    
    try {
        // Drop table if exists (for recreation)
        $pdo->exec("DROP TABLE IF EXISTS get_only_receivers");
        echo "🗑️  Dropped existing table (if any)\n";
        
        // Create table with complete structure
        $createTableSQL = "
        CREATE TABLE get_only_receivers (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            name varchar(255) NOT NULL COMMENT 'Descriptive name for the receiver',
            description text DEFAULT NULL COMMENT 'Optional description',
            endpoint varchar(255) NOT NULL COMMENT 'URL endpoint (alphanumeric, dash, underscore only)',
            data_source enum('table','api','custom') NOT NULL COMMENT 'Source of data: database table, external API, or custom query',
            source_table varchar(255) DEFAULT NULL COMMENT 'Database table name (if data_source is table)',
            source_api_url varchar(500) DEFAULT NULL COMMENT 'External API URL (if data_source is api)',
            source_api_headers json DEFAULT NULL COMMENT 'Headers for external API requests',
            custom_query text DEFAULT NULL COMMENT 'Custom SQL query (if data_source is custom)',
            api_token_id bigint(20) unsigned DEFAULT NULL COMMENT 'Associated API token for authentication',
            require_token tinyint(1) NOT NULL DEFAULT 0 COMMENT 'Whether authentication is required',
            allowed_tokens json DEFAULT NULL COMMENT 'List of allowed token IDs',
            response_format enum('json','xml','csv') NOT NULL DEFAULT 'json' COMMENT 'Response format',
            response_fields json DEFAULT NULL COMMENT 'Fields to include in response',
            default_filters json DEFAULT NULL COMMENT 'Default filters to apply',
            allowed_filters json DEFAULT NULL COMMENT 'Allowed filter parameters',
            default_limit int(11) NOT NULL DEFAULT 50 COMMENT 'Default number of records to return',
            max_limit int(11) NOT NULL DEFAULT 1000 COMMENT 'Maximum allowed limit',
            enable_pagination tinyint(1) NOT NULL DEFAULT 1 COMMENT 'Whether to enable pagination',
            rate_limit_per_minute int(11) NOT NULL DEFAULT 60 COMMENT 'Rate limit per minute',
            rate_limit_per_hour int(11) NOT NULL DEFAULT 1000 COMMENT 'Rate limit per hour',
            allowed_ips json DEFAULT NULL COMMENT 'IP whitelist',
            enable_cors tinyint(1) NOT NULL DEFAULT 1 COMMENT 'Enable CORS headers',
            cors_origins json DEFAULT NULL COMMENT 'Allowed CORS origins',
            enable_cache tinyint(1) NOT NULL DEFAULT 1 COMMENT 'Enable response caching',
            cache_duration int(11) NOT NULL DEFAULT 300 COMMENT 'Cache duration in seconds',
            cache_key_prefix varchar(100) DEFAULT NULL COMMENT 'Cache key prefix',
            validation_rules json DEFAULT NULL COMMENT 'Validation rules for parameters',
            required_parameters json DEFAULT NULL COMMENT 'Required query parameters',
            optional_parameters json DEFAULT NULL COMMENT 'Optional query parameters',
            is_active tinyint(1) NOT NULL DEFAULT 1 COMMENT 'Whether the receiver is active',
            total_requests int(11) NOT NULL DEFAULT 0 COMMENT 'Total number of requests received',
            total_success int(11) NOT NULL DEFAULT 0 COMMENT 'Total successful responses',
            total_errors int(11) NOT NULL DEFAULT 0 COMMENT 'Total error responses',
            last_accessed_at timestamp NULL DEFAULT NULL COMMENT 'Last time the receiver was accessed',
            avg_response_time decimal(8,3) DEFAULT NULL COMMENT 'Average response time in seconds',
            metadata json DEFAULT NULL COMMENT 'Additional metadata',
            created_by bigint(20) unsigned DEFAULT NULL COMMENT 'User who created this receiver',
            created_at timestamp NULL DEFAULT NULL,
            updated_at timestamp NULL DEFAULT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY get_only_receivers_endpoint_unique (endpoint),
            KEY get_only_receivers_api_token_id_foreign (api_token_id),
            KEY get_only_receivers_created_by_foreign (created_by),
            KEY get_only_receivers_data_source_index (data_source),
            KEY get_only_receivers_is_active_index (is_active),
            KEY get_only_receivers_response_format_index (response_format)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='GET-only API receivers for providing data endpoints'
        ";
        
        $pdo->exec($createTableSQL);
        echo "✅ Table 'get_only_receivers' created successfully\n";
        
        // Add migration record
        try {
            $stmt = $pdo->prepare("INSERT IGNORE INTO migrations (migration, batch) VALUES (?, ?)");
            $stmt->execute(['2025_07_05_104252_create_get_only_receivers_table', 1]);
            echo "✅ Migration record added\n";
        } catch (PDOException $e) {
            echo "⚠️  Could not add migration record: " . $e->getMessage() . "\n";
        }
        
    } catch (PDOException $e) {
        echo "❌ Error creating table: " . $e->getMessage() . "\n";
        echo "\n📝 Manual SQL to create table:\n";
        echo "Copy this SQL to phpMyAdmin and run it:\n\n";
        echo $createTableSQL . "\n";
        exit(1);
    }
} else {
    echo "\n✅ Table structure is correct\n";
}

// Step 4: Test table functionality
echo "\n📋 Step 4: Testing table functionality...\n";

try {
    // Test insert
    $testData = [
        'name' => 'Test Receiver',
        'endpoint' => 'test-' . time(),
        'data_source' => 'table',
        'source_table' => 'users',
        'response_format' => 'json',
        'default_limit' => 50,
        'max_limit' => 1000,
        'cache_duration' => 300,
        'created_at' => date('Y-m-d H:i:s'),
        'updated_at' => date('Y-m-d H:i:s')
    ];
    
    $columns = implode(', ', array_keys($testData));
    $placeholders = ':' . implode(', :', array_keys($testData));
    
    $stmt = $pdo->prepare("INSERT INTO get_only_receivers ($columns) VALUES ($placeholders)");
    $stmt->execute($testData);
    
    $testId = $pdo->lastInsertId();
    echo "✅ Test insert successful (ID: $testId)\n";
    
    // Test select
    $stmt = $pdo->prepare("SELECT * FROM get_only_receivers WHERE id = ?");
    $stmt->execute([$testId]);
    $result = $stmt->fetch();
    
    if ($result) {
        echo "✅ Test select successful\n";
    } else {
        echo "❌ Test select failed\n";
    }
    
    // Clean up test data
    $stmt = $pdo->prepare("DELETE FROM get_only_receivers WHERE id = ?");
    $stmt->execute([$testId]);
    echo "✅ Test data cleaned up\n";
    
} catch (PDOException $e) {
    echo "❌ Table functionality test failed: " . $e->getMessage() . "\n";
}

// Step 5: Check Laravel application
echo "\n📋 Step 5: Checking Laravel application...\n";

// Check if we can connect to Laravel
if (file_exists('artisan')) {
    echo "✅ Laravel application found\n";
    
    // Check if Model exists
    if (file_exists('app/Models/GetOnlyReceiver.php')) {
        echo "✅ GetOnlyReceiver model exists\n";
    } else {
        echo "❌ GetOnlyReceiver model missing\n";
    }
    
    // Check if Controller exists
    if (file_exists('app/Http/Controllers/GetOnlyReceiverController.php')) {
        echo "✅ GetOnlyReceiverController exists\n";
    } else {
        echo "❌ GetOnlyReceiverController missing\n";
    }
    
    // Check routes
    if (file_exists('routes/web.php')) {
        $routes = file_get_contents('routes/web.php');
        if (strpos($routes, 'get-only-receivers') !== false) {
            echo "✅ Routes configured\n";
        } else {
            echo "❌ Routes not configured\n";
        }
    }
    
} else {
    echo "❌ Laravel application not found\n";
}

// Step 6: Create test form
echo "\n📋 Step 6: Creating test form...\n";

$testFormHTML = '<!DOCTYPE html>
<html>
<head>
    <title>Test GET-Only Receiver Form</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <style>
        body { font-family: Arial, sans-serif; max-width: 800px; margin: 20px auto; padding: 20px; }
        .form-group { margin-bottom: 15px; }
        label { display: block; margin-bottom: 5px; font-weight: bold; }
        input, select, textarea { width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; }
        button { background: #007bff; color: white; padding: 10px 20px; border: none; border-radius: 4px; cursor: pointer; }
        button:hover { background: #0056b3; }
        .required { color: red; }
        .success { color: green; }
        .error { color: red; }
    </style>
</head>
<body>
    <h2>Test GET-Only Receiver Form</h2>
    <p class="success">✅ Database table is ready!</p>
    
    <form action="https://copilotproduction.site/get-only-receivers" method="POST">
        <input type="hidden" name="_token" value="REPLACE_WITH_CSRF_TOKEN">
        
        <div class="form-group">
            <label>Name <span class="required">*</span></label>
            <input type="text" name="name" required placeholder="My Test API">
        </div>
        
        <div class="form-group">
            <label>Description</label>
            <textarea name="description" placeholder="Optional description"></textarea>
        </div>
        
        <div class="form-group">
            <label>Endpoint <span class="required">*</span></label>
            <input type="text" name="endpoint" required placeholder="my-endpoint" pattern="[a-zA-Z0-9\\-_]+">
        </div>
        
        <div class="form-group">
            <label>Data Source <span class="required">*</span></label>
            <select name="data_source" required>
                <option value="">Select Data Source</option>
                <option value="table">Database Table</option>
                <option value="api">External API</option>
                <option value="custom">Custom Query</option>
            </select>
        </div>
        
        <div class="form-group">
            <label>Source Table</label>
            <select name="source_table">
                <option value="">Select Table</option>
                <option value="users">users</option>
                <option value="api_logs">api_logs</option>
                <option value="api_clients">api_clients</option>
            </select>
        </div>
        
        <div class="form-group">
            <label>Response Format <span class="required">*</span></label>
            <select name="response_format" required>
                <option value="json">JSON</option>
                <option value="xml">XML</option>
                <option value="csv">CSV</option>
            </select>
        </div>
        
        <div class="form-group">
            <label>Default Limit <span class="required">*</span></label>
            <input type="number" name="default_limit" value="50" min="1" max="1000" required>
        </div>
        
        <div class="form-group">
            <label>Maximum Limit <span class="required">*</span></label>
            <input type="number" name="max_limit" value="1000" min="1" max="10000" required>
        </div>
        
        <div class="form-group">
            <label>Rate Limit Per Minute <span class="required">*</span></label>
            <input type="number" name="rate_limit_per_minute" value="60" min="1" max="1000" required>
        </div>
        
        <div class="form-group">
            <label>Rate Limit Per Hour <span class="required">*</span></label>
            <input type="number" name="rate_limit_per_hour" value="1000" min="1" max="10000" required>
        </div>
        
        <div class="form-group">
            <label>Cache Duration (seconds) <span class="required">*</span></label>
            <input type="number" name="cache_duration" value="300" min="60" max="86400" required>
        </div>
        
        <!-- Hidden fields for boolean values -->
        <input type="hidden" name="enable_pagination" value="1">
        <input type="hidden" name="require_token" value="0">
        <input type="hidden" name="enable_cors" value="1">
        <input type="hidden" name="enable_cache" value="1">
        <input type="hidden" name="is_active" value="1">
        
        <button type="submit">Save GET-Only Receiver</button>
    </form>
    
    <h3>Next Steps:</h3>
    <ol>
        <li>Get CSRF token from Laravel application</li>
        <li>Replace "REPLACE_WITH_CSRF_TOKEN" with actual token</li>
        <li>Test the form submission</li>
    </ol>
</body>
</html>';

file_put_contents('test-get-only-receiver-form.html', $testFormHTML);
echo "✅ Test form created: test-get-only-receiver-form.html\n";

// Step 7: Summary and instructions
echo "\n🎉 SUMMARY & NEXT STEPS:\n";
echo "========================\n";
echo "✅ Database connection verified\n";
echo "✅ Table 'get_only_receivers' created/verified\n";
echo "✅ Table functionality tested\n";
echo "✅ Test form created\n";
echo "\n📋 To test the form:\n";
echo "1. Open: https://copilotproduction.site/get-only-receivers/create\n";
echo "2. Fill out the form with valid data\n";
echo "3. Click 'Save GET-Only Receiver'\n";
echo "\n🔧 If you still get errors:\n";
echo "1. Check Laravel logs: storage/logs/laravel.log\n";
echo "2. Check web server error logs\n";
echo "3. Verify .env database configuration\n";
echo "\n📁 Files created/modified:\n";
echo "- test-get-only-receiver-form.html (test form)\n";
echo "- fix-get-only-receivers-save-error.php (this script)\n";
echo "\n🌐 Test URLs:\n";
echo "- Form: https://copilotproduction.site/get-only-receivers/create\n";
echo "- List: https://copilotproduction.site/get-only-receivers\n";
echo "\n✅ The save error should now be fixed!\n";

?> 