<?php
// Fix Import Route - Clear Cache and Debug
echo "<h1>🔧 Fix Import Route Issue</h1>";

try {
    // Try to load Laravel
    if (file_exists(__DIR__ . '/vendor/autoload.php')) {
        require_once __DIR__ . '/vendor/autoload.php';
        $app = require_once __DIR__ . '/bootstrap/app.php';
        
        echo "<h2>✅ Laravel Loaded Successfully</h2>";
        
        // Clear cache files manually
        $cacheCleared = [];
        
        // Clear route cache
        if (file_exists(__DIR__ . '/bootstrap/cache/routes.php')) {
            unlink(__DIR__ . '/bootstrap/cache/routes.php');
            $cacheCleared[] = "Route cache file deleted";
        }
        
        // Clear config cache
        if (file_exists(__DIR__ . '/bootstrap/cache/config.php')) {
            unlink(__DIR__ . '/bootstrap/cache/config.php');
            $cacheCleared[] = "Config cache file deleted";
        }
        
        // Clear view cache
        $viewCacheDir = __DIR__ . '/storage/framework/views';
        if (is_dir($viewCacheDir)) {
            $files = glob($viewCacheDir . '/*');
            $count = 0;
            foreach ($files as $file) {
                if (is_file($file)) {
                    unlink($file);
                    $count++;
                }
            }
            if ($count > 0) {
                $cacheCleared[] = "Deleted $count view cache files";
            }
        }
        
        // Clear application cache
        $appCacheDir = __DIR__ . '/storage/framework/cache/data';
        if (is_dir($appCacheDir)) {
            $files = glob($appCacheDir . '/*');
            $count = 0;
            foreach ($files as $file) {
                if (is_file($file)) {
                    unlink($file);
                    $count++;
                }
            }
            if ($count > 0) {
                $cacheCleared[] = "Deleted $count app cache files";
            }
        }
        
        echo "<h2>🧹 Cache Clearing Results:</h2>";
        if (empty($cacheCleared)) {
            echo "<p>✓ No cache files found (already clean)</p>";
        } else {
            foreach ($cacheCleared as $result) {
                echo "<p>✓ $result</p>";
            }
        }
        
        // Try to check routes
        try {
            $kernel = $app->make(Illuminate\Contracts\Http\Kernel::class);
            $request = Illuminate\Http\Request::capture();
            
            // Get routes
            $routes = app('router')->getRoutes();
            
            echo "<h2>🔍 Route Debug:</h2>";
            
            $importFound = false;
            $apiSourceRoutes = [];
            
            foreach ($routes as $route) {
                $uri = $route->uri();
                $methods = implode('|', $route->methods());
                
                if (strpos($uri, 'api-sources') !== false) {
                    $apiSourceRoutes[] = "$methods $uri";
                    
                    if (strpos($uri, 'import') !== false) {
                        echo "<p style='color: green; font-weight: bold;'>✅ IMPORT ROUTE FOUND: $methods $uri</p>";
                        $importFound = true;
                    }
                }
            }
            
            if (!$importFound) {
                echo "<p style='color: red; font-weight: bold;'>❌ IMPORT ROUTE NOT FOUND</p>";
            }
            
            echo "<h3>All API Sources Routes:</h3>";
            foreach ($apiSourceRoutes as $route) {
                echo "<p>$route</p>";
            }
            
        } catch (Exception $e) {
            echo "<h2>❌ Error checking routes:</h2>";
            echo "<p>" . $e->getMessage() . "</p>";
        }
        
    } else {
        echo "<h2>❌ Laravel Not Found</h2>";
        echo "<p>vendor/autoload.php not found</p>";
    }
    
} catch (Exception $e) {
    echo "<h2>❌ Error:</h2>";
    echo "<p>" . $e->getMessage() . "</p>";
}

echo "<hr>";
echo "<h2>📋 Manual Steps if Above Doesn't Work:</h2>";
echo "<ol>";
echo "<li><strong>Check .htaccess file:</strong> Make sure it exists in the public directory</li>";
echo "<li><strong>Check file permissions:</strong> Ensure Laravel files have correct permissions</li>";
echo "<li><strong>Check PHP version:</strong> Laravel 10+ requires PHP 8.1+</li>";
echo "<li><strong>Check web server config:</strong> Make sure mod_rewrite is enabled</li>";
echo "</ol>";

echo "<hr>";
echo "<p><strong>Next Steps:</strong></p>";
echo "<p>1. If import route is found above, try the import function again</p>";
echo "<p>2. If still not working, check browser console for JavaScript errors</p>";
echo "<p>3. Contact hosting support if server configuration issues persist</p>";

echo "<p><a href='/api-sources' style='background: #007cba; color: white; padding: 10px 15px; text-decoration: none; border-radius: 5px;'>← Back to API Sources</a></p>";
?> 