<?php
/**
 * Script untuk memperbaiki masalah Bearer token 401 Unauthorized
 * Jalankan dengan: php fix_bearer_token_issue.php
 */

require_once 'vendor/autoload.php';
$app = require_once 'bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

use App\Models\ApiReceiver;
use App\Models\ApiClient;
use App\Models\ApiToken;
use Illuminate\Support\Str;

echo "=== FIX BEARER TOKEN 401 UNAUTHORIZED ===\n\n";

// Ganti dengan endpoint yang bermasalah
$endpoint = 'cilentapi'; // Ganti dengan endpoint Anda

try {
    // 1. Cek receiver
    echo "1. Checking API Receiver...\n";
    $receiver = ApiReceiver::where('endpoint', $endpoint)->first();
    
    if (!$receiver) {
        echo "❌ Receiver not found for endpoint: {$endpoint}\n";
        exit;
    }
    
    echo "✅ Receiver found: {$receiver->name}\n";
    echo "   Current auth_type: {$receiver->auth_type}\n";
    echo "   Current auth_token: " . ($receiver->auth_token ?: 'NULL') . "\n";
    echo "   API Client ID: " . ($receiver->api_client_id ?: 'NULL') . "\n\n";
    
    // 2. Fix auth_type jika bukan bearer_token
    if ($receiver->auth_type !== 'bearer_token') {
        echo "2. Fixing auth_type...\n";
        $receiver->update(['auth_type' => 'bearer_token']);
        echo "✅ Changed auth_type to 'bearer_token'\n\n";
    } else {
        echo "2. Auth type is already 'bearer_token' ✅\n\n";
    }
    
    // 3. Cek dan fix API Client
    if ($receiver->api_client_id) {
        echo "3. Checking API Client...\n";
        $client = ApiClient::find($receiver->api_client_id);
        
        if (!$client) {
            echo "❌ API Client not found! Creating new one...\n";
            $client = ApiClient::create([
                'name' => $receiver->name . ' Client',
                'client_id' => (string) Str::uuid(),
                'client_secret' => Str::random(48),
                'usertoken' => Str::random(40),
                'passtoken' => Str::random(64),
                'is_active' => true,
                'rate_limit' => 1000
            ]);
            $receiver->update(['api_client_id' => $client->id]);
            echo "✅ Created new API Client: {$client->name}\n";
        } else {
            echo "✅ API Client found: {$client->name}\n";
        }
        
        // 4. Cek dan buat token jika belum ada
        echo "4. Checking API Tokens...\n";
        $existingToken = ApiToken::where('api_client_id', $client->id)
            ->where('is_revoked', false)
            ->first();
            
        if (!$existingToken) {
            echo "❌ No active token found. Creating new token...\n";
            $newToken = 'at_' . Str::random(40);
            
            $apiToken = ApiToken::create([
                'api_client_id' => $client->id,
                'name' => 'Auto-generated Token',
                'token' => $newToken,
                'refresh_token' => 'rt_' . Str::random(40),
                'scope' => '*',
                'is_revoked' => false
            ]);
            
            echo "✅ Created new API Token: {$newToken}\n";
            echo "   Use this token in your Authorization header:\n";
            echo "   Authorization: Bearer {$newToken}\n\n";
        } else {
            echo "✅ Found existing token: {$existingToken->token}\n";
            echo "   Use this token in your Authorization header:\n";
            echo "   Authorization: Bearer {$existingToken->token}\n\n";
        }
        
        // 5. Update receiver auth_token dengan token dari api_tokens
        $activeToken = ApiToken::where('api_client_id', $client->id)
            ->where('is_revoked', false)
            ->first();
            
        if ($activeToken && $receiver->auth_token !== $activeToken->token) {
            echo "5. Syncing receiver auth_token with API token...\n";
            $receiver->update(['auth_token' => $activeToken->token]);
            echo "✅ Updated receiver auth_token\n\n";
        }
        
    } else {
        echo "3. No API Client linked. Creating one...\n";
        
        // Buat API Client baru
        $client = ApiClient::create([
            'name' => $receiver->name . ' Client',
            'client_id' => (string) Str::uuid(),
            'client_secret' => Str::random(48),
            'usertoken' => Str::random(40),
            'passtoken' => Str::random(64),
            'is_active' => true,
            'rate_limit' => 1000
        ]);
        
        // Link receiver ke client
        $receiver->update(['api_client_id' => $client->id]);
        echo "✅ Created and linked API Client: {$client->name}\n";
        
        // Buat token
        $newToken = 'at_' . Str::random(40);
        $apiToken = ApiToken::create([
            'api_client_id' => $client->id,
            'name' => 'Auto-generated Token',
            'token' => $newToken,
            'refresh_token' => 'rt_' . Str::random(40),
            'scope' => '*',
            'is_revoked' => false
        ]);
        
        // Update receiver auth_token
        $receiver->update(['auth_token' => $newToken]);
        
        echo "✅ Created API Token: {$newToken}\n";
        echo "   Use this token in your Authorization header:\n";
        echo "   Authorization: Bearer {$newToken}\n\n";
    }
    
    // 6. Final verification
    echo "6. Final verification...\n";
    $receiver->refresh();
    $client = $receiver->apiClient;
    $token = $receiver->auth_token;
    
    echo "   Receiver auth_type: {$receiver->auth_type}\n";
    echo "   Receiver auth_token: {$token}\n";
    echo "   Client active: " . ($client->is_active ? 'Yes' : 'No') . "\n";
    
    $apiToken = ApiToken::where('token', $token)
        ->where('is_revoked', false)
        ->first();
        
    if ($apiToken) {
        echo "   Token in api_tokens: ✅ Found\n";
        echo "   Token expires: " . ($apiToken->expires_at ?: 'Never') . "\n";
    } else {
        echo "   Token in api_tokens: ❌ Not found\n";
    }
    
    echo "\n=== FIX COMPLETE ===\n";
    echo "Now test your API with:\n";
    echo "Authorization: Bearer {$token}\n";
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "File: " . $e->getFile() . ":" . $e->getLine() . "\n";
}
