<?php
/**
 * EMERGENCY CACHE CLEARING SCRIPT
 * This script manually deletes all Laravel cache files and forces route reloading
 */

echo "<h1>🚨 Emergency Cache Clearing Script</h1>";
echo "<p>This script will manually delete all cache files and force Laravel to reload routes.</p>";

$basePath = __DIR__;
$errors = [];
$success = [];

// 1. Manual file deletion approach
$cacheFiles = [
    'bootstrap/cache/config.php',
    'bootstrap/cache/routes.php',
    'bootstrap/cache/services.php',
    'bootstrap/cache/packages.php',
    'storage/framework/cache/data',
    'storage/framework/views',
    'storage/framework/sessions'
];

echo "<h2>🗑️ Step 1: Manual Cache File Deletion</h2>";
foreach ($cacheFiles as $file) {
    $fullPath = $basePath . '/' . $file;
    
    if (file_exists($fullPath)) {
        if (is_dir($fullPath)) {
            // Delete directory contents
            $files = glob($fullPath . '/*');
            foreach ($files as $f) {
                if (is_file($f)) {
                    unlink($f);
                }
            }
            $success[] = "✅ Cleared directory: $file";
        } else {
            // Delete file
            unlink($fullPath);
            $success[] = "✅ Deleted file: $file";
        }
    } else {
        $success[] = "ℹ️ File not found (OK): $file";
    }
}

// 2. Try Laravel Artisan commands
echo "<h2>⚡ Step 2: Laravel Artisan Commands</h2>";
$artisanCommands = [
    'route:clear',
    'config:clear',
    'cache:clear',
    'view:clear',
    'optimize:clear'
];

// Check if we can run artisan
if (file_exists($basePath . '/artisan')) {
    foreach ($artisanCommands as $command) {
        try {
            $output = shell_exec("cd $basePath && php artisan $command 2>&1");
            if ($output !== null) {
                $success[] = "✅ Artisan $command: " . trim($output);
            } else {
                $errors[] = "❌ Artisan $command failed";
            }
        } catch (Exception $e) {
            $errors[] = "❌ Artisan $command error: " . $e->getMessage();
        }
    }
} else {
    $errors[] = "❌ Artisan file not found";
}

// 3. Force route registration check
echo "<h2>🔍 Step 3: Route Registration Check</h2>";
try {
    // Include Laravel bootstrap
    require_once $basePath . '/vendor/autoload.php';
    $app = require_once $basePath . '/bootstrap/app.php';
    
    // Boot the application
    $app->make(\Illuminate\Contracts\Http\Kernel::class);
    
    // Get all routes
    $routes = $app['router']->getRoutes();
    $importRoutes = [];
    
    foreach ($routes as $route) {
        $uri = $route->uri();
        if (strpos($uri, 'api-sources') !== false && strpos($uri, 'import') !== false) {
            $importRoutes[] = $route->methods()[0] . ' ' . $uri;
        }
        if (strpos($uri, 'import-data') !== false) {
            $importRoutes[] = $route->methods()[0] . ' ' . $uri;
        }
    }
    
    if (count($importRoutes) > 0) {
        $success[] = "✅ Import routes found: " . implode(', ', $importRoutes);
    } else {
        $errors[] = "❌ No import routes found in route collection";
    }
    
} catch (Exception $e) {
    $errors[] = "❌ Route check failed: " . $e->getMessage();
}

// 4. Create direct import endpoint
echo "<h2>🔧 Step 4: Creating Direct Import Endpoint</h2>";
$directImportCode = '<?php
// DIRECT IMPORT ENDPOINT - BYPASS ROUTING
if (isset($_GET["action"]) && $_GET["action"] === "import" && isset($_GET["id"])) {
    $id = (int)$_GET["id"];
    
    try {
        // Include Laravel
        require_once __DIR__ . "/vendor/autoload.php";
        $app = require_once __DIR__ . "/bootstrap/app.php";
        $app->make(\Illuminate\Contracts\Http\Kernel::class);
        
        // Get API Source
        $apiSource = \App\Models\ApiSource::findOrFail($id);
        
        // Make HTTP request
        $response = \Illuminate\Support\Facades\Http::timeout(30)->get($apiSource->url);
        
        if ($response->successful()) {
            $data = $response->json();
            $recordsCount = is_array($data) ? count($data) : 1;
            
            // Log the import
            \App\Models\ApiLog::create([
                "api_source_id" => $apiSource->id,
                "endpoint" => $apiSource->url,
                "method" => "GET",
                "status_code" => $response->status(),
                "response_code" => $response->status(),
                "status" => "success",
                "response_time" => ($response->handlerStats()["total_time"] ?? 0) * 1000,
                "response_body" => is_array($data) ? json_encode($data) : $data,
                "records_imported" => $recordsCount,
                "requested_at" => now()
            ]);
            
            $apiSource->update(["last_fetched_at" => now()]);
            
            header("Content-Type: application/json");
            echo json_encode([
                "success" => true,
                "message" => "Data imported successfully via direct endpoint",
                "records" => $recordsCount
            ]);
            exit;
        } else {
            throw new Exception("API request failed with status: " . $response->status());
        }
        
    } catch (Exception $e) {
        header("Content-Type: application/json");
        echo json_encode([
            "success" => false,
            "message" => "Import failed: " . $e->getMessage()
        ]);
        exit;
    }
}
?>';

try {
    file_put_contents($basePath . '/direct-import.php', $directImportCode);
    $success[] = "✅ Created direct import endpoint: /direct-import.php";
} catch (Exception $e) {
    $errors[] = "❌ Failed to create direct import endpoint: " . $e->getMessage();
}

// 5. Results
echo "<h2>📊 Results</h2>";

if (count($success) > 0) {
    echo "<div style='background: #d4edda; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h3 style='color: #155724;'>✅ Successful Operations:</h3>";
    foreach ($success as $msg) {
        echo "<p style='margin: 5px 0; color: #155724;'>$msg</p>";
    }
    echo "</div>";
}

if (count($errors) > 0) {
    echo "<div style='background: #f8d7da; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h3 style='color: #721c24;'>❌ Errors:</h3>";
    foreach ($errors as $msg) {
        echo "<p style='margin: 5px 0; color: #721c24;'>$msg</p>";
    }
    echo "</div>";
}

echo "<h2>🎯 Next Steps</h2>";
echo "<div style='background: #cce5ff; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
echo "<h3>Option 1: Try Import Again</h3>";
echo "<p>Go back to your API Sources page and try the import function again.</p>";
echo "<p><a href='/api-sources' style='background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>← Back to API Sources</a></p>";

echo "<h3>Option 2: Use Direct Import</h3>";
echo "<p>If the normal import still doesn't work, you can use the direct import endpoint:</p>";
echo "<p><strong>For API Source ID 3:</strong> <a href='/direct-import.php?action=import&id=3' target='_blank'>Direct Import ID 3</a></p>";
echo "<p><strong>For API Source ID 4:</strong> <a href='/direct-import.php?action=import&id=4' target='_blank'>Direct Import ID 4</a></p>";
echo "</div>";

echo "<h2>🔄 Manual Cache Clear</h2>";
echo "<p><a href='/force-clear-cache.php' style='background: #28a745; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>Run This Script Again</a></p>";
?> 