<?php
/**
 * STANDALONE API DATA IMPORT SCRIPT
 * This script bypasses Laravel routing and directly handles API data import
 */

// Set headers for JSON response
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-CSRF-TOKEN');

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Method not allowed. Only POST requests are accepted.'
    ]);
    exit();
}

// Get the API Source ID from URL or POST data
$apiSourceId = null;

// Check if ID is in URL path (e.g., /import-api-data.php/4)
$pathInfo = $_SERVER['PATH_INFO'] ?? '';
if (preg_match('/\/(\d+)$/', $pathInfo, $matches)) {
    $apiSourceId = (int)$matches[1];
}

// Check if ID is in POST data
if (!$apiSourceId && isset($_POST['id'])) {
    $apiSourceId = (int)$_POST['id'];
}

// Check if ID is in JSON body
if (!$apiSourceId) {
    $input = json_decode(file_get_contents('php://input'), true);
    if (isset($input['id'])) {
        $apiSourceId = (int)$input['id'];
    }
}

// Check if ID is in query string
if (!$apiSourceId && isset($_GET['id'])) {
    $apiSourceId = (int)$_GET['id'];
}

if (!$apiSourceId) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'API Source ID is required'
    ]);
    exit();
}

try {
    // Bootstrap Laravel application
    require_once __DIR__ . '/vendor/autoload.php';
    $app = require_once __DIR__ . '/bootstrap/app.php';
    
    // Boot the application
    $kernel = $app->make(Illuminate\Contracts\Http\Kernel::class);
    
    // Get the API Source from database
    $apiSource = \App\Models\ApiSource::find($apiSourceId);
    
    if (!$apiSource) {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'message' => "API Source with ID {$apiSourceId} not found"
        ]);
        exit();
    }
    
    // Make HTTP request to the API
    $startTime = microtime(true);
    $response = \Illuminate\Support\Facades\Http::timeout(30)->get($apiSource->url);
    $endTime = microtime(true);
    $responseTime = ($endTime - $startTime) * 1000; // Convert to milliseconds
    
    if ($response->successful()) {
        $data = $response->json();
        $recordsCount = is_array($data) ? count($data) : 1;
        
        // Log the successful import
        \App\Models\ApiLog::create([
            'api_source_id' => $apiSource->id,
            'endpoint' => $apiSource->url,
            'method' => 'GET',
            'status_code' => $response->status(),
            'response_code' => $response->status(),
            'status' => 'success',
            'response_time' => round($responseTime, 2),
            'response_body' => is_array($data) ? json_encode($data) : $data,
            'records_imported' => $recordsCount,
            'requested_at' => now()
        ]);
        
        // Update last fetched timestamp
        $apiSource->update(['last_fetched_at' => now()]);
        
        // Return success response
        echo json_encode([
            'success' => true,
            'message' => 'Data imported successfully via standalone script',
            'records' => $recordsCount,
            'response_time' => round($responseTime, 2) . 'ms',
            'status_code' => $response->status(),
            'api_source' => [
                'id' => $apiSource->id,
                'name' => $apiSource->name,
                'url' => $apiSource->url
            ]
        ]);
        
    } else {
        throw new Exception('API request failed with status: ' . $response->status());
    }
    
} catch (Exception $e) {
    // Log the error
    try {
        if (isset($apiSource)) {
            \App\Models\ApiLog::create([
                'api_source_id' => $apiSource->id,
                'endpoint' => $apiSource->url ?? 'unknown',
                'method' => 'GET',
                'status_code' => 0,
                'response_code' => 0,
                'status' => 'error',
                'response_time' => 0,
                'records_imported' => 0,
                'error_message' => $e->getMessage(),
                'requested_at' => now()
            ]);
        }
    } catch (Exception $logError) {
        // If logging fails, just continue
    }
    
    // Return error response
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Import failed: ' . $e->getMessage(),
        'error_details' => [
            'file' => $e->getFile(),
            'line' => $e->getLine()
        ]
    ]);
}
?> 