<?php
/**
 * Cache Clearing Script for POST-Only Receivers
 * Use this if you're getting "Route [post-only-receivers.index] not defined" error
 */

// Set time limit
set_time_limit(300);

// Change to Laravel root directory
$laravelRoot = dirname(__DIR__);
chdir($laravelRoot);

echo "<h2>🧹 Clearing Cache for POST-Only Receivers</h2>";
echo "<p><strong>Laravel Root:</strong> " . $laravelRoot . "</p>";
echo "<hr>";

$commands = [
    'route:clear' => 'Route Cache',
    'config:clear' => 'Config Cache',
    'view:clear' => 'View Cache',
    'cache:clear' => 'Application Cache',
    'optimize:clear' => 'Optimized Files'
];

$results = [];

foreach ($commands as $command => $description) {
    echo "<p>🔄 Clearing {$description}...</p>";
    
    try {
        $output = [];
        $returnCode = 0;
        
        // Execute artisan command
        exec("php artisan {$command} 2>&1", $output, $returnCode);
        
        if ($returnCode === 0) {
            echo "<p style='color: green;'>✅ {$description} cleared successfully</p>";
            $results[$command] = 'success';
        } else {
            echo "<p style='color: red;'>❌ Failed to clear {$description}</p>";
            echo "<pre>" . implode("\n", $output) . "</pre>";
            $results[$command] = 'failed';
        }
        
    } catch (Exception $e) {
        echo "<p style='color: red;'>❌ Error clearing {$description}: " . $e->getMessage() . "</p>";
        $results[$command] = 'error';
    }
    
    echo "<br>";
}

// Check if routes are now available
echo "<h3>🔍 Checking POST-Only Receiver Routes</h3>";

try {
    $output = [];
    exec("php artisan route:list --name=post-only 2>&1", $output);
    
    if (!empty($output)) {
        echo "<p style='color: green;'>✅ POST-Only Receiver routes found:</p>";
        echo "<pre style='background: #f8f9fa; padding: 10px; border-radius: 5px;'>";
        echo implode("\n", $output);
        echo "</pre>";
    } else {
        echo "<p style='color: orange;'>⚠️ No POST-Only Receiver routes found. This might be normal if they're not cached yet.</p>";
    }
    
} catch (Exception $e) {
    echo "<p style='color: red;'>❌ Error checking routes: " . $e->getMessage() . "</p>";
}

// Regenerate autoload
echo "<h3>🔄 Regenerating Autoload Files</h3>";

try {
    $output = [];
    exec("composer dump-autoload --optimize 2>&1", $output);
    echo "<p style='color: green;'>✅ Autoload files regenerated</p>";
    echo "<pre style='background: #f8f9fa; padding: 10px; border-radius: 5px;'>";
    echo implode("\n", array_slice($output, -5)); // Show last 5 lines
    echo "</pre>";
} catch (Exception $e) {
    echo "<p style='color: red;'>❌ Error regenerating autoload: " . $e->getMessage() . "</p>";
}

// Summary
echo "<hr>";
echo "<h3>📊 Summary</h3>";

$successCount = count(array_filter($results, function($result) { return $result === 'success'; }));
$totalCount = count($results);

if ($successCount === $totalCount) {
    echo "<p style='color: green; font-weight: bold;'>✅ All cache clearing operations completed successfully!</p>";
    echo "<p>You can now try accessing the POST-Only Receivers page again.</p>";
} else {
    echo "<p style='color: orange; font-weight: bold;'>⚠️ Some operations failed. Check the details above.</p>";
}

echo "<h3>🔗 Quick Links</h3>";
echo "<ul>";
echo "<li><a href='/post-only-receivers' target='_blank'>📧 POST-Only Receivers</a></li>";
echo "<li><a href='/api-clients' target='_blank'>👥 API Clients</a></li>";
echo "<li><a href='/api-tokens' target='_blank'>🔑 API Tokens</a></li>";
echo "<li><a href='/' target='_blank'>🏠 Dashboard</a></li>";
echo "</ul>";

echo "<h3>🛠️ Manual Steps (if still having issues)</h3>";
echo "<ol>";
echo "<li>SSH to your server</li>";
echo "<li>Navigate to your Laravel root directory</li>";
echo "<li>Run: <code>php artisan route:clear</code></li>";
echo "<li>Run: <code>php artisan config:clear</code></li>";
echo "<li>Run: <code>php artisan view:clear</code></li>";
echo "<li>Run: <code>composer dump-autoload</code></li>";
echo "<li>Check file permissions on storage and bootstrap/cache directories</li>";
echo "</ol>";

echo "<hr>";
echo "<p><small>Generated at: " . date('Y-m-d H:i:s') . "</small></p>";
echo "<p><small>Script: clear-cache-post-receivers.php</small></p>";
?> 