<?php

// Simple script to create test data for API Source 4
// Access via: https://copilotproduction.site/create-test-data.php

require_once '../vendor/autoload.php';

// Bootstrap Laravel
$app = require_once '../bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

try {
    echo "<h2>🔍 Creating Test Data for API Source 4</h2>";
    
    // Find API source with ID 4
    $apiSource = App\Models\ApiSource::find(4);
    
    if (!$apiSource) {
        echo "<p style='color: red;'>❌ API Source with ID 4 not found.</p>";
        echo "<h3>📋 Available API Sources:</h3>";
        $sources = App\Models\ApiSource::all();
        foreach ($sources as $source) {
            echo "<p>   - ID: {$source->id}, Name: {$source->name}, Table: {$source->table_name}</p>";
        }
        exit;
    }
    
    echo "<p>✅ Found API Source ID 4: <strong>{$apiSource->name}</strong></p>";
    echo "<p>   - URL: {$apiSource->url}</p>";
    echo "<p>   - Table: {$apiSource->table_name}</p>";
    echo "<p>   - Status: {$apiSource->status}</p>";
    
    // Check existing logs
    $totalLogs = $apiSource->apiLogs()->count();
    $successLogs = $apiSource->apiLogs()->where('status', 'success')->count();
    
    echo "<h3>📊 Current Status:</h3>";
    echo "<p>   - Total Logs: {$totalLogs}</p>";
    echo "<p>   - Success Logs: {$successLogs}</p>";
    
    if ($successLogs == 0) {
        echo "<p style='color: orange;'>⚠️ No successful import logs found. Creating test data...</p>";
        
        // Create sample data based on the table name 'get2'
        $sampleData = [
            [
                'id' => 1,
                'name' => 'Ahmad Dhani',
                'email' => 'ahmad@example.com',
                'phone' => '+62812345678',
                'address' => 'Jakarta, Indonesia',
                'status' => 'active',
                'created_at' => '2025-01-15 10:30:00',
                'updated_at' => '2025-01-15 10:30:00'
            ],
            [
                'id' => 2,
                'name' => 'Siti Nurhaliza',
                'email' => 'siti@example.com',
                'phone' => '+62812345679',
                'address' => 'Bandung, Indonesia',
                'status' => 'active',
                'created_at' => '2025-01-15 11:45:00',
                'updated_at' => '2025-01-15 11:45:00'
            ],
            [
                'id' => 3,
                'name' => 'Budi Santoso',
                'email' => 'budi@example.com',
                'phone' => '+62812345680',
                'address' => 'Surabaya, Indonesia',
                'status' => 'inactive',
                'created_at' => '2025-01-15 14:20:00',
                'updated_at' => '2025-01-15 14:20:00'
            ],
            [
                'id' => 4,
                'name' => 'Dewi Sartika',
                'email' => 'dewi@example.com',
                'phone' => '+62812345681',
                'address' => 'Yogyakarta, Indonesia',
                'status' => 'active',
                'created_at' => '2025-01-15 16:10:00',
                'updated_at' => '2025-01-15 16:10:00'
            ],
            [
                'id' => 5,
                'name' => 'Rudi Hartono',
                'email' => 'rudi@example.com',
                'phone' => '+62812345682',
                'address' => 'Medan, Indonesia',
                'status' => 'pending',
                'created_at' => '2025-01-15 18:30:00',
                'updated_at' => '2025-01-15 18:30:00'
            ],
            [
                'id' => 6,
                'name' => 'Maya Sari',
                'email' => 'maya@example.com',
                'phone' => '+62812345683',
                'address' => 'Bali, Indonesia',
                'status' => 'active',
                'created_at' => '2025-01-16 09:15:00',
                'updated_at' => '2025-01-16 09:15:00'
            ],
            [
                'id' => 7,
                'name' => 'Joko Widodo',
                'email' => 'joko@example.com',
                'phone' => '+62812345684',
                'address' => 'Solo, Indonesia',
                'status' => 'active',
                'created_at' => '2025-01-16 12:30:00',
                'updated_at' => '2025-01-16 12:30:00'
            ]
        ];
        
        // Create multiple import logs to simulate different import sessions
        for ($i = 1; $i <= 3; $i++) {
            $importDate = now()->subDays($i);
            $recordsCount = count($sampleData);
            
            $log = App\Models\ApiLog::create([
                'api_source_id' => $apiSource->id,
                'endpoint' => $apiSource->url,
                'method' => 'GET',
                'status_code' => 200,
                'response_code' => 200,
                'status' => 'success',
                'response_time' => rand(100, 500),
                'response_body' => json_encode($sampleData),
                'records_imported' => $recordsCount,
                'requested_at' => $importDate,
                'created_at' => $importDate,
                'updated_at' => $importDate
            ]);
            
            echo "<p>✅ Created import log #{$i} with ID: {$log->id}</p>";
        }
        
        // Update last_fetched_at
        $apiSource->update(['last_fetched_at' => now()]);
        
        echo "<h3 style='color: green;'>🎉 Test Data Created Successfully!</h3>";
        echo "<p>📋 Summary:</p>";
        echo "<ul>";
        echo "<li>Created 3 successful import logs</li>";
        echo "<li>Each log contains " . count($sampleData) . " sample records</li>";
        echo "<li>Total records: " . (count($sampleData) * 3) . "</li>";
        echo "<li>Updated last_fetched_at timestamp</li>";
        echo "</ul>";
        
    } else {
        echo "<p style='color: green;'>✅ Found {$successLogs} successful import logs already.</p>";
        
        $latestLog = $apiSource->apiLogs()->where('status', 'success')->latest()->first();
        if ($latestLog) {
            echo "<p>   - Latest Import: {$latestLog->created_at->format('Y-m-d H:i:s')}</p>";
            echo "<p>   - Records: {$latestLog->records_imported}</p>";
            echo "<p>   - Response Time: {$latestLog->response_time}ms</p>";
        }
    }
    
    echo "<hr>";
    echo "<h3>🌐 Next Steps:</h3>";
    echo "<p>1. <a href='/api-sources/4/data' target='_blank'>Visit the Data Page</a></p>";
    echo "<p>2. The page should now show real imported data instead of sample data</p>";
    echo "<p>3. If still showing old data, try refreshing the page (Ctrl+F5)</p>";
    
    echo "<hr>";
    echo "<p><a href='/api-sources'>← Back to API Sources</a></p>";
    
} catch (Exception $e) {
    echo "<h3 style='color: red;'>❌ ERROR:</h3>";
    echo "<p>" . $e->getMessage() . "</p>";
    echo "<h4>📝 Stack trace:</h4>";
    echo "<pre>" . $e->getTraceAsString() . "</pre>";
} 