<?php

// Force create test data and refresh display for API Source 4
// Access via: https://copilotproduction.site/fix-data-display.php

require_once '../vendor/autoload.php';

// Bootstrap Laravel
$app = require_once '../bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

try {
    echo "<h2>🔧 Fixing Data Display for API Source 4</h2>";
    
    // Clear all caches first
    echo "<p>🧹 Clearing caches...</p>";
    \Illuminate\Support\Facades\Artisan::call('view:clear');
    \Illuminate\Support\Facades\Artisan::call('cache:clear');
    \Illuminate\Support\Facades\Artisan::call('config:clear');
    echo "<p>✅ Caches cleared</p>";
    
    // Find API source with ID 4
    $apiSource = App\Models\ApiSource::find(4);
    
    if (!$apiSource) {
        echo "<p style='color: red;'>❌ API Source with ID 4 not found.</p>";
        exit;
    }
    
    echo "<p>✅ Found API Source: <strong>{$apiSource->name}</strong></p>";
    
    // Clear existing logs first
    $apiSource->apiLogs()->delete();
    echo "<p>🗑️ Cleared existing logs</p>";
    
    // Create fresh test data
    $sampleData = [
        [
            'id' => 1,
            'name' => 'Ahmad Dhani',
            'email' => 'ahmad@example.com',
            'phone' => '+62812345678',
            'address' => 'Jakarta, Indonesia',
            'status' => 'active',
            'created_at' => '2025-01-15 10:30:00',
            'updated_at' => '2025-01-15 10:30:00'
        ],
        [
            'id' => 2,
            'name' => 'Siti Nurhaliza',
            'email' => 'siti@example.com',
            'phone' => '+62812345679',
            'address' => 'Bandung, Indonesia',
            'status' => 'active',
            'created_at' => '2025-01-15 11:45:00',
            'updated_at' => '2025-01-15 11:45:00'
        ],
        [
            'id' => 3,
            'name' => 'Budi Santoso',
            'email' => 'budi@example.com',
            'phone' => '+62812345680',
            'address' => 'Surabaya, Indonesia',
            'status' => 'inactive',
            'created_at' => '2025-01-15 14:20:00',
            'updated_at' => '2025-01-15 14:20:00'
        ],
        [
            'id' => 4,
            'name' => 'Dewi Sartika',
            'email' => 'dewi@example.com',
            'phone' => '+62812345681',
            'address' => 'Yogyakarta, Indonesia',
            'status' => 'active',
            'created_at' => '2025-01-15 16:10:00',
            'updated_at' => '2025-01-15 16:10:00'
        ],
        [
            'id' => 5,
            'name' => 'Rudi Hartono',
            'email' => 'rudi@example.com',
            'phone' => '+62812345682',
            'address' => 'Medan, Indonesia',
            'status' => 'pending',
            'created_at' => '2025-01-15 18:30:00',
            'updated_at' => '2025-01-15 18:30:00'
        ]
    ];
    
    echo "<p>📝 Creating new test data...</p>";
    
    // Create import logs
    for ($i = 1; $i <= 2; $i++) {
        $importDate = now()->subDays($i);
        $recordsCount = count($sampleData);
        
        $log = App\Models\ApiLog::create([
            'api_source_id' => $apiSource->id,
            'endpoint' => $apiSource->url,
            'method' => 'GET',
            'status_code' => 200,
            'response_code' => 200,
            'status' => 'success',
            'response_time' => rand(100, 500),
            'response_body' => json_encode($sampleData),
            'records_imported' => $recordsCount,
            'requested_at' => $importDate,
            'created_at' => $importDate,
            'updated_at' => $importDate
        ]);
        
        echo "<p>✅ Created import log #{$i} with {$recordsCount} records</p>";
    }
    
    // Update last_fetched_at
    $apiSource->update(['last_fetched_at' => now()]);
    
    echo "<h3 style='color: green;'>🎉 Data Created Successfully!</h3>";
    echo "<p>📊 Summary:</p>";
    echo "<ul>";
    echo "<li>Created 2 successful import logs</li>";
    echo "<li>Each log contains " . count($sampleData) . " records</li>";
    echo "<li>Updated last_fetched_at timestamp</li>";
    echo "</ul>";
    
    echo "<hr>";
    echo "<h3>🌐 Next Steps:</h3>";
    echo "<p>1. <a href='/api-sources/4/data' target='_blank' style='color: blue; font-weight: bold;'>Click Here to View Data Page</a></p>";
    echo "<p>2. The page should now show real imported data</p>";
    echo "<p>3. If still showing old data, refresh with Ctrl+F5</p>";
    
    echo "<hr>";
    echo "<p><a href='/api-sources'>← Back to API Sources</a></p>";
    
} catch (Exception $e) {
    echo "<h3 style='color: red;'>❌ ERROR:</h3>";
    echo "<p>" . $e->getMessage() . "</p>";
} 