<?php
/**
 * Route Fix Script - Specifically for post-only-receivers routes
 * Diagnoses and fixes route registration issues
 */

// Set time limit
set_time_limit(300);

echo "<h1>🛣️ Route Fix for post-only-receivers</h1>";
echo "<p><strong>Time:</strong> " . date('Y-m-d H:i:s') . "</p>";
echo "<hr>";

// Check Laravel root
$laravelRoot = dirname(__DIR__);
echo "<p><strong>Laravel Root:</strong> {$laravelRoot}</p>";

// Step 1: Check if .env exists
echo "<h2>📋 Step 1: Environment Check</h2>";
$envExists = file_exists($laravelRoot . '/.env');
echo "<p><strong>.env file:</strong> " . ($envExists ? '✅ Exists' : '❌ Missing') . "</p>";

if (!$envExists) {
    echo "<div style='background: #fff3cd; color: #856404; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
    echo "<h3>⚠️ .env File Missing</h3>";
    echo "<p>Creating .env file first...</p>";
    
    $envContent = 'APP_NAME="API Management System"
APP_ENV=production
APP_KEY=base64:' . base64_encode(random_bytes(32)) . '
APP_DEBUG=false
APP_URL=https://copilotproduction.site

LOG_CHANNEL=stack
LOG_LEVEL=debug

DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=copg1493_api
DB_USERNAME=copg1493_api
DB_PASSWORD=P7H_s;Aco25Vv??J

BROADCAST_DRIVER=log
CACHE_DRIVER=file
FILESYSTEM_DISK=local
QUEUE_CONNECTION=sync
SESSION_DRIVER=file
SESSION_LIFETIME=120
';

    if (file_put_contents($laravelRoot . '/.env', $envContent)) {
        echo "<p style='color: green;'>✅ .env file created!</p>";
    } else {
        echo "<p style='color: red;'>❌ Failed to create .env file</p>";
    }
    echo "</div>";
}

// Step 2: Clear cache files manually
echo "<h2>🧹 Step 2: Manual Cache Clearing</h2>";

$cacheFiles = [
    $laravelRoot . '/bootstrap/cache/config.php',
    $laravelRoot . '/bootstrap/cache/routes.php',
    $laravelRoot . '/bootstrap/cache/services.php',
    $laravelRoot . '/bootstrap/cache/packages.php'
];

$cleared = 0;
foreach ($cacheFiles as $file) {
    if (file_exists($file)) {
        if (unlink($file)) {
            $cleared++;
            echo "<p style='color: green;'>✅ Deleted " . basename($file) . "</p>";
        } else {
            echo "<p style='color: red;'>❌ Failed to delete " . basename($file) . "</p>";
        }
    } else {
        echo "<p style='color: gray;'>⚪ " . basename($file) . " (not cached)</p>";
    }
}

echo "<p><strong>Cache files cleared:</strong> {$cleared}</p>";

// Step 3: Check if required files exist
echo "<h2>📁 Step 3: File Verification</h2>";

$requiredFiles = [
    '/routes/web.php' => 'Routes definition file',
    '/app/Http/Controllers/PostOnlyReceiverController.php' => 'Controller file',
    '/app/Models/PostOnlyReceiver.php' => 'Model file',
    '/app/Models/PostOnlyReceiveLog.php' => 'Log model file'
];

$allFilesExist = true;
foreach ($requiredFiles as $file => $description) {
    if (file_exists($laravelRoot . $file)) {
        echo "<p style='color: green;'>✅ {$description}: " . basename($file) . "</p>";
    } else {
        echo "<p style='color: red;'>❌ {$description}: " . basename($file) . " MISSING</p>";
        $allFilesExist = false;
    }
}

// Step 4: Check routes/web.php content
echo "<h2>🔍 Step 4: Routes File Analysis</h2>";

if (file_exists($laravelRoot . '/routes/web.php')) {
    $webRoutes = file_get_contents($laravelRoot . '/routes/web.php');
    
    // Check for use statement
    $hasUseStatement = strpos($webRoutes, 'use App\Http\Controllers\PostOnlyReceiverController;') !== false;
    echo "<p><strong>PostOnlyReceiverController import:</strong> " . ($hasUseStatement ? '✅ Found' : '❌ Missing') . "</p>";
    
    // Check for resource route
    $hasResourceRoute = strpos($webRoutes, "Route::resource('post-only-receivers', PostOnlyReceiverController::class);") !== false;
    echo "<p><strong>Resource route definition:</strong> " . ($hasResourceRoute ? '✅ Found' : '❌ Missing') . "</p>";
    
    // Check for additional routes
    $hasLogsRoute = strpos($webRoutes, 'post-only-receivers.logs') !== false;
    echo "<p><strong>Additional routes (logs/stats):</strong> " . ($hasLogsRoute ? '✅ Found' : '❌ Missing') . "</p>";
    
    if (!$hasUseStatement || !$hasResourceRoute) {
        echo "<div style='background: #ffe6e6; color: #721c24; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
        echo "<h3>❌ Routes Configuration Issue</h3>";
        echo "<p>The routes are not properly configured in web.php</p>";
        echo "</div>";
    }
} else {
    echo "<p style='color: red;'>❌ routes/web.php file not found!</p>";
    $allFilesExist = false;
}

// Step 5: Test database connection
echo "<h2>🔌 Step 5: Database Connection Test</h2>";

try {
    $pdo = new PDO(
        'mysql:host=127.0.0.1;dbname=copg1493_api',
        'copg1493_api',
        'P7H_s;Aco25Vv??J'
    );
    echo "<p style='color: green;'>✅ Database connection successful!</p>";
    
    // Check if tables exist
    $stmt = $pdo->query("SHOW TABLES LIKE 'post_only_receivers'");
    $tableExists = $stmt->rowCount() > 0;
    echo "<p><strong>post_only_receivers table:</strong> " . ($tableExists ? '✅ Exists' : '❌ Missing') . "</p>";
    
    if (!$tableExists) {
        echo "<p style='color: orange;'>⚠️ Table missing - this might cause model loading issues</p>";
    }
    
} catch (PDOException $e) {
    echo "<p style='color: red;'>❌ Database connection failed: " . $e->getMessage() . "</p>";
}

// Step 6: Test Laravel bootstrap
echo "<h2>🚀 Step 6: Laravel Bootstrap Test</h2>";

try {
    // Check if vendor/autoload.php exists
    if (file_exists($laravelRoot . '/vendor/autoload.php')) {
        echo "<p style='color: green;'>✅ Composer autoload found</p>";
        
        // Try to check if Laravel can load without executing artisan
        if (file_exists($laravelRoot . '/bootstrap/app.php')) {
            echo "<p style='color: green;'>✅ Laravel bootstrap file found</p>";
        } else {
            echo "<p style='color: red;'>❌ Laravel bootstrap file missing</p>";
        }
    } else {
        echo "<p style='color: red;'>❌ Composer autoload missing - run 'composer install'</p>";
    }
} catch (Exception $e) {
    echo "<p style='color: red;'>❌ Bootstrap test failed: " . $e->getMessage() . "</p>";
}

// Step 7: Create a simple route test
echo "<h2>🧪 Step 7: Creating Route Test</h2>";

$routeTestContent = '<?php
// Simple route test for post-only-receivers
header("Content-Type: application/json");

// Test if we can access Laravel without full bootstrap
$response = [
    "timestamp" => date("Y-m-d H:i:s"),
    "test" => "post-only-receivers-routes",
    "status" => "testing"
];

// Check if .env exists
if (!file_exists(__DIR__ . "/../.env")) {
    $response["error"] = ".env file missing";
    $response["status"] = "failed";
    echo json_encode($response, JSON_PRETTY_PRINT);
    exit;
}

// Try to load Laravel minimally
try {
    require_once __DIR__ . "/../vendor/autoload.php";
    
    // Test if we can create the route manually
    $response["laravel_autoload"] = "success";
    
    // Check if controller exists
    if (class_exists("App\\Http\\Controllers\\PostOnlyReceiverController")) {
        $response["controller_exists"] = "yes";
    } else {
        $response["controller_exists"] = "no";
    }
    
    // Check if model exists
    if (class_exists("App\\Models\\PostOnlyReceiver")) {
        $response["model_exists"] = "yes";
    } else {
        $response["model_exists"] = "no";
    }
    
    $response["status"] = "success";
    
} catch (Exception $e) {
    $response["error"] = $e->getMessage();
    $response["status"] = "failed";
}

echo json_encode($response, JSON_PRETTY_PRINT);
?>';

if (file_put_contents($laravelRoot . '/public/test-post-routes.php', $routeTestContent)) {
    echo "<p style='color: green;'>✅ Route test created: <a href='/test-post-routes.php' target='_blank'>Test Routes</a></p>";
} else {
    echo "<p style='color: red;'>❌ Failed to create route test</p>";
}

// Summary and recommendations
echo "<hr>";
echo "<h2>📊 Route Fix Summary</h2>";

if ($allFilesExist && $envExists) {
    echo "<div style='background: #d4edda; color: #155724; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
    echo "<h3>✅ Route Fix Complete</h3>";
    echo "<p>All required files exist and routes should be working now.</p>";
    echo "</div>";
} else {
    echo "<div style='background: #fff3cd; color: #856404; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
    echo "<h3>⚠️ Issues Found</h3>";
    echo "<p>Some files are missing or routes are not properly configured.</p>";
    echo "</div>";
}

echo "<h3>🔗 Test Links</h3>";
echo "<ul>";
echo "<li><a href='/' target='_blank' style='color: #007bff;'>🏠 Main Site</a></li>";
echo "<li><a href='/test-post-routes.php' target='_blank' style='color: #007bff;'>🧪 Route Test</a></li>";
echo "<li><a href='/post-only-receivers' target='_blank' style='color: #007bff;'>📧 POST-Only Receivers (should work now)</a></li>";
echo "</ul>";

echo "<h3>🛠️ Manual Steps if Still Failing</h3>";
echo "<div style='background: #f8f9fa; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
echo "<p>If routes still don't work, try these steps:</p>";
echo "<ol>";
echo "<li>Run: <a href='/shared-hosting-fix.php'>Shared Hosting Fix</a> for complete setup</li>";
echo "<li>Check file permissions on bootstrap/cache directory</li>";
echo "<li>Verify database tables exist</li>";
echo "<li>Contact hosting provider about Laravel requirements</li>";
echo "</ol>";
echo "</div>";

echo "<hr>";
echo "<p><small>Route fix completed at: " . date('Y-m-d H:i:s') . "</small></p>";
echo "<p><small>Script: route-fix.php</small></p>";
?> 