import './bootstrap';

// Add jQuery easing functions to prevent errors
if (typeof jQuery !== 'undefined') {
    jQuery.extend(jQuery.easing, {
        easeInOutCubic: function (x, t, b, c, d) {
            if ((t/=d/2) < 1) return c/2*t*t*t + b;
            return c/2*((t-=2)*t*t + 2) + b;
        },
        easeOutCubic: function (x, t, b, c, d) {
            return c*((t=t/d-1)*t*t + 1) + b;
        },
        easeInCubic: function (x, t, b, c, d) {
            return c*(t/=d)*t*t + b;
        }
    });
}

// Global Toast Notification System
window.Toast = {
    show: function(message, type = 'info', duration = 5000) {
        const container = this.getContainer();
        const toast = document.createElement('div');
        toast.className = `toast-notification ${type} show`;
        
        toast.innerHTML = `
            <div class="toast-content">
                <i class="fas ${this.getIcon(type)} toast-icon"></i>
                <span class="toast-message">${message}</span>
                <button class="toast-close" onclick="this.parentElement.parentElement.remove()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="toast-progress"></div>
        `;
        
        container.appendChild(toast);
        
        // Auto remove after duration
        setTimeout(() => {
            if (toast.parentElement) {
                toast.classList.remove('show');
                setTimeout(() => toast.remove(), 300);
            }
        }, duration);
        
        return toast;
    },
    
    success: function(message, duration = 5000) {
        return this.show(message, 'success', duration);
    },
    
    error: function(message, duration = 7000) {
        return this.show(message, 'error', duration);
    },
    
    warning: function(message, duration = 6000) {
        return this.show(message, 'warning', duration);
    },
    
    info: function(message, duration = 5000) {
        return this.show(message, 'info', duration);
    },
    
    getContainer: function() {
        let container = document.querySelector('.toast-container');
        if (!container) {
            container = document.createElement('div');
            container.className = 'toast-container';
            document.body.appendChild(container);
        }
        return container;
    },
    
    getIcon: function(type) {
        const icons = {
            success: 'fa-check-circle',
            error: 'fa-exclamation-circle',
            warning: 'fa-exclamation-triangle',
            info: 'fa-info-circle'
        };
        return icons[type] || icons.info;
    }
};

// Enhanced DOM Content Loaded
document.addEventListener('DOMContentLoaded', function() {
    console.log('🚀 API Management System - Enhanced UI Loaded');
    
    try {
        // Initialize all enhanced features
        initializeEnhancedUI();
        initializeSidebar();
        initializeAnimations();
        initializeTooltips();
        initializeModals();
        initializeFormEnhancements();
        initializeLiveUpdates();
    } catch (error) {
        console.error('❌ Error during initialization:', error);
    }
});

// Enhanced UI Initialization
function initializeEnhancedUI() {
    // Add loading states to buttons (excluding delete buttons)
    document.querySelectorAll('.btn[type="submit"]').forEach(btn => {
        // Skip delete buttons to avoid interference with confirmation dialogs
        if (!btn.closest('form[action*="destroy"]') && !btn.closest('form[action*="delete"]')) {
            btn.addEventListener('click', function(e) {
                // Allow opt-out for specific buttons
                if (this.hasAttribute('data-no-loading')) {
                    return;
                }
                const form = this.closest('form');
                // If form is invalid, let native HTML5 validation run and DO NOT disable the button
                if (form && !form.checkValidity()) {
                    return;
                }
                if (!this.disabled) {
                    showButtonLoading(this);
                }
            });
        }
    });

    // Add hover effects to cards
    document.querySelectorAll('.modern-card, .stats-card').forEach(card => {
        addCardHoverEffects(card);
    });

    // Add smooth scroll behavior
    document.documentElement.style.scrollBehavior = 'smooth';

    // Initialize ripple effects
    initializeRippleEffects();
}

// Enhanced Sidebar Functionality
function initializeSidebar() {
    const sidebar = document.getElementById('sidebar');
    const sidebarToggle = document.getElementById('sidebar-toggle');
    const mainContent = document.querySelector('.main-content');

    if (sidebarToggle && sidebar) {
        sidebarToggle.addEventListener('click', function() {
            sidebar.classList.toggle('show');
            
            // Add backdrop for mobile
            if (window.innerWidth <= 768) {
                toggleBackdrop();
            }
        });
    }

    // Auto-hide sidebar on mobile when clicking outside
    document.addEventListener('click', function(e) {
        if (window.innerWidth <= 768) {
            if (sidebar && sidebar.classList.contains('show') && 
                !sidebar.contains(e.target) && 
                !sidebarToggle.contains(e.target)) {
                sidebar.classList.remove('show');
                removeBackdrop();
            }
        }
    });

    // Update active nav links
    updateActiveNavLinks();
}

// Enhanced Animations
function initializeAnimations() {
    // Intersection Observer for scroll animations
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };

    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('animate-in');
            }
        });
    }, observerOptions);

    // Observe elements for animation
    document.querySelectorAll('.fade-in-up, .slide-in-left, .slide-in-right, .modern-card, .stats-card').forEach(el => {
        observer.observe(el);
    });

    // Add stagger animation to stats cards
    const statsCards = document.querySelectorAll('.stats-card');
    statsCards.forEach((card, index) => {
        card.style.animationDelay = `${index * 0.1}s`;
    });
}

// Enhanced Tooltips
function initializeTooltips() {
    // Initialize Bootstrap tooltips if available
    if (typeof bootstrap !== 'undefined') {
        const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
        tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
    }

    // Custom tooltip for elements without Bootstrap
    document.querySelectorAll('[title]').forEach(el => {
        if (!el.hasAttribute('data-bs-toggle')) {
            addCustomTooltip(el);
        }
    });
}

// Enhanced Modal Functionality
function initializeModals() {
    // Add loading states to modal forms
    document.querySelectorAll('.modal form').forEach(form => {
        form.addEventListener('submit', function() {
            const submitBtn = form.querySelector('button[type="submit"]');
            if (submitBtn) {
                showButtonLoading(submitBtn);
            }
        });
    });

    // Auto-focus first input in modals
    document.querySelectorAll('.modal').forEach(modal => {
        modal.addEventListener('shown.bs.modal', function() {
            const firstInput = modal.querySelector('input, select, textarea');
            if (firstInput) {
                firstInput.focus();
            }
        });
    });
}

// Enhanced Form Functionality
function initializeFormEnhancements() {
    // Add floating labels effect
    document.querySelectorAll('.form-control').forEach(input => {
        addFloatingLabelEffect(input);
    });

    // Add form validation feedback
    document.querySelectorAll('form').forEach(form => {
        form.addEventListener('submit', function(e) {
            addFormValidationFeedback(form);
        });
        // If browser blocks submission due to invalid inputs, ensure submit button returns to normal state
        form.addEventListener('invalid', function() {
            const submitBtn = form.querySelector('button[type="submit"]');
            if (submitBtn) {
                hideButtonLoading(submitBtn);
            }
        }, true);
    });

    // Add auto-resize for textareas
    document.querySelectorAll('textarea').forEach(textarea => {
        addAutoResize(textarea);
    });
}

// Live Updates Functionality
function initializeLiveUpdates() {
    // Update timestamps
    updateTimestamps();
    setInterval(updateTimestamps, 60000); // Every minute

    // Update current time in sidebar
    updateCurrentTime();
    setInterval(updateCurrentTime, 1000); // Every second

    // Update system status (if applicable)
    if (document.getElementById('api-count')) {
        updateSystemStatus();
        setInterval(updateSystemStatus, 30000); // Every 30 seconds
    }
}

// Helper Functions

function showButtonLoading(button) {
    const originalText = button.innerHTML;
    const loadingText = button.getAttribute('data-loading-text') || 'Loading...';
    
    button.innerHTML = `
        <span class="spinner-border spinner-border-sm me-2" role="status" aria-hidden="true"></span>
        ${loadingText}
    `;
    button.disabled = true;
    
    // Store original text to restore later
    button.setAttribute('data-original-text', originalText);
}

function hideButtonLoading(button) {
    const originalText = button.getAttribute('data-original-text');
    if (originalText) {
        button.innerHTML = originalText;
        button.disabled = false;
        button.removeAttribute('data-original-text');
    }
}

function addCardHoverEffects(card) {
    card.addEventListener('mouseenter', function() {
        this.style.transform = 'translateY(-8px) scale(1.02)';
    });

    card.addEventListener('mouseleave', function() {
        this.style.transform = 'translateY(0) scale(1)';
    });
}

function toggleBackdrop() {
    let backdrop = document.querySelector('.sidebar-backdrop');
    if (!backdrop) {
        backdrop = document.createElement('div');
        backdrop.className = 'sidebar-backdrop';
        backdrop.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 999;
            opacity: 0;
            transition: opacity 0.3s ease;
        `;
        document.body.appendChild(backdrop);
        
        // Trigger opacity animation
        setTimeout(() => backdrop.style.opacity = '1', 10);
        
        backdrop.addEventListener('click', function() {
            document.getElementById('sidebar').classList.remove('show');
            removeBackdrop();
        });
    }
}

function removeBackdrop() {
    const backdrop = document.querySelector('.sidebar-backdrop');
    if (backdrop) {
        backdrop.style.opacity = '0';
        setTimeout(() => backdrop.remove(), 300);
    }
}

function updateActiveNavLinks() {
    const currentPath = window.location.pathname;
    const navLinks = document.querySelectorAll('.nav-link');
    
    navLinks.forEach(link => {
        const href = link.getAttribute('href');
        if (href && currentPath.includes(href.replace(window.location.origin, ''))) {
            link.classList.add('active');
        } else {
            link.classList.remove('active');
        }
    });
}

function initializeRippleEffects() {
    document.querySelectorAll('.btn, .nav-link, .action-btn').forEach(element => {
        if (element.hasAttribute('data-no-ripple')) {
            return;
        }
        element.addEventListener('click', function(e) {
            const ripple = document.createElement('span');
            const rect = this.getBoundingClientRect();
            const size = Math.max(rect.width, rect.height);
            const x = e.clientX - rect.left - size / 2;
            const y = e.clientY - rect.top - size / 2;
            
            ripple.style.cssText = `
                position: absolute;
                border-radius: 50%;
                background: rgba(255, 255, 255, 0.6);
                transform: scale(0);
                animation: ripple-animation 0.6s linear;
                left: ${x}px;
                top: ${y}px;
                width: ${size}px;
                height: ${size}px;
                pointer-events: none;
            `;
            
            this.style.position = 'relative';
            this.style.overflow = 'hidden';
            this.appendChild(ripple);
            
            setTimeout(() => ripple.remove(), 600);
        });
    });
    
    // Add ripple animation CSS if not exists
    if (!document.getElementById('ripple-styles')) {
        const style = document.createElement('style');
        style.id = 'ripple-styles';
        style.textContent = `
            @keyframes ripple-animation {
                to {
                    transform: scale(4);
                    opacity: 0;
                }
            }
        `;
        document.head.appendChild(style);
    }
}

function addCustomTooltip(element) {
    if (!element) return;
    
    const title = element.getAttribute('title');
    if (!title) return;
    
    element.removeAttribute('title');
    
    element.addEventListener('mouseenter', function(e) {
        const tooltip = document.createElement('div');
        tooltip.className = 'custom-tooltip';
        tooltip.textContent = title;
        tooltip.style.cssText = `
            position: absolute;
            background: rgba(0, 0, 0, 0.9);
            color: white;
            padding: 8px 12px;
            border-radius: 6px;
            font-size: 12px;
            pointer-events: none;
            z-index: 10000;
            opacity: 0;
            transition: opacity 0.3s ease;
            white-space: nowrap;
        `;
        
        document.body.appendChild(tooltip);
        
        const rect = e.target.getBoundingClientRect();
        tooltip.style.left = rect.left + rect.width / 2 - tooltip.offsetWidth / 2 + 'px';
        tooltip.style.top = rect.top - tooltip.offsetHeight - 8 + 'px';
        
        setTimeout(() => tooltip.style.opacity = '1', 10);
        
        element.tooltipElement = tooltip;
    });
    
    element.addEventListener('mouseleave', function() {
        if (element.tooltipElement) {
            element.tooltipElement.style.opacity = '0';
            setTimeout(() => {
                if (element.tooltipElement) {
                    element.tooltipElement.remove();
                    element.tooltipElement = null;
                }
            }, 300);
        }
    });
}

function addFloatingLabelEffect(input) {
    if (!input) return;
    
    if (input.value) {
        input.classList.add('has-value');
    }
    
    input.addEventListener('focus', function() {
        this.classList.add('is-focused');
    });
    
    input.addEventListener('blur', function() {
        this.classList.remove('is-focused');
        if (this.value) {
            this.classList.add('has-value');
        } else {
            this.classList.remove('has-value');
        }
    });
    
    input.addEventListener('input', function() {
        if (this.value) {
            this.classList.add('has-value');
        } else {
            this.classList.remove('has-value');
        }
    });
}

function addFormValidationFeedback(form) {
    if (!form) return;
    
    const inputs = form.querySelectorAll('input, select, textarea');
    inputs.forEach(input => {
        if (!input.checkValidity()) {
            input.classList.add('is-invalid');
            
            // Add shake animation
            input.style.animation = 'shake 0.5s ease-in-out';
            setTimeout(() => input.style.animation = '', 500);
        } else {
            input.classList.remove('is-invalid');
            input.classList.add('is-valid');
        }
    });
}

function addAutoResize(textarea) {
    if (!textarea) return;
    
    textarea.style.resize = 'none';
    textarea.style.overflow = 'hidden';
    
    const resize = () => {
        textarea.style.height = 'auto';
        textarea.style.height = textarea.scrollHeight + 'px';
    };
    
    textarea.addEventListener('input', resize);
    resize(); // Initial resize
}

function updateTimestamps() {
    document.querySelectorAll('[data-timestamp]').forEach(element => {
        const timestamp = element.getAttribute('data-timestamp');
        if (timestamp) {
            element.textContent = formatTimeAgo(new Date(timestamp));
        }
    });
}

function updateCurrentTime() {
    const timeElement = document.getElementById('current-time');
    if (timeElement) {
        const now = new Date();
        timeElement.textContent = now.toLocaleString('id-ID', {
            day: '2-digit',
            month: 'short',
            year: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    }
}

function updateSystemStatus() {
    const apiCountElement = document.getElementById('api-count');
    const requestCountElement = document.getElementById('request-count');
    const uptimeElement = document.getElementById('uptime');
    
    // You can replace this with actual API calls
    if (apiCountElement) {
        animateNumber(apiCountElement, parseInt(apiCountElement.textContent) || 0, Math.floor(Math.random() * 10) + 1);
    }
    
    if (requestCountElement) {
        const current = parseInt(requestCountElement.textContent) || 0;
        animateNumber(requestCountElement, current, current + Math.floor(Math.random() * 50) + 1);
    }
}

function animateNumber(element, from, to) {
    if (!element) return;
    
    const duration = 1000;
    const start = Date.now();
    
    const animate = () => {
        const elapsed = Date.now() - start;
        const progress = Math.min(elapsed / duration, 1);
        const current = Math.floor(from + (to - from) * progress);
        
        element.textContent = current.toLocaleString();
        
        if (progress < 1) {
            requestAnimationFrame(animate);
        }
    };
    
    animate();
}

function formatTimeAgo(date) {
    const now = new Date();
    const diffInSeconds = Math.floor((now - date) / 1000);
    
    if (diffInSeconds < 60) {
        return 'Just now';
    } else if (diffInSeconds < 3600) {
        const minutes = Math.floor(diffInSeconds / 60);
        return `${minutes} minute${minutes > 1 ? 's' : ''} ago`;
    } else if (diffInSeconds < 86400) {
        const hours = Math.floor(diffInSeconds / 3600);
        return `${hours} hour${hours > 1 ? 's' : ''} ago`;
    } else {
        const days = Math.floor(diffInSeconds / 86400);
        return `${days} day${days > 1 ? 's' : ''} ago`;
    }
}

// Enhanced Error Handling
window.addEventListener('error', function(e) {
    console.error('🚨 JavaScript Error:', e.error || e.message || 'Unknown error');
    console.error('Error details:', {
        filename: e.filename,
        lineno: e.lineno,
        colno: e.colno,
        stack: e.error ? e.error.stack : 'No stack trace available'
    });
    // You can add error reporting here
    return true; // Prevent default browser error handling
});

// Handle unhandled promise rejections
window.addEventListener('unhandledrejection', function(e) {
    console.error('🚨 Unhandled Promise Rejection:', e.reason);
    e.preventDefault(); // Prevent default browser error handling
});

// Enhanced Performance Monitoring
if ('performance' in window) {
    window.addEventListener('load', function() {
        setTimeout(function() {
            const perfData = performance.timing;
            const loadTime = perfData.loadEventEnd - perfData.navigationStart;
            console.log(`⚡ Page Load Time: ${loadTime}ms`);
        }, 0);
    });
}

// Safe jQuery initialization
function initializeJQueryFeatures() {
    if (typeof jQuery === 'undefined') {
        console.warn('⚠️ jQuery not loaded, skipping jQuery-dependent features');
        return;
    }
    
    // Initialize jQuery-dependent features here
    console.log('✅ jQuery features initialized');
}

// Initialize jQuery features when jQuery is ready
if (typeof jQuery !== 'undefined') {
    jQuery(document).ready(function($) {
        initializeJQueryFeatures();
    });
} else {
    // Fallback: try to initialize after a delay
    setTimeout(initializeJQueryFeatures, 1000);
}

// Export functions for global use
window.showButtonLoading = showButtonLoading;
window.hideButtonLoading = hideButtonLoading;
window.animateNumber = animateNumber;
window.initializeJQueryFeatures = initializeJQueryFeatures;
